# -*- coding: utf-8 -*-
import logging
import os

from sandbox import sdk2
from sandbox import common

from sandbox.projects.resource_types import REPORT_STATIC_PACKAGE
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.resources import ImagesMicroPackage
from sandbox.projects.sandbox_ci.resources import VideoMicroPackage
from sandbox.projects.sandbox_ci.task import BaseMetaTask
from sandbox.projects.sandbox_ci.utils import flow, git
from sandbox.projects.sandbox_ci.managers.arc.arc_cli import save_arc_changed_files_with_statuses
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.managers import s3mds
from sandbox.projects.sandbox_ci.managers.artifacts import ArtifactCacheStatus


def format_tartifacts_cmd(artifact):
    return 'ARTIFACT_FILENAME="{}.tar.gz" ARTIFACTS_DIR="{{working_dir}}" BUNDLES="{{bundles}}" make -Bj tartifacts'.format(artifact)


class SandboxCiFijiBuild(BaseMetaTask):
    """Австосборка Картинок и Видео (mm-interfaces/fiji)"""

    class Parameters(BaseMetaTask.Parameters):
        with sdk2.parameters.Group('Services') as service_block:
            services = sdk2.parameters.List(
                'Services',
                default=('images', 'video'),
                required=True,
                description='Список сервисов, которые нужно собирать',
            )

        with sdk2.parameters.Group('Флаги') as flags_input_block:
            upload_created_exp_flags = sdk2.parameters.Bool(
                'Загрузить добавленные флаги',
                description=u'Новые флаги будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )
            update_changed_exp_flags = sdk2.parameters.Bool(
                'Обновить измененные флаги',
                description=u'Изменения в проектных флагах будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Build') as output_build_block:
                is_static_uploaded = sdk2.parameters.Bool(
                    'Is static uploaded',
                    description='Была ли загружена статика',
                )

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'build': 'npm run ci:build -- {bundles}',
        'artifacts_tests': 'tar -zcf {working_dir}/fiji.tar.gz --exclude-vcs --exclude=./node_modules --exclude=./freeze .',
        'artifacts_expflags': format_tartifacts_cmd(artifact="expflags"),
        'artifacts_static': format_tartifacts_cmd(artifact="fiji-static"),
        'artifacts_micropackage': format_tartifacts_cmd(artifact="fiji-micropackage"),
    }

    lifecycle_artifacts_map = {
        'artifacts_tests': ['fiji'],
        'artifacts_expflags': ['expflags'],
        'artifacts_static': ['fiji-static'],
        'artifacts_micropackage': ['fiji-micropackage'],
    }

    project_name = 'fiji'
    github_context = u'[Sandbox CI] Сборка'
    report_description = 'build'

    def configure(self):
        self.prepare_build_environ()

        super(SandboxCiFijiBuild, self).configure()

    def prepare_sources(self):
        super(SandboxCiFijiBuild, self).prepare_sources()

        if self.Parameters.use_arc:
            save_arc_changed_files_with_statuses(self.project_dir, 'changed-files.json')
        else:
            git.save_git_changed_files_with_statuses(self.project_dir, 'changed-files.json')

    def build(self):
        super(SandboxCiFijiBuild, self).build()

        flow.parallel(apply, [
            lambda: self.__create_artifacts(['artifacts_tests', 'artifacts_expflags', 'artifacts_static', 'artifacts_micropackage']),
            self.upload_static
        ])

    def create_artifacts(self, lifecycle_step=None):
        if not lifecycle_step:
            logging.debug('All artifacts are already built')
            return

        super(SandboxCiFijiBuild, self).create_artifacts(lifecycle_step)

    def __create_artifacts(self, artifact_types):
        flow.parallel(apply, map(lambda i: lambda: self.__create_artifact(i), artifact_types))

    def __create_artifact(self, lifecycle_step):
        super(SandboxCiFijiBuild, self).create_artifacts(lifecycle_step)

    @property
    def artifacts_resource_types(self):
        """
        :return: Тип и названия ресурсов.
        :rtype: dict
        """
        return {
            'fiji-static': REPORT_STATIC_PACKAGE,
            'fiji-micropackage': {
                'images': ImagesMicroPackage,
                'video': VideoMicroPackage,
            }.get(self.sub_project, SANDBOX_CI_ARTIFACT)
        }

    @property
    def sub_project(self):
        """
        :return: Подпроект - images или video. None - для пулл-реквестов.
        :rtype: str or None
        """
        is_images = 'images' in self.Parameters.services
        is_video = 'video' in self.Parameters.services

        if is_images and is_video:
            return None

        if is_images:
            return 'images'

        if is_video:
            return 'video'

    @property
    @common.utils.singleton
    def lifecycle(self):
        lifecycle = super(SandboxCiFijiBuild, self).lifecycle
        lifecycle.update_vars(
            bundles=' '.join(self.Parameters.services),
        )
        return lifecycle

    def _build(self, cache_status):
        super(SandboxCiFijiBuild, self)._build(cache_status)

        if cache_status is ArtifactCacheStatus.FOUND:
            self.Parameters.is_static_uploaded = True

    def upload_static(self):
        if self.need_to_skip_check('static'):
            logging.info('Skip static uploading')
            self.Parameters.is_static_uploaded = False
            return

        with self.profile_action(actions_constants['DEPLOY_STATIC'], 'Deploy static'):
            abs_freeze_path = str(self.working_path(os.path.join(self.project_name, 'freeze')))
            is_release_mode = self.is_release and self.Parameters.project_build_context != 'release_test'

            if is_release_mode:
                bucket_name = 'fiji-static'
                key_prefix = '_'
            else:
                bucket_name = 'serp-static-testing'
                key_prefix = os.path.join(self.project_name, 'static', self.project_name, 'freeze')

            self.Parameters.is_static_uploaded = s3mds.upload_dir(
                freeze_path=abs_freeze_path,
                bucket_name=bucket_name,
                key_prefix=key_prefix,
                should_compress=self.is_release,
                key_id=self.vault.read('env.SEARCH_INTERFACES_S3_ACCESS_KEY_ID'),
                access_key=self.vault.read('env.SEARCH_INTERFACES_S3_SECRET_ACCESS_KEY')
            )
