# -*- coding: utf-8 -*-
import os
import logging

from sandbox import sdk2

from sandbox.common.utils import singleton_property
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import REPORT_STATIC_PACKAGE

from sandbox.projects.sandbox_ci.task import BaseBuildTask
from sandbox.projects.sandbox_ci.sandbox_ci_static_s3 import SandboxCiStaticS3

from sandbox.projects.sandbox_ci.resources.template_packages import VideoGrannyDesktopMicroPackage
from sandbox.projects.sandbox_ci.resources.template_packages import VideoGrannyTouchPhoneMicroPackage

from sandbox.projects.sandbox_ci.resources.template_packages import ImagesGrannyDesktopMicroPackage
from sandbox.projects.sandbox_ci.resources.template_packages import ImagesSmartMicroPackage
from sandbox.projects.sandbox_ci.resources.template_packages import ImagesExpMicroPackage

from sandbox.projects.sandbox_ci.resources.template_packages import ImagesTemporaryMicroPackage


class SandboxCiFijiBuildPlatform(nanny.ReleaseToNannyTask2, BaseBuildTask):
    """Сборка различных платформ Картинок или Видео (mm-interfaces/fiji)"""

    project_name = 'fiji'
    github_context = u'[Sandbox CI] Сборка'

    class Parameters(BaseBuildTask.Parameters):
        project_github_owner = sdk2.parameters.String(
            'GitHub owner',
            description='Логин владельца репозитория или название организации',
            default='mm-interfaces',
        )

        project_github_repo = sdk2.parameters.String(
            'GitHub repo',
            description='Название репозитория',
            default='fiji',
        )

        with BaseBuildTask.Parameters.project_build_block() as project_build_block:
            deploy_static = sdk2.parameters.Bool(
                'Deploy debian static',
                description='Собрать и задеплоить debian статику',
                default=True
            )

            deploy_static_to_s3 = sdk2.parameters.Bool(
                'Deploy static to s3',
                description='Собрать и задеплоить статику в s3',
                default=False,
            )

            arguments = sdk2.parameters.List(
                'Arguments',
                description='Аргументы, флаги сборки',
                default='',
            )

            service = sdk2.parameters.String(
                'Service',
                required=True,
                description='Сервис, который нужно собрать',
                default='images',
                choices=[
                    ('images', 'images'),
                    ('video', 'video'),
                ],
                sub_fields={
                    'images': ['platform_images'],
                    'video': ['platform_video'],
                }
            )

            platform_images = sdk2.parameters.String(
                'Platform',
                required=True,
                description='Платформа, которую нужно собрать',
                default='desktop',
                choices=[
                    ('desktop', 'desktop'),
                    ('touch-pad', 'touch-pad'),
                    ('touch-phone', 'touch-phone'),
                    ('smart', 'smart'),
                ]
            )

            platform_video = sdk2.parameters.String(
                'Platform',
                required=True,
                description='Платформа, которую нужно собрать',
                default='desktop',
                choices=[
                    ('desktop', 'desktop'),
                    ('touch-pad', 'touch-pad'),
                    ('touch-phone', 'touch-phone'),
                    ('tv', 'tv'),
                ]
            )

    @property
    def lifecycle_artifacts_map(self):
        artifacts_map = {
            'artifacts_tests': ['fiji'],
            'artifacts': ['fiji-micropackage'],
        }

        if self.Parameters.deploy_static_to_s3:
            artifacts_map['artifacts'].append('fiji-static')

        return artifacts_map

    @property
    def artifacts_resource_types(self):
        service_platform = '{granny}{service}/{platform}'.format(
            service=self.Parameters.service,
            platform=self.platform,
            granny='granny-' if '-granny/' in self.Parameters.project_git_base_ref else '',
        )

        dynamic_resource_type = {
            'granny-video/desktop': VideoGrannyDesktopMicroPackage,
            'granny-video/touch-phone': VideoGrannyTouchPhoneMicroPackage,

            'granny-images/desktop': ImagesGrannyDesktopMicroPackage,
            'images/smart': ImagesSmartMicroPackage,

            'images/desktop': ImagesExpMicroPackage,
            'images/touch-pad': ImagesExpMicroPackage,
        }.get(service_platform, ImagesTemporaryMicroPackage)

        resource_types = {
            'fiji-micropackage': dynamic_resource_type,
        }

        if self.Parameters.deploy_static_to_s3:
            resource_types['fiji-static'] = REPORT_STATIC_PACKAGE

        logging.debug('Dynamic resource types for {service} is: {types}'.format(
            service=service_platform,
            types=resource_types,
        ))

        return resource_types

    lifecycle_steps = {
        'npm_install': 'make node_modules',
        'build': 'bin/fiji build {service}/{platform} {arguments} --no-cache',
        'artifacts': 'BUNDLES="{service}/{platform}" ARTIFACTS_DIR="../" make -Bj artifacts',
        'artifacts_tests': 'rm -rf ../fiji.tar.gz && tar -zcf ../fiji.tar.gz --exclude-vcs --exclude=./node_modules .',
    }

    @singleton_property
    def platform(self):
        return getattr(self.Parameters, 'platform_{}'.format(self.Parameters.service))

    @singleton_property
    def lifecycle(self):
        lifecycle = super(SandboxCiFijiBuildPlatform, self).lifecycle
        lifecycle.update_vars(
            service=self.Parameters.service,
            platform=self.platform,
            arguments=' '.join(self.Parameters.arguments),
        )
        return lifecycle

    def declare_subtasks(self):
        subtasks = [
            self.create_deploy_static_subtask(self.Parameters.service, self.platform)
        ]

        return subtasks

    def create_deploy_static_subtask(self, service, platform, wait_tasks=None):
        if self.Parameters.deploy_static_to_s3 and self.config.is_enabled('deploy', 'static_s3'):
            return self.meta.declare_subtask(
                task_type=SandboxCiStaticS3,
                wait_tasks=wait_tasks,
                project_name=self.project_name,
                description=self.Parameters.description,
                static_resource=self.get_registered_artifact_id('{}-static'.format(self.project_name)),
                bucket_name='fiji-static',
                is_release=True,
                freeze_path=os.path.join(self.project_name, 'freeze'),
                key_prefix='_',
                compress_needed=True,
                additional_tags=[service, platform],
            )

    def get_nanny_webhook_urls(self, additional_parameters):
        release_type = additional_parameters['release_status']

        if release_type == 'testing':
            return [
                'https://webhook-handler.si.yandex-team.ru/v1/sandbox/release-check-static?genisys-section={}'.format(
                    'search-interfaces.releases.empty'
                )
            ]

        if release_type == 'stable':
            return ['https://webhook-handler.si.yandex-team.ru/v1/nanny/release-request-tickets-status-change']

        return []

    def get_nanny_webhook_type(self, additional_parameters):
        return 'RELEASE_WITH_TICKETS'

    def get_nanny_duplicate_policy_type(self, additional_parameters):
        release_type = additional_parameters['release_status']

        return 'CLOSE_AS_DUPLICATE' if release_type == 'stable' else 'IGNORE'

    def on_release(self, additional_parameters):
        """
        :param additional_parameters: введенные данные из формы релизов
        :type additional_parameters: dict
        """
        release_type = additional_parameters['release_status']

        tags = set(self.Parameters.tags + [release_type])

        logging.debug('Releasing task in {release_type} with parameters: {parameters} and tags: {tags}'.format(
            release_type=release_type,
            parameters=additional_parameters,
            tags=tags,
        ))

        self.Parameters.tags = list(tags)

        BaseBuildTask.on_release(self, additional_parameters)
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
