# -*- coding: utf-8 -*-

import os
import copy

from sandbox import sdk2
from sandbox.common.utils import singleton_property

from sandbox.projects.sandbox_ci.task import BaseHermioneTask


class SandboxCiFijiHermione(BaseHermioneTask):
    """
    Hermione-тесты Картинок и Видео (mm-interfaces/fiji)
    """

    project_name = 'fiji'

    class Requirements(BaseHermioneTask.Requirements):
        disk_space = 20 * 1024
        cores = 32
        ram = 32 * 1024

    class Parameters(BaseHermioneTask.Parameters):
        with sdk2.parameters.String('Service', required=True) as service:
            service.values[''] = service.Value('---', default=True)
            service.values['images'] = 'images'
            service.values['video'] = 'video'

        selective_run = sdk2.parameters.Bool('Run tests selectively', default=False)

    lifecycle_steps = copy.deepcopy(BaseHermioneTask.lifecycle_steps)
    lifecycle_steps.update({
        'test': 'npm run ci:hermione -- {run_opts} {run_custom_opts}',
    })

    @property
    def run_opts(self):
        return '--remote-grid --log-level=debug --service={} --platform={}'.format(self.Parameters.service, self.Parameters.platform)

    @property
    def report_description(self):
        return '{}/{}'.format(self.Parameters.service, self.Parameters.platform)

    @singleton_property
    def cache_parameters(self):
        parameters = super(SandboxCiFijiHermione, self).cache_parameters
        parameters.update(service=self.Parameters.service)

        # Селективные тесты могут реиспользовать и селективные, и полные тесты.
        # Полные тесты могут реиспользовать только полные.
        if self.Parameters.selective_run is False:
            parameters.update(selective_run=False)

        return parameters

    def set_environments(self):
        super(SandboxCiFijiHermione, self).set_environments()

        os.environ.update(self.browsers_config.format_sessions_per_browser_env_options())

    @property
    def json_reporter_file_path(self):
        return 'hermione-{}-{}.json'.format(self.Parameters.service, self.Parameters.platform)

    @property
    def report_common_attributes(self):
        return dict(
            super(SandboxCiFijiHermione, self).report_common_attributes,
            # у класса Resource уже есть атрибут service
            project_service=self.Parameters.service
        )

    def configure(self):
        super(SandboxCiFijiHermione, self).configure()

        if self.Parameters.selective_run:
            os.environ['selective_enabled'] = 'true'
            os.environ['selective_platform'] = self.Parameters.platform

            self.selective_coverage.build_selection(
                commit=self.Parameters.project_github_commit,
                platform=self.Parameters.platform,
                project_service=self.Parameters.service,
            )
            self.selective_coverage.build_changed_files_from_diff_lines()

    def _get_reports_attrs(self, status):
        reports_attrs = super(SandboxCiFijiHermione, self)._get_reports_attrs(status)

        if self.is_coverage_enabled():
            reports_attrs.append(self.selective_coverage.get_log_report_attrs(commit=self.Parameters.project_github_commit))

        if self.Parameters.selective_run:
            reports_attrs = reports_attrs + self.selective_coverage.get_selection_reports_attrs(commit=self.Parameters.project_github_commit)

        return reports_attrs

    def is_coverage_enabled(self):
        return self.environ.get('COVERAGE') == 'true'
