# -*- coding: utf-8 -*-

import json
import logging
import StringIO
import tarfile

from sandbox import sdk2

from sandbox.projects.sandbox_ci.sandbox_ci_fiji import SandboxCiFiji
from sandbox.projects.sandbox_ci.sandbox_ci_fiji_experiments.task import SandboxCiFijiExperiments
from sandbox.projects.sandbox_ci.task import BasePullRequestRunnerTask


class SandboxCiFijiPullRequestRunner(BasePullRequestRunnerTask):
    """Запуск проверок fiji из ПР с флагами"""

    class Parameters(BasePullRequestRunnerTask.Parameters):
        service = sdk2.parameters.RadioGroup(
            'Service',
            choices=(
                ('images', 'images'),
                ('video', 'video'),
            ),
            required=True,
        )
        tools = sdk2.parameters.CheckGroup(
            'Tools',
            choices=(
                ('Hermione', 'hermione'),
                ('Hermione e2e', 'hermione-e2e'),
            ),
            default=['hermione', 'hermione-e2e'],
            required=True,
        )
        platforms = sdk2.parameters.CheckGroup(
            'Platforms',
            choices=(
                ('desktop', 'desktop'),
                ('touch-pad', 'touch-pad'),
                ('touch-phone', 'touch-phone'),
            ),
            default=['desktop', 'touch-pad', 'touch-phone'],
            required=True,
        )

        with sdk2.parameters.Group('Флаги') as flags_block:
            use_file_to_pass_flags = sdk2.parameters.Bool(
                'Use a file as a transport of flags to templates',
                default=True,
            )

    project_name = 'fiji'
    build_task_type = SandboxCiFiji
    experiments_subtask_type = SandboxCiFijiExperiments
    flags_resource_filename = 'exp-flags.json'

    def create_experiments_subtask(self, build_task, **custom_task_params):
        flags_resource = self.create_flags_resource() if self.Parameters.use_file_to_pass_flags else None

        return super(SandboxCiFijiPullRequestRunner, self).create_experiments_subtask(
            build_task,

            flags_resource=flags_resource,

            service=self.Parameters.service,

            **custom_task_params
        )

    def create_flags_resource(self):
        flags_archive = 'exp-flags.tar.gz'

        with tarfile.open(flags_archive, 'w:gz') as tar:
            data = json.dumps(self.flags)

            tarinfo = tarfile.TarInfo(self.flags_resource_filename)
            tarinfo.size = len(data)

            logging.debug('Adding to {archive} flags: {flags}'.format(
                archive=flags_archive,
                flags=data,
            ))

            tar.addfile(tarinfo, StringIO.StringIO(data))

        resource = self.artifacts.create_artifact_resource(
            relative_path=flags_archive,
            resource_type=SANDBOX_CI_ARTIFACT,
            type='exp-flags',
        )

        logging.debug('Resource with {archive}: {resource}'.format(
            archive=flags_archive,
            resource=resource,
        ))

        sdk2.ResourceData(resource).ready()

        return resource

    def get_flags_env(self):
        if self.Parameters.use_file_to_pass_flags:
            return dict(
                HERMIONE_URL_QUERY_EXP_FLAGS_FROM_FILE=self.flags_resource_filename,
            )

        return dict(
            HERMIONE_URL_QUERY_TEMPLATE_EXP_FLAGS=';'.join(self.formatted_flags),
        )
