# -*- coding: utf-8 -*-
import json
import logging

from sandbox import sdk2

from sandbox.projects.resource_types import REPORT_STATIC_PACKAGE

from sandbox.projects.sandbox_ci.task import BaseBuildTask
from sandbox.projects.sandbox_ci.sandbox_ci_static_s3 import SandboxCiStaticS3

from sandbox.common.utils import singleton_property


class SandboxCiFijiVideoPlayer(BaseBuildTask):
    """Автосборка проекта video-player (mm-interfaces/fiji/contribs/video)"""

    class Parameters(BaseBuildTask.Parameters):
        with BaseBuildTask.Parameters.project_build_block() as project_build_block:
            project_build_folder = sdk2.parameters.String(
                'Build in project folder',
                description='Директория, в которой должна произойти сборка, относительно корня репозитория.',
                default='',
            )

    project_name = 'fiji'
    sub_project_name = 'video-player'

    skip_lfs_checkout = True

    lifecycle_steps = {
        'build': {'cmd': 'make build', 'work_dir': '{project_build_folder}'},
        'artifacts': {'cmd': 'ARTIFACTS_DIR={working_dir} make artifacts', 'work_dir': '{project_build_folder}'},
    }

    lifecycle_artifacts_map = {
        'artifacts': ['video-player', 'video-player-static'],
    }

    artifacts_resource_types = {
        'video-player-static': REPORT_STATIC_PACKAGE,
    }

    @property
    def project_build_folder(self):
        return self.working_path(self.project_name, self.Parameters.project_build_folder)

    @singleton_property
    def lifecycle(self):
        return super(SandboxCiFijiVideoPlayer, self).lifecycle.update_vars(
            project_build_folder=self.project_build_folder,
        )

    @singleton_property
    def bucket_path(self):
        return 'contribs/video-player/configs/bucket.json'

    @property
    def github_context(self):
        return u'[Sandbox CI] Автосборка'

    def declare_subtasks(self):
        changed_files = self._get_changed_files()

        if self.bucket_path in changed_files:
            conf = self.get_config()
            key_prefix = conf.get('path')[1:]
            prefix_bundles = key_prefix + '/bundles-common'
            prefix_pages = key_prefix + '/pages-common'

            return [
                self.create_deploy_static_subtask(
                    conf,
                    'video-player/bundles-common',
                    prefix_bundles
                ),
                self.create_deploy_static_subtask(
                    conf,
                    'video-player/pages-common',
                    prefix_pages
                ),
            ]

        return []

    def get_config(self):
        conf_path = str(self.working_path(self.project_name, self.bucket_path))
        with open(conf_path) as f:
            return json.load(f)

    def _get_changed_files(self):
        changed_files = []
        event = self.Context.githubEvent

        try:
            changed_files = event.get('payload').get('extra').get('changed_files')
        except Exception as e:
            logging.exception('Unable to get changed files: {}'.format(e))

        return changed_files

    def create_deploy_static_subtask(self, conf, freeze_path, key_prefix):
        return self.meta.declare_subtask(
            task_type=SandboxCiStaticS3,
            description=self.Parameters.description,
            static_resource=self.get_registered_artifact_id('{}-static'.format(self.sub_project_name)),
            project_name=self.sub_project_name,
            bucket_name=conf.get('bucket'),
            freeze_path=freeze_path,
            key_prefix=key_prefix,
            compress_needed=True,
            recursive_update=False,
        );
