# -*- coding: utf-8 -*-

import os
import random
import logging

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt, resource as ctr
from sandbox.common.types.client import Tag
from sandbox.common.utils import get_task_link, singleton_property

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.sandbox_ci_hermione_e2e_subtask import SandboxCiHermioneE2eSubtask
from sandbox.projects.sandbox_ci.sandbox_ci_hermione_subtask import SandboxCiHermioneSubtask
from sandbox.projects.sandbox_ci.utils import flow, env
from sandbox.projects.sandbox_ci.utils.sb_links import get_task_children_link

from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, OverlayfsMixin
from sandbox.projects.sandbox_ci.task.test_task import HermioneTask
from sandbox.projects.sandbox_ci.task.test_task.LinkBuilder import LinkBuilder

DEFAULT_HERMIONE_CHUNKS_COUNT = 3
RAMDRIVE_SIZE = 15 * 1024

SURFWAX_EXP_TAG = 'SURFWAX-EXP';


class SandboxCiHermione(PrepareWorkingCopyMixin, OverlayfsMixin, HermioneTask):
    class Parameters(HermioneTask.Parameters):
        project_tree_hash = parameters.project_tree_hash()
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()
        platforms = sdk2.parameters.List('Platforms', sdk2.parameters.String)
        tool = sdk2.parameters.String('Tool', default='hermione')
        sqsh_build_artifacts_resources = sdk2.parameters.Resource(
            'Sqsh build artifacts',
            resource_type=SANDBOX_CI_ARTIFACT,
            multiple=True,
            register_dependency=False,
        )
        build_artifacts_resources = sdk2.parameters.Resource(
            'Build artifacts',
            resource_type=SANDBOX_CI_ARTIFACT,
            multiple=True,
            register_dependency=False,
        )
        ref = sdk2.parameters.String('Build branch')
        beta_domain = sdk2.parameters.String('Домен беты из PR')
        hermione_base_url = sdk2.parameters.String('Hermione base url')
        hermione_config_path = sdk2.parameters.String('Hermione config path')
        issue_keys = sdk2.parameters.List('Startrek issue keys', sdk2.parameters.String)
        custom_command = sdk2.parameters.String('Кастомная команда перед запуском тестов', multiline=True)
        custom_opts = sdk2.parameters.String('Опции, которые будут добавлены при запуске тестов')
        project_base_tree_hash = sdk2.parameters.String('Base tree hash')
        changed_files = sdk2.parameters.String('Path to the list of changed files', default='')
        selective_run = sdk2.parameters.Bool('Run tests selectively', default=False)
        task_retries = sdk2.parameters.Integer(
            'Максимальное количество перезапусков упавших подзадач',
            description='Подзадачи будут перезапущены n раз только для упавших тестов',
            default=None
        )
        task_retries_delay = sdk2.parameters.List(
            'Паузы в минутах между перезапусками упавших подзадач',
            description='Массив длительностей пауз в минутах между перезапусками упавших подзадач',
            default=[]
        )
        html_reporter_use_sqlite = sdk2.parameters.Bool('Save test results to sqlite database (FEI-15320)', default=False)

        # HermioneTask наследуется от BaseTask, а не от BaseTestTask,
        # поэтому приходится добавлять здесь send_comment_to_issue
        with HermioneTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        data_center = parameters.data_center()

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

        disable_auto_mute = sdk2.parameters.Bool('Disable auto-muted tests plugin', default=False)

        send_results_to_testcop = sdk2.parameters.Bool('Send tests results to testcop', default=False)

    class Requirements(HermioneTask.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

    class Context(HermioneTask.Context):
        subtasks = []
        detailed_current_status = {}

    github_context = '[Sandbox CI] Hermione'
    skip_ci_scripts_checkout = False
    mod = 'default'
    report_description = 'merged'

    lifecycle_steps = {
        'npm_install': 'npm ci --registry=https://npm.yandex-team.ru',
    }

    @classmethod
    def format_github_context(cls, description):
        return SandboxCiHermioneSubtask.format_github_context(description)

    @property
    def custom_github_statuses(self):
        """
        Это свойство необходимо для того, чтобы проектная таска в MQ корректно проставила все статусы по платформам
        """
        return self.__get_github_statuses()

    @property
    def subtasks(self):
        return map(lambda task_id: sdk2.Task[task_id], self.Context.subtasks)

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    @property
    def tool(self):
        return self.Parameters.tool

    @property
    def report_common_attributes(self):
        return {
            'released': ctt.ReleaseStatus.STABLE,
            'tool': self.tool,
            'mod': self.mod,
            'report_description': self.report_description,
        }

    @property
    def report_status(self):
        return ctt.Status.FAILURE if self.meta.is_any_failed(self.meta.subtasks) else ctt.Status.SUCCESS

    @sdk2.header()
    def header(self):
        report = super(SandboxCiHermione, self).header()

        linkBuilder = LinkBuilder(self)

        report.update({
            '<h3>Tests</h3>': '<p>{}</p>'.format(
                linkBuilder.make_testcop_link()
            )
        })

        return report

    @property
    def __browsers_config(self):
        return self.project_conf.get('tests', {}).get(self.tool, {}).get('browsers', {})

    @property
    def __json_report_path(self):
        return self.project_path('{}.json'.format(self.tool))

    @property
    def __selective_index_path(self):
        return self.project_path('selective.log')

    @property
    def _tool_config_name(self):
        return self.tool.replace('-', '_')

    @singleton_property
    def __enable_surfwax_exp_for(self):
        return self.config.get_deep_value(['tests', self._tool_config_name, 'enable_surfwax_exp_for'])

    @singleton_property
    def __is_under_surfwax_exp(self):
        if SURFWAX_EXP_TAG in self.Parameters.tags:
            logging.debug('Surfwax exp is enabled because of {} tag'.format(SURFWAX_EXP_TAG))

            return True

        exp_for = self.__enable_surfwax_exp_for

        if exp_for is None:
            return False

        if not 0 <= exp_for <= 1:
            logging.debug('\'enable_surfwax_exp_for\' should be in range (0, 1) but got \'{}\''.format(exp_for))
            exp_for = 0

        logging.debug('Surfwax exp is enabled for {} % of tasks'.format(exp_for * 100))

        return random.random() < exp_for

    @property
    def __hermione_chunks_count(self):
        return int(os.getenv('hermione_chunks_count', DEFAULT_HERMIONE_CHUNKS_COUNT))

    def get_conf_environ(self):
        return env.merge((
            super(SandboxCiHermione, self).get_conf_environ(),
            self.config.get_deep_value(['tests', 'environ'], {}),
            self.config.get_deep_value(['tests', self._tool_config_name, 'environ'], {})
        ))

    def on_save(self):
        super(SandboxCiHermione, self).on_save()

        if self.use_git_in_overlayfs_mode:
            self.set_ramdrive_size(25 * 1024)

    def on_enqueue(self):
        """
        При первом запуске корневая таска не должна ждать свои ресурсы (в режиме use_oveflayfs)
        или иметь какаие-либо ограничения по хостам, а должна незамедлительно запускать свои сабтаски, но при втором запуске,
        когда нужно обработать результаты сабтасок, необходимо дождаться всех ресурсов и запуститься на ssd
        """
        super(SandboxCiHermione, self).on_enqueue()

        # данные о параллельности для FEI-18225
        if self.project_name == 'web4' and self.tool == 'hermione-e2e':
            self.set_semaphore(name='hermione_e2e_web4')

        with self.memoize_stage.on_enqueue:
            self.__report_pending_scp_statuses(description=u"Ожидает в очереди")

            if not self.Parameters.use_overlayfs:
                self.__register_deps()

            return

        self.__register_deps()

        if bool(self.config.get_deep_value(['tests', self._tool_config_name, 'hermione_ssd_only'], False)):
            self.Requirements.client_tags &= Tag.SSD

    def __register_deps(self):
        from sandbox.yasandbox.manager.task import TaskManager
        map(lambda resource: TaskManager.register_dep_resource(self.id, resource), self.Parameters.build_artifacts_resources)

    def wait_dependency_tasks(self):
        """
        Корневая таска не должна ждать таски, указанные в `wait_tasks`, их ждут сабтаски корневой таски
        """
        pass

    def wait_dependency_output_parameters(self):
        """
        Корневая таска не должна ждать выходные параметры, указанные в `wait_output_parameters`, их ждут сабтаски корневой таски
        """
        pass

    def on_prepare(self):
        """
        При первом запуске корневой таске не нужны CI-скрипты,
        но необходимы при втором, когда нужно обработать результаты сабтасок
        """
        with self.memoize_stage.on_prepare:
            self.skip_ci_scripts_checkout = self.use_arc or not self.use_overlayfs

            self.__report_pending_scp_statuses(description=u"Выполняется")

        super(SandboxCiHermione, self).on_prepare()

    def on_before_end(self, status):
        super(SandboxCiHermione, self).on_before_end(status)

        self.__send_comment_to_issue(status)

    def __send_comment_to_issue(self, status):
        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)

    def __report_pending_scp_statuses(self, description=''):
        for platform in self.Parameters.platforms:
            self.scp_feedback.report_status_to_current_sha(
                context=self.__format_platform_github_context(platform),
                state=self.scp_feedback.convert_task_status_to_scp_state(self.Context.current_status),
                description=description,
                url=get_task_link(self.id),
            )

    def __format_platform_github_context(self, platform):
        return '{}: {}'.format(self.github_context, platform)

    def execute(self):
        self.set_environments()

        with self.memoize_stage.run_subtasks:
            logging.debug('is_under_surfwax_exp = {}'.format(self.__is_under_surfwax_exp))
            if self.__is_under_surfwax_exp and self.__enable_surfwax_exp_for != 1: # не выставляем тег, если эксперимент раскатан на 100%
                self.Parameters.tags = list(set(self.Parameters.tags + [SURFWAX_EXP_TAG]))

            started_subtasks = self.__run_subtasks()

            with self.profile_action(actions_constants['CREATE_DRAFT_SQLITE_REPORT'], 'Creating draft sqlite report'):
                if self.Parameters.html_reporter_use_sqlite:
                    self.__create_draft_sqlite_report()

            raise sdk2.WaitTask(
                tasks=started_subtasks,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            )

        self.__handle_results()

    # note: не является переопределением BaseTestTask.set_environments,
    # поскольку данная таска не является потомком BaseTestTask
    def set_environments(self):
        self.html_report.set_extra_items()

    def __run_subtasks(self):
        with self.profile_action(actions_constants['CREATE_SUBTASKS'], 'Creating subtasks'):
            created_subtasks = flow.parallel(self.__create_hermione_subtask, range(1, self.__hermione_chunks_count + 1))

            self.Context.subtasks += map(lambda task: task.id, created_subtasks)

        with self.profile_action(actions_constants['START_SUBTASKS'], 'Starting subtasks'):
            return self.meta.start_subtasks(created_subtasks)

    @singleton_property
    def __environ(self):
        env = self.Parameters.environ

        if self.__is_under_surfwax_exp:
            env['SURFWAX_EXP'] = '1'

        return env

    @singleton_property
    def __subtask_params(self):
        blockstat_resource = self.__find_blockstat_resource()
        skip_list_id = self.skip_list.get_id(
            project=self.project_name,
            tool=self.tool,
            branch=self.Parameters.ref if 'release' in getattr(self.Parameters, 'ref', '') else 'dev',
        )

        return dict(
            reusable=True,
            wait_tasks=self.Parameters.wait_tasks,
            wait_output_parameters=self.Parameters.wait_output_parameters,
            build_artifacts_resources=self.Parameters.sqsh_build_artifacts_resources,
            ref=self.Parameters.ref,
            beta_domain=self.Parameters.beta_domain,
            issue_keys=self.Parameters.issue_keys,
            custom_command=self.Parameters.custom_command,
            custom_opts=self.Parameters.custom_opts,
            selective_run=self.Parameters.selective_run,
            changed_files=self.Parameters.changed_files,
            project_base_hash=self.Parameters.project_base_hash,
            project_base_tree_hash=self.Parameters.project_base_tree_hash,
            project_hash=self.Parameters.project_hash,
            task_retries=self.Parameters.task_retries,
            task_retries_delay=self.Parameters.task_retries_delay,
            environ=self.__environ,
            skip_list_id=skip_list_id,
            blockstat_resource=blockstat_resource,
            use_overlayfs=self.Parameters.use_overlayfs,
            disable_auto_mute=self.Parameters.disable_auto_mute,
            send_results_to_testcop=self.Parameters.send_results_to_testcop,
        )

    def __create_hermione_subtask(self, test_chunk_to_run):
        # TODO: https://st.yandex-team.ru/FEI-17611
        task_type = SandboxCiHermioneSubtask if self.tool == 'hermione' else SandboxCiHermioneE2eSubtask

        return self.meta.create_subtask(
            task_type=task_type,
            tool=self.tool,
            hermione_base_url=self.Parameters.hermione_base_url,
            hermione_config_path=self.Parameters.hermione_config_path,
            platforms=self.Parameters.platforms,
            test_chunks_count=self.__hermione_chunks_count,
            test_chunk_to_run=test_chunk_to_run,
            description=self.get_test_subtask_description('chunk {}'.format(test_chunk_to_run)),
            report_github_statuses=False,
            report_arcanum_checks=False,
            send_comment_to_searel=False,
            send_comment_to_issue='',
            data_center=self.Parameters.data_center,
            html_reporter_use_sqlite=self.Parameters.html_reporter_use_sqlite,
            **self.__subtask_params
        )

    @in_case_of('use_overlayfs', '_create_draft_sqlite_report_in_overlayfs_mode')
    def __create_draft_sqlite_report(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self._install_dependencies()
        self.__publish_draft_sqlite_report()

    def _create_draft_sqlite_report_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            with self._overlayfs(lower_dirs=[self.project_sources_dir], target_dir=self.project_dir):
                self.__publish_draft_sqlite_report()

    @in_case_of('use_overlayfs', '_handle_results_in_overlayfs_mode')
    def __handle_results(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self._install_dependencies()

        self.__merge_reports()

    def _handle_results_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            with self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                self.__merge_reports()

    def __merge_reports(self):
        reports = [
            self.__publish_merged_stability_index,
            self.__publish_merged_json_reports,
            self.__publish_merged_selective_index,
            self.__publish_merged_stat_reports,
            self.__publish_merged_template_errors,
            self.__publish_merged_plugins_profiler_reports,
        ]

        if self.Parameters.html_reporter_use_sqlite:
            reports.append(self.__publish_final_sqlite_reports)
        else:
            reports.append(self.__publish_merged_html_reports)

        with self.profile_action(actions_constants['MERGE_REPORTS'], 'Merging reports'):
            flow.parallel(apply, reports)

        if self.meta.is_any_failed(self.meta.subtasks):
            raise TaskFailure('Has failed subtasks, see reports for more details')

    def __publish_merged_stability_index(self):
        return self.stability_index.publish_merged(tasks=self.meta.subtasks)

    def __publish_draft_sqlite_report(self):
        # Смерженный отчёт не содержит собственных данных, только ссылки на них.
        # Его можно просматривать сразу: будут отображены данные из прошедших к этому моменту чанков.
        report_type = '{}-report'.format(self.tool)
        output_report_type = '{}-draft-report'.format(self.tool)

        # Публикуем со статусом WAIT_TASK на случай, если после завершения всех сабтасок отчёт
        # не будет переопубликован с правильным статусом
        is_published = self.__publish_merged_sqlite_report(ctt.Status.WAIT_TASK, report_type, output_report_type, running_task_ids=self.Context.subtasks)

        if is_published:
            draft_html_report = self.__find_draft_sqlite_report()
            report_url = 'https://proxy.sandbox.yandex-team.ru/{}/index.html'.format(draft_html_report.id)
            self.set_info('Draft hermione <a href="{}">report</a>'.format(report_url), do_escape=False)
        else:
            self.set_info('Can not publish draft hermione report')

    def __publish_final_sqlite_reports(self):
        report_type = '{}-report'.format(self.tool)

        draft_html_report = self.__find_draft_sqlite_report()
        is_published = self.__publish_merged_sqlite_report(self.report_status, report_type, finished_task_ids=self.Context.subtasks)

        if is_published and draft_html_report:
            self.Context.report_resources.remove(draft_html_report.id)

    def __publish_merged_sqlite_report(self, status, report_type, output_report_type=None, running_task_ids=[], finished_task_ids=[]):
        return self.html_report.publish_merged_sqlite_report(
            output_dirname=output_report_type or report_type,
            running_task_ids=running_task_ids,
            finished_task_ids=finished_task_ids,
            report_type=report_type,
            output_report_type=output_report_type,
            status=status,
            attrs=self.report_common_attributes,
        )

    def __find_draft_sqlite_report(self):
        draft_report_type = '{}-draft-report'.format(self.tool)
        return sdk2.Resource.find(task_id=self.id, attrs={'type': draft_report_type}).first()

    def __publish_merged_html_reports(self):
        output_report_type = '{}-report'.format(self.tool)
        report_type = '{}-archive'.format(output_report_type)
        subtask_with_merged_html_report = self.__get_task_with_merged_html_report(tasks=self.Context.subtasks, report_type=report_type)

        self.html_report.publish_merged_html_reports(
            task_ids=self.Context.subtasks if not subtask_with_merged_html_report else [subtask_with_merged_html_report],
            report_type=report_type,
            output_dirname=output_report_type,
            output_report_type=output_report_type,
            status=self.report_status,
            tar=True,
            attrs=self.report_common_attributes,
        )

    def __publish_merged_plugins_profiler_reports(self):
        self.plugins_profiler_report.publish_merged_reports(
            task_ids=self.Context.subtasks,
            status=self.report_status,
            attrs=self.report_common_attributes,
        )

    def __get_task_with_merged_html_report(self, tasks, report_type):
        for task in tasks:
            report_res = sdk2.Resource.find(task_id=task, state=ctr.State.READY, attrs=dict(type=report_type, **self.report_common_attributes)).first()

            if report_res:
                logging.debug('Found already merged html-report in task with id {}'.format(task))
                return task

    def __publish_merged_json_reports(self):
        self.json_report.publish_merged_json_reports(
            task_ids=self.Context.subtasks,
            report_path=self.__json_report_path,
            status=self.report_status,
            tags=','.join(self.Parameters.tags),
            attrs=dict(
                self.report_common_attributes,
                automated=self.is_automated,
            ),
        )

    def __publish_merged_selective_index(self):
        self.selective.publish_merged_index(
            task_ids=self.Context.subtasks,
            index_path=self.__selective_index_path,
            attrs=self.report_common_attributes,
        )

    def __publish_merged_stat_reports(self):
        self.stat_report.publish_merged_stat_reports(
            task_ids=self.Context.subtasks,
            attrs=self.report_common_attributes,
        )


    def __publish_merged_template_errors(self):
        return self.template_errors.publish_merged_template_errors(
            subtask_ids=self.Context.subtasks,
            status=self.report_status,
            attrs=self.report_common_attributes
        )

    def _report_scp_feedback(self, state, description=''):
        for github_status in self.__get_github_statuses():
            github_status.update({'description': description})
            self.scp_feedback.report_status_to_current_sha(**github_status)

    def on_finish(self, prev_status, status):
        super(SandboxCiHermione, self).on_finish(prev_status, status)

        self.Context.detailed_current_status = self.__get_detailed_current_status()

    def __get_detailed_current_status(self):
        detailed_current_status = {}
        json_report_path = self.artifacts.get_artifact_path(self.__json_report_path)
        json_report = self.json_report.safe_read(report_path=json_report_path)

        for platform in self.Parameters.platforms:
            detailed_current_status[platform] = self.__get_platform_status_by_json_report(platform, json_report)

        return detailed_current_status

    def __get_github_statuses(self):
        github_statuses = []
        report_url = self.html_report.get_html_report_url(tool=self.tool)

        for platform in self.Parameters.platforms:
            github_statuses.append(dict(
                context=self.__format_platform_github_context(platform),
                state=self.scp_feedback.convert_task_status_to_scp_state(self.__get_platform_status(platform)),
                url=self.html_report.format_platform_html_report_url(report_url, platform, default=get_task_children_link(self.id))
            ))

        github_statuses.append(dict(
            context=self.github_context,
            state=self.scp_feedback.convert_task_status_to_scp_state(self.Context.current_status),
            url=self.html_report.format_html_report_url(report_url),
        ))

        return github_statuses

    def __get_platform_status(self, platform):
        return self.Context.detailed_current_status[platform] if self.Context.detailed_current_status.get(platform) else ctt.Status.FAILURE

    def __get_platform_status_by_json_report(self, platform, json_report):
        if self.Context.current_status == ctt.Status.SUCCESS:
            return ctt.Status.SUCCESS

        if not len(json_report.failed_tests()):
            return ctt.Status.FAILURE

        return ctt.Status.FAILURE if json_report.has_failed_tests(self.browsers_config.get_platform_browsers(platform)) else ctt.Status.SUCCESS

    def __find_blockstat_resource(self):
        return SANDBOX_CI_ARTIFACT.find(
            state=ctr.State.READY,
            attrs=dict(
                dict='blockstat',
                format='object',
            ),
        ).limit(1).first()
