# -*- coding: utf-8 -*-

import os
import logging
import copy

from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.common.types.client import Tag
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.task import BaseHermioneTask, SelectiveMixin
from sandbox.projects.sandbox_ci.task.test_task import HermioneTask
from sandbox.projects.sandbox_ci.utils.lock import yt_lock
from sandbox.projects.sandbox_ci.utils.sb_links import get_task_children_link
from sandbox.projects.sandbox_ci.utils import flow


class SandboxCiHermioneSubtask(SelectiveMixin, BaseHermioneTask):
    class Requirements(BaseHermioneTask.Requirements):
        disk_space = 25 * 1024
        cores = 32
        ram = 150 * 1024  # Должно быть уменьшено в FEI-20122
        environments = BaseHermioneTask.Requirements.environments.default + (
            PipEnvironment('ylock[yt]'),
        )
        kill_timeout = 2400

    class Parameters(BaseHermioneTask.Parameters):
        platform = sdk2.parameters.String('Platform')
        platforms = sdk2.parameters.List('Platforms', sdk2.parameters.String)
        tool = sdk2.parameters.String('Tool', default='hermione')
        hermione_base_url = sdk2.parameters.String('Hermione base url')
        hermione_config_path = sdk2.parameters.String('Hermione config path')
        test_chunks_count = sdk2.parameters.Integer('Test chunks count', default=1)
        test_chunk_to_run = sdk2.parameters.Integer('Test chunk to run', default=1)
        project_base_tree_hash = sdk2.parameters.String('Base tree hash')
        selective_run = sdk2.parameters.Bool('Run tests selectively', default=True)
        changed_files = sdk2.parameters.String('Path to the list of changed files', default='')
        data_center = parameters.data_center()
        skip_list_id = sdk2.parameters.Integer('Skip list id')
        blockstat_resource = sdk2.parameters.Resource(
            'Ресурс с блокстатом',
            required=False,
            resource_type=SANDBOX_CI_ARTIFACT,
            register_dependency=False,
        )

        disable_auto_mute = sdk2.parameters.Bool('Disable auto-muted tests plugin', default=False)

        send_results_to_testcop = sdk2.parameters.Bool('Send tests results to testcop', default=False)

        with sdk2.parameters.Output():
            report_data_urls = sdk2.parameters.String(
                'Report data urls',
                description='Ссылки на данные отчета',
            )

    class Context(BaseHermioneTask.Context):
        per_platform_test_run_result = {}

    lifecycle_steps = copy.deepcopy(BaseHermioneTask.lifecycle_steps)
    lifecycle_steps.update({
        'test': 'npm run ci:{tool} -- {run_opts} {run_custom_opts}',
        'warmup': 'npm run ci:hermione:warmup'
    })

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    @property
    def run_opts(self):
        opts = []

        if self.Parameters.hermione_config_path != '':
            opts.append('--config {}'.format(self.Parameters.hermione_config_path))

        opts.append(super(SandboxCiHermioneSubtask, self).run_opts)

        return ' '.join(opts)

    @singleton_property
    def cache_parameters(self):
        parameters = super(SandboxCiHermioneSubtask, self).cache_parameters
        parameters.update(
            test_chunks_count=self.Parameters.test_chunks_count,
            test_chunk_to_run=self.Parameters.test_chunk_to_run,
            platforms=self.Parameters.platforms,
        )
        return parameters

    def _install_dependencies(self):
        if os.path.islink(str(self.project_path('node_modules'))):  # оторвать этот костыль после https://st.yandex-team.ru/FEI-14115
            logging.debug('dependencies are already installed, skip installing or getting them from cache')
            pass
        else:
            return super(SandboxCiHermioneSubtask, self)._install_dependencies()

    def on_enqueue(self):
        super(BaseHermioneTask, self).on_enqueue()

        if self.Parameters.data_center != 'RANDOM':
            self.Requirements.client_tags &= Tag[self.Parameters.data_center]

        if bool(self.config.get_deep_value(['tests', self.tool, 'hermione_subtask_ssd_only'], False)):
            self.Requirements.client_tags &= Tag.SSD

    def on_before_end(self, status):
        super(BaseHermioneTask, self).on_before_end(status)

        if self.Context.reused_same_task and self.__should_report_github_statuses():
            self.__for_each_platform_test_run_result(ctt.Status.SUCCESS)

        if not self.Context.per_platform_test_run_result and self.__should_report_github_statuses():
            self.__for_each_platform_test_run_result(ctt.Status.FAILURE)

    def __for_each_platform_test_run_result(self, status):
        for platform in self.Parameters.platforms:
            self.Context.per_platform_test_run_result[platform] = status

        self.Context.save()

    def run_tests(self):
        wait_task = False
        try:
            super(BaseHermioneTask, self).run_tests()
        except sdk2.WaitTask as e:
            wait_task = True
            raise e
        finally:
            if not wait_task:
                self.__handle_results()

    def make_reports(self, status):
        if not self.Parameters.report_data_urls:
            self.Parameters.report_data_urls = self.html_report.get_data_urls()
        super(BaseHermioneTask, self).make_reports(status)

    def _get_self_subtask_attrs(self):
        task_attrs = super(BaseHermioneTask, self)._get_self_subtask_attrs()
        task_attrs.pop('report_data_urls', None)
        return task_attrs

    def __handle_results(self):
        if not self.__should_report_github_statuses():
            logging.debug('Prerequisites for reporting of github statuses are not satisfied')
            return

        try:
            with self.profile_action(actions_constants['REPORT_RESULTS'], 'Reporting results'):
                with yt_lock(name=str(self.parent.id), block=True):
                    self.__safe_report_results()
        except Exception as e:
            logging.debug('Error while taking lock: {}'.format(e))
            self.__safe_report_results()

    def __should_report_github_statuses(self):
        report_github_statuses_from_subtasks = self.config.get_deep_value(['tests', self._tool_config_name, 'report_{}_statuses_from_subtasks'.format(self.scp_feedback.reporter_type)], False)

        return isinstance(self.parent, HermioneTask) and report_github_statuses_from_subtasks and self.scp_feedback.report_statuses_config and self.parent.scp_feedback.get_report_statuses_parameter()

    def __safe_report_results(self):
        try:
            self.__report_results()
        except Exception as e:
            logging.debug('Cannot report results: {}'.format(e))

    def __report_results(self):
        with self.profile_action(actions_constants['SET_RESULT'], 'Setting per platform test run result'):
            self.__set_per_platform_test_run_result()

        subtasks = self.parent.subtasks

        with self.profile_action(actions_constants['CHECK_TASKS_FOR_COMPLETNESS'], 'Сhecking all subtasks for completeness'):
            if not self.__is_test_run_finished_in_tasks(subtasks):
                logging.debug('Test run is not finished in all subtasks, so github statuses will not be reporterd')
                return

            logging.debug('Test run is finished in all subtasks')

        if self.Parameters.html_reporter_use_sqlite:
            # в основной hermione таске значение в github статусах должно перебиваться
            report_url = self.__get_draft_sqlite_report_url()
        else:
            with self.profile_action(actions_constants['MERGE_REPORTS'], 'Merging html reports'):
                report_url = self.__get_merged_html_report_url(subtasks)

        logging.debug('Github statuses will be reported from the current task with the following report_url: "{}"'.format(report_url))
        self.__report_github_statuses_by_tasks_results(subtasks, report_url)

    def __set_per_platform_test_run_result(self):
        self.Context.per_platform_test_run_result = self.__get_per_platform_test_run_result()
        self.Context.save()

    def __get_per_platform_test_run_result(self):
        json_report = self.json_report.safe_load(self.json_report.get_task_report_resource(self.id))

        per_platform_test_run_result = {}
        for platform in self.Parameters.platforms:
            per_platform_test_run_result[platform] = self.__get_platform_status_by_json_report(platform, json_report)

        return per_platform_test_run_result

    def __get_platform_status_by_json_report(self, platform, json_report):
        if self.Context.task_status == ctt.Status.SUCCESS:
            return ctt.Status.SUCCESS

        if not len(json_report.failed_tests()):
            return ctt.Status.FAILURE

        return ctt.Status.FAILURE if json_report.has_failed_tests(self.browsers_config.get_platform_browsers(platform)) else ctt.Status.SUCCESS

    def __is_test_run_finished_in_tasks(self, tasks):
        results = flow.parallel(apply, map(lambda task: lambda: task.Context.per_platform_test_run_result, tasks))

        for res in results:
            if not res:
                return False

        return True

    def __get_draft_sqlite_report_url(self):
        draft_report_type = '{}-draft-report'.format(self.Parameters.tool)
        draft_html_report = sdk2.Resource.find(task_id=self.parent.id, attrs={'type': draft_report_type}).first()
        if draft_html_report:
            return 'https://proxy.sandbox.yandex-team.ru/{}/index.html'.format(draft_html_report.id)

    def __get_merged_html_report_url(self, tasks):
        output_report_type = '{}-report'.format(self.tool)

        is_published = self.html_report.publish_merged_html_reports(
            task_ids=map(lambda s: s.id, tasks),
            report_type='{}-archive'.format(output_report_type),
            output_dirname='{}-merged'.format(output_report_type),
            output_report_type=output_report_type,
            status=self.__get_platforms_status_by_tasks(self.Parameters.platforms, tasks),
            tar=True,
            attrs=dict(self.parent.report_common_attributes, add_to_context=False),
        )

        if is_published:
            return self.html_report.get_html_report_url(tool=self.tool)

    def __report_github_statuses_by_tasks_results(self, tasks, report_url):
        for platform in self.Parameters.platforms:
            self.scp_feedback.report_status_to_current_sha(
                context=self.format_github_context(platform),
                state=self.scp_feedback.convert_task_status_to_scp_state(self.__get_platforms_status_by_tasks([platform], tasks)),
                url=self.html_report.format_platform_html_report_url(report_url, platform, default=get_task_children_link(self.parent.id)),
                description='',
                force=True,
            )

    def __get_platforms_status_by_tasks(self, platforms, tasks):
        for task in tasks:
            result = task.Context.per_platform_test_run_result
            for platform in platforms:
                if not result.get(platform) or result[platform] == ctt.Status.FAILURE:
                    return ctt.Status.FAILURE

        return ctt.Status.SUCCESS

    def send_statistic(self):
        super(BaseHermioneTask, self).send_statistic()

        self.assert_metrics.send_coverage_to_statface()

    def set_environments(self):
        self.html_report.set_extra_items(parent=self.parent)
        self.__configure_hermione_chunks()

        super(SandboxCiHermioneSubtask, self).set_environments()

        if self.Parameters.skip_list_id:
            branch = os.environ.get('BUILD_BRANCH') if 'release' in os.environ.get('BUILD_BRANCH', '') else 'dev'
            env_key = 'EXTERNAL_SKIP_{}_{}_{}'.format(self.tool, self.project_name, branch.replace('/', '.')).upper()
            os.environ[env_key] = str(self.Parameters.skip_list_id)

        if self.Parameters.blockstat_resource:
            os.environ['TEMPLAR_BLOCKSTAT_DICT_RESOURCE_ID'] = str(self.Parameters.blockstat_resource.id)

        if self.Parameters.hermione_base_url:
            os.environ['hermione_base_url'] = self.Parameters.hermione_base_url

        if self.Parameters.disable_auto_mute:
            os.environ['hermione_muted_tests_auto_mute_enabled'] = 'false'
            os.environ['hermione_muted_tests_testcop_stats_enabled'] = 'false'

    def __configure_hermione_chunks(self):
        os.environ['hermione_chunks_enabled'] = 'true'
        os.environ['hermione_chunks_count'] = str(self.Parameters.test_chunks_count)
        os.environ['hermione_chunks_run'] = str(self.Parameters.test_chunk_to_run)

    def get_selenium_grid_semaphores(self):
        selenium_grid_semaphores = []

        for semaphore in super(SandboxCiHermioneSubtask, self).get_selenium_grid_semaphores() or []:
            selenium_grid_semaphores.append(self.__recreate_sg_semaphore_by_test_chunks_count(semaphore))

        return selenium_grid_semaphores

    def __recreate_sg_semaphore_by_test_chunks_count(self, semaphore):
        return ctt.Semaphores.Acquire(name=semaphore.name, weight=self.__calc_weight_by_test_chunks_count(semaphore.weight), capacity=semaphore.capacity)

    def get_browsers_to_run(self):
        browsers_to_run = super(SandboxCiHermioneSubtask, self).get_browsers_to_run()

        return map(lambda bro: self.__redefine_browser_sessions_by_test_chunks_count(bro), browsers_to_run)

    def __redefine_browser_sessions_by_test_chunks_count(self, browser):
        browser.update({'sessions': browser['sessions'] if not browser['sessions'] else self.__calc_weight_by_test_chunks_count(browser['sessions'])})

        return browser

    def __calc_weight_by_test_chunks_count(self, weight):
        return max(int(round(weight / int(self.Parameters.test_chunks_count))), 1)
