# -*- coding: utf-8 -*-

import logging
import copy

from sandbox import sdk2
from sandbox import common
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt

from sandbox.projects.trendbox_ci.beta.tasks.trendbox_ci_job import TrendboxCiJob


class SandboxCiMegamindGoodwinE2E(sdk2.Task):
    """
        Wrapper to run Goodwin E2E-tests in Megamind release machine
    """
    name = 'SANDBOX_CI_MEGAMIND_GOODWIN_E2E'

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Run test parameters') as param_block:
            megamind_url = sdk2.parameters.String('Megamind url', required=True)
            bass_url = sdk2.parameters.String('Bass url')
            begemot_url = sdk2.parameters.String('Begemot url')
            experiments = sdk2.parameters.JSON('Experiments')
            options = sdk2.parameters.JSON('Custom test options')

        with sdk2.parameters.Group('Release info') as release_block:
            release_ticket = sdk2.parameters.String('Release ticket')

    class Context(sdk2.Task.Context):
        subtask_id = None

    parameters_to_throw = ('megamind_url', 'bass_url', 'begemot_url', 'experiments', 'options', 'release_ticket')
    env_prefix = 'GOODWIN'

    def on_execute(self):
        release_task = self.find_last_release_goodwin_task()
        modified_trendbox_config = self.modify_trendbox_config(
            release_task.Parameters.trendbox_config,
            self.get_envs_from_parameters(self.parameters_to_throw)
        )

        trendbox_config_with_tag = self.replace_branch_to_tag(modified_trendbox_config)    
        trendbox_config_with_branch = modified_trendbox_config

        with self.memoize_stage.create_children:
            subtasks = {}
            subtask_with_tag = self.create_and_run_child_task(release_task, trendbox_config_with_tag)
            subtasks['tag'] = subtask_with_tag.id
            subtask_with_branch = self.create_and_run_child_task(release_task, trendbox_config_with_branch)
            subtasks['branch'] = subtask_with_branch.id
            self.Context.subtasks = subtasks

            # Из-за особенностей релизов гудвина, запускаем две таски для получения копии гудвина разными способами
            # 1 - по релизному тегу, 2 - по релизной ветке, один из запусков гарантировано упадёт
            # Подробнее о проблеме в https://st.yandex-team.ru/SERP-122068
            raise sdk2.WaitTask(list(subtasks.values()), list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), wait_all=True)

        subtask_with_tag = self.find(id=self.Context.subtasks['tag']).first()
        subtask_with_branch = self.find(id=self.Context.subtasks['branch']).first()

        if subtask_with_tag.status not in ctt.Status.Group.SUCCEED and subtask_with_branch.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure(
                'All subtasks was finished with the status not succeed status'
            )

    def create_and_run_child_task(self, release_task, trendbox_config):
        return TrendboxCiJob(
            self,
            **self.get_subtask_parameters(release_task, trendbox_config)
        ).enqueue()

    def find_last_release_goodwin_task(self):
        trendbox_job_name = 'Hermione e2e: release.*'
        goodwin_branch_reg_exp = 'search-interfaces/frontend@release/goodwin/.*'

        task = sdk2.Task.find(
            type=TrendboxCiJob,
            description='{}\n.*{}'.format(trendbox_job_name, goodwin_branch_reg_exp),
            status=ctt.Status.Group.FINISH
        ).first()

        if task:
            logging.debug('Find last release goodwin task: {task}'.format(task=task))

        if task is None:
            raise TaskFailure('Could not find success {task} with name {name} and branch regexp: {regexp}'.format(
                task=TrendboxCiJob,
                name=trendbox_job_name,
                regexp=goodwin_branch_reg_exp
            ))

        return task

    def modify_trendbox_config(self, original_trendbox_config, value):
        trendbox_config = copy.deepcopy(original_trendbox_config)
        pathToEnv = ['lifecycle', 'steps', 'options', 'env']

        for env in self._get_subconfigs(trendbox_config, pathToEnv):
            env.update(value)

        return trendbox_config

    def replace_branch_to_tag(self, original_trendbox_config):
        trendbox_config = copy.deepcopy(original_trendbox_config)

        # После успешного релиза, релизная ветка удалаяется,
        # но можно получить содержимое релиза по тегу, который можно вычислить из релизной ветки
        branch = trendbox_config['workcopies']['after']['branch']
        tag_from_release_branch = branch.replace('release/', '')
        trendbox_config['workcopies']['after']['tag'] = tag_from_release_branch
        trendbox_config['meta']['tag'] = tag_from_release_branch

        del trendbox_config['workcopies']['after']['branch']
        del trendbox_config['workcopies']['after']['commit']
        del trendbox_config['meta']['branch']
    
        return trendbox_config

    def modify_trendbox_branch(self, original_trendbox_config, branch):
        trendbox_config = copy.deepcopy(original_trendbox_config)
        trendbox_config['meta']['branch'] = branch

        trendbox_config['workcopies']['after']['branch'] = branch

        del trendbox_config['workcopies']['after']['commit']
        del trendbox_config['workcopies']['after']['tag']
        del trendbox_config['meta']['tag']

        return trendbox_config

    def _get_subconfigs(self, config, path):
        step = path[0]
        sub_path = path[1:]
        config_part = config.get(step)

        if not sub_path:
            return config_part

        if isinstance(config_part, list):
            return [self._get_subconfigs(el, sub_path) for el in config_part]

        return self._get_subconfigs(config_part, sub_path)

    def get_envs_from_parameters(self, parameter_list):
        envs = {}

        for name, value in self.Parameters:
            if name in parameter_list:
                env_name = self._make_env_name(name)
                envs[env_name] = value

        return envs

    def _make_env_name(self, parameter_name):
        return '{prefix}_{parameter}'.format(
            prefix=self.env_prefix,
            parameter=parameter_name.upper()
        )

    def get_subtask_parameters(self, task_to_clone, trendbox_config):
        return dict(
            # Common params
            description='Hermione e2e: other components',

            owner=task_to_clone.owner,

            _container=task_to_clone.Parameters._container,

            build_id=task_to_clone.Parameters.build_id,
            build_request_id=task_to_clone.Parameters.build_request_id,

            webhook_url=task_to_clone.Parameters.webhook_url,
            webhook_payload=task_to_clone.Parameters.webhook_payload,

            use_last_binary_archive=task_to_clone.Parameters.use_last_binary_archive,

            trendbox_cli_resource=task_to_clone.Parameters.trendbox_cli_resource,

            job_id=task_to_clone.Parameters.job_id,

            is_privileged_container=task_to_clone.Parameters.is_privileged_container,

            workcopies=task_to_clone.Parameters.workcopies,

            trendbox_web_logs_url_template=task_to_clone.Parameters.trendbox_web_logs_url_template,

            # Modified config
            trendbox_config=trendbox_config
        )
