# -*- coding: utf-8 -*-

import logging
import os
import json
from contextlib import nested
from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import misc as ctm

from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import prioritizer, list_utils
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node, Debug
from sandbox.projects.sandbox_ci.managers.arc.context import create_arc_context
from sandbox.projects.sandbox_ci.managers.scripts import RunJsScriptError
from sandbox.projects.sandbox_ci.managers.errors.merge_queue import ArcanumMergeQueueCliError
from sandbox.projects.sandbox_ci.utils.ref_formatter import strip_ref_prefix, is_arcanum_ref, format_github_ref
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.sandbox_ci.utils.ref_adapter import PullRequestAdapter, ReviewRequestAdapter

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.sandbox_ci.task import ProjectPathsMixin


RAMDRIVE_SIZE = 15 * 1024
MQ_ROBOT_LOGIN = 'robot-merge-queue'


class SandboxCiMerge(TasksResourceRequirement, ManagersTaskMixin, ProjectPathsMixin, sdk2.Task):
    """Очередь для влития Pull Request'ов в дефолтную ветку проекта"""

    lifecycle_steps = {}

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL
        environments = (
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        )
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

    class Parameters(parameters.CommonParameters):
        kill_timeout = 3 * 3600
        project_github_repository = parameters.ProjectGitHubRepositoryParameters

        with sdk2.parameters.Group('Merge Queue') as mq_block:
            ssh_key_vault_name = sdk2.parameters.String('Имя переменной в vault-е, в которй лежит приватный SSH ключ пользователя')
            github_api_token_vault_name = sdk2.parameters.String('Имя переменной в vault-е (github api token)')
            refs_to_merge = parameters.refs_to_merge()

        with sdk2.parameters.Group(u'GitHub → Arcanum') as arcanum_migration_block:
            pull_request_merge_provider = parameters.pull_request_merge_provider()

        scripts_sources = parameters.ScriptsSourcesParameters
        environment = parameters.EnvironmentParameters

        node_js = parameters.NodeJsParameters

        with sdk2.parameters.Output():
            merged_pr = sdk2.parameters.List('Merged pull requests')

    class Context(sdk2.Task.Context):
        merged_pr = []
        # Отключает создание файла с отладочным выводом содержимого ramdrive в файл ramdrive_usage.yaml
        __do_not_dump_ramdrive = True

    @property
    def project_name(self):
        return self.parent.Parameters.project_github_repo

    @property
    def mq_config(self):
        return self.parent.config

    @property
    def after_merge(self):
        return self.mq_config.get('afterMerge', {})

    def on_save(self):
        super(SandboxCiMerge, self).on_save()
        self.Parameters.priority = prioritizer.get_priority(self)
        setattr(self.Context, 'project_git_url', self.parent.Context.project_git_url)
        semaphore_name = 'mq_{}_{}_merge'.format(self.parent.Parameters.project_github_owner, self.parent.Parameters.project_github_repo)
        self.mq_manager.acquire_semaphore(semaphore_name)

    def set_environments(self):
        os.environ['SANDBOX_TASK_ID'] = str(self.id)
        os.environ['SANDBOX_SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)
        os.environ['PR_NUMBER'] = strip_ref_prefix(list_utils.last(list_utils.map_prop(self.Parameters.refs_to_merge, 'ref')))
        os.environ['REPO'] = self.parent.Parameters.project_github_repo
        # Выравниваю переменные окружения с окружением Publish в Trendbox CI
        # см: https://github.yandex-team.ru/search-interfaces/frontend/blob/ee7e662ffd201431416eeb067c124710ae987eb4/.trendbox.yml#L32-L34
        os.environ['NPM_CONFIG_REGISTRY'] = 'https://npm.yandex-team.ru'
        os.environ['NPM_CONFIG_AUDIT'] = 'false'
        os.environ['NPM_CONFIG_PREFER_OFFLINE'] = 'true'

        repo = '{}/{}'.format(self.parent.Parameters.project_github_owner, self.parent.Parameters.project_github_repo)
        if repo == 'search-interfaces/frontend':
            os.environ['NPM_CONFIG_USER_AGENT'] = 'npm/6.2.0 (verdaccio yandex canary)'

        if repo == 'test-web4/frontend':
            os.environ['NPM_CONFIG_REGISTRY'] = 'http://npm-testing.in.yandex-team.ru'

        owner = self.parent.config.get('mergeQueueTask', {}).get('owner')

        for secret_name in self.after_merge.get('secrets', []):
            # если имя секрета начинается на `env.`, то отрезаем это
            env_name = secret_name[4:] if secret_name.startswith('env.') else secret_name
            os.environ[env_name] = self.vault.read(secret_name, owner)

    def on_prepare(self):
        self.set_environments()

        with self.parent.profiler.actions.clone_scripts('Cloning scripts'):
            self.scripts.sync_resource()

    def on_execute(self):
        try:
            if self.Parameters.pull_request_merge_provider == "arcanum":
                with self.parent.profiler.actions.git_merge('Merging the pull requests'):
                    self.__merge_review_requests_by_arcanum()
                    return

            with self.parent.profiler.actions.clone_project_for_merge('Cloning project for merging'):
                self.mq_manager.clone_project()

            with self.parent.profiler.actions.git_merge('Merging the pull requests'):
                self.__merge_refs(self.Parameters.refs_to_merge)
        except Exception as e:
            logging.exception('Exception occurred during merge phase')
            self.Context.merge_queue_task_exception = e.message
            self.mq_manager.merge_error_handler(e)

            raise

    def exec_after_merge_commands(self):
        for cmd in self.after_merge.get('commands', []):
            if self.__should_run_after_merge(cmd):
                self.lifecycle_steps.update({cmd['name']: cmd['value']})
                self.lifecycle(cmd['name'])
            else:
                logging.info('Command {} should be run only once. Skipping...'.format(cmd['name']))

    def __should_run_after_merge(self, cmd):
        is_first_cmd_run = len(self.Context.merged_pr) is 0
        return not bool(cmd.get('shouldRunOnce')) or is_first_cmd_run

    def __merge_review_requests_by_arcanum(self):
        if self.__need_arc_mount():
            with self.__arc_mount_ctx():
                self.__run_arcanum_merge_queue_cli()
            return

        self.__run_arcanum_merge_queue_cli()

    def __need_arc_mount(self):
        return 'beforeMerge' in self.mq_config.keys()

    def __run_arcanum_merge_queue_cli(self):
        contexts = (self.__node_debug_context(), self.__node_context(), self.__ssh_context(),)
        arcanum_secrets = self.__arcanum_secrets()

        os.environ['ARC_TOKEN'] = arcanum_secrets.get('ARC_TOKEN', '')
        os.environ['ARC_LOGIN'] = 'robot-merge-queue'
        os.environ['LOGIN'] = 'robot-merge-queue'
        os.environ['ARCANUM_API_OAUTH_TOKEN'] = arcanum_secrets.get('ARCANUM_API_OAUTH_TOKEN', '')
        os.environ['NODE_EXTRA_CA_CERTS'] = arcanum_secrets.get('NODE_EXTRA_CA_CERTS', '')

        # нужен для frontend.ci.autopublish для публикации в npm
        os.environ['ROBOT_FRONTEND_NPM_PASS'] = self.vault.read('env.ROBOT_FRONTEND_NPM_PASS')

        # нужен для frontend.ci.autopublish для оповещений в Slack
        os.environ['SLACK_TOKEN'] = self.vault.read('env.SLACK_TOKEN')

        mq_config_path = str(self.working_path('mq_config'))

        with open(mq_config_path, 'w') as f:
            f.write(json.dumps(self.mq_config, ensure_ascii=False).encode('utf8'))

        cmd = ['merge']
        cmd += map(lambda slug: '--pr ' + slug, self.__pr_slugs)
        cmd += ['--config {}'.format(mq_config_path)]

        if self.__need_arc_mount():
            mq_auth_url = 'http://merge-queue.si.yandex-team.ru/v2/oauth/token'

            # нужен для frontend.ci.autopublish для поиска ресурсов
            os.environ['SANDBOX_OAUTH_TOKEN'] = self.vault.read('env.SANDBOX_OAUTH_TOKEN')

            # нужен для получения делегированных токенов пользователей для автопубликации
            os.environ['OAUTH_APP_PASSWORD'] = self.vault.read('env.OAUTH_FEI_CLIENT_APP_PASSWORD')

            cmd += ['--oauthStoreEndpoint {}'.format(mq_auth_url)]
            cmd += ['--arcMountPath {}'.format(self.arc_mount_path)]
            cmd += ['--arcStorePath {}'.format(self.arc_store_path)]

        try:
            with nested(*contexts):
                return self.scripts.run_js(
                    'script/arcanum-merge-queue.js',
                    ' '.join(cmd)
                )
        except RunJsScriptError as error:
            raise ArcanumMergeQueueCliError(error.stdout)

    @property
    def __pr_slugs(self):
        return map(self.__pr_slug_from_ref, self.Parameters.refs_to_merge)

    def __pr_slug_from_ref(self, ref):
        ref_name = ref.get("ref", "")
        head_sha = ref.get("head_sha", "")
        pr_id = strip_ref_prefix(ref_name)

        return pr_id + "#" + head_sha

    def __merge_refs(self, refs):
        contexts = (self.__node_debug_context(), self.__node_context(), GitRetryWrapper(), self.__ssh_context(),)

        refs_names = list_utils.map_prop(refs, 'ref')
        has_arcanum_ref = any([is_arcanum_ref(ref) for ref in refs_names])

        logging.debug('Merge refs {merge_refs} contain arcanum ref: {has_arcanum_ref}'.format(
            merge_refs=refs_names,
            has_arcanum_ref=has_arcanum_ref,
        ))

        if has_arcanum_ref:
            arcanum_secrets = self.__arcanum_secrets()

            os.environ['ARC_TOKEN'] = arcanum_secrets.get("ARC_TOKEN", "")
            os.environ['ARCANUM_API_OAUTH_TOKEN'] = arcanum_secrets.get("ARCANUM_API_OAUTH_TOKEN", "")
            os.environ['NODE_EXTRA_CA_CERTS'] = arcanum_secrets.get("NODE_EXTRA_CA_CERTS", "")
            os.environ['GITHUB_API_TOKEN'] = self.vault.read(self.Parameters.github_api_token_vault_name)

            contexts += (self.__arc_context(),)

        filtered_refs = filter(self.__is_opened, refs)
        os.environ['REFS_TO_MERGE'] = json.dumps(filtered_refs)
        self.__check_refs(filtered_refs)

        with nested(*contexts):
            for ref in filtered_refs:
                os.environ['REFS_TO_MERGE'] = json.dumps([ref])
                self.__git_merge(ref)

    def __node_debug_context(self):
        return Debug('serp:*,kurwahl*,diffalka*,hubby:*,si:ci:*,arcanum-merge-queue-cli:*,arcanum-review-request:*,oauth-fei-client')

    def __node_context(self):
        return Node(self.Parameters.node_js_version)

    def __ssh_context(self):
        return self.vault.ssh_key(self.Parameters.ssh_key_vault_name)

    def __arcanum_secrets(self):
        return dict(
            ARC_TOKEN=self.vault.read('env.ARC_TOKEN'),
            ARCANUM_API_OAUTH_TOKEN=self.vault.read('env.ARCANUM_API_OAUTH_TOKEN'),
            NODE_EXTRA_CA_CERTS=self.vault.read('env.NODE_EXTRA_CA_CERTS'),
        )

    def __arc_context(self):
        project_config = self.mq_manager.get_project_config(
            project_name=self.parent.Parameters.project_github_repo
        )
        arc_config = project_config.get('arc', {})

        extra_params = []
        if arc_config.get('no_threads', False):
            extra_params.append('--no-threads')
        if arc_config.get('use_vfs2', False):
            extra_params.extend(['--vfs-version', '2'])

        mount_options = dict(
            mount_point=self.arc_mount_path,
            store_path=self.arc_store_path,
            extra_params=extra_params,
        )

        return create_arc_context(**mount_options)

    def __arc_mount_ctx(self):
        arc = Arc(arc_oauth_token=self.vault.read('env.ARC_TOKEN'))

        return arc.mount_path(
            mount_point=str(self.arc_mount_path),
            store_path=str(self.arc_store_path),
            path="",
            fetch_all=False,
            # only mount https://a.yandex-team.ru/arc_vcs/sandbox/projects/common/vcs/arc.py?rev=a83cecc23a497802fb43aacb495c9a8b260db7f5#L815
            changeset=None,
        )

    def __is_opened(self, ref):
        latest_ref_state = self.__get_latest_ref_state(ref)

        return latest_ref_state.is_opened

    def __check_refs(self, refs):
        for ref_to_merge in refs:
            ref_name = ref_to_merge['ref']
            ref_head_sha = ref_to_merge['head_sha']

            latest_ref_state = self.__get_latest_ref_state(ref_to_merge)

            if ref_head_sha != latest_ref_state.head_sha:
                raise TaskFailure('"{}" has been updated from "{}" to "{}"'.format(ref_name, ref_head_sha, latest_ref_state.head_sha))

    def __get_latest_ref_state(self, ref):
        ref_name = ref['ref']
        ref_id = strip_ref_prefix(ref_name)

        return ReviewRequestAdapter(self.__get_rr_info(ref_id)) if is_arcanum_ref(ref_name) else PullRequestAdapter(self.__get_pr_info(ref_id))

    def __get_pr_info(self, pr_number):
        return GitHubApi().get_pr_info(self.parent.Parameters.project_github_owner, self.parent.Parameters.project_github_repo, pr_number)

    def __get_rr_info(self, rr_id):
        fields = [
            'id',
            'state',
            'vcs(type,name,from_branch)',
            'checks(system,type,required,satisfied,status,system_check_uri)',
            'active_diff_set(id,status,arc_branch_heads(from_id,to_id,merge_id))',
        ]

        return self.scripts.run_js(
            'script/merge-queue/fetch-arcanum-review.js',
            str(rr_id),
            {'fields': ','.join(fields)}
        )

    def __git_merge(self, ref_to_merge):
        project_dir = self.project_dir
        base_ref = self.parent.Context.project_git_base_ref
        arcadia_path = self.parent.Context.arcadia_path
        ref_name = ref_to_merge['ref']
        repo_params = {
            'owner': self.parent.Parameters.project_github_owner,
            'repo': self.parent.Parameters.project_github_repo,
            'email': ref_to_merge['email'],
            'login': ref_to_merge['author'],
        }
        additional_params = dict()

        if is_arcanum_ref(ref_name):
            logging.debug('Processing Arcanum review request ref: {}'.format(ref_name))

            rr = self.scripts.run_js('script/merge-queue/fetch-arcanum-review.js', ref_name)
            rr_id = rr.get('id')
            rr_author = rr.get('author', {}).get('name', MQ_ROBOT_LOGIN)
            rr_issues = rr.get('issues', [])

            meta_info = {
                "Arcanum-Review-Request-Id": rr_id,
                "Arcanum-Review-Request-Author": rr_author,
                "Arcanum-Review-Request-Issues": ', '.join(rr_issues)
            }

            meta_info_msg = '\n'.join(
                ['{}: {}'.format(key, value) for key, value in meta_info.items() if value]
            )
            additional_params.update({ "commit-message-body": meta_info_msg })

            logging.debug('Transplanting {}'.format(rr_id))

            logging.debug('Meta information {}'.format(meta_info))

            self.scripts.run_js(
                'script/merge-queue/transplant-arcanum-review.js',
                str(rr_id),
                {
                    'mount-path': self.arc_mount_path,
                    'store-path': self.arc_store_path,
                    'arcadia-path': arcadia_path,
                    'git-repo-path': project_dir,
                },
                work_dir=project_dir,
            )

            logging.debug('Creating GitHub pull request')

            pr = self.scripts.run_js(
                'script/merge-queue/create-github-pr.js',
                {
                    'owner': self.parent.Parameters.project_github_owner,
                    'repo': self.parent.Parameters.project_github_repo,
                    'title': rr.get('summary'),
                    'description': 'Arcanum PR: [{id}](https://a.yandex-team.ru/review/{id})'.format(
                        id=rr_id,
                    ),
                    # head получаем из рабочей директории
                    'base': base_ref,
                    'assignee': rr_author,
                },
                work_dir=project_dir,
            )

            logging.debug('Review request {rr_id} transplanted to GitHub pull request: {pr_url}'.format(
                rr_id=rr_id,
                pr_url=pr.get('html_url'),
            ))

            ref_name = format_github_ref(pr.get('number'))

            logging.debug('New GitHub specific ref {} for further rebasing and merging'.format(ref_name))

            # HEAD указывает на временную ветку, чекаутимся обратно в base_ref чтобы не получить ошибку
            # Error: fatal: Refusing to fetch into current branch refs/heads/{branch} of non-bare repository
            commit = self.project_git_exec('rev-parse', base_ref)
            self.project_git_exec('checkout', '-f', commit)

        self.scripts.run_js(
            'script/merge-queue/rebase-and-force-push.js',
            ref_name,
            base_ref,
            repo_params,
            work_dir=project_dir,
        )

        project_config = self.mq_manager.get_project_config()
        unified_merge_commit = project_config.get('unified_merge_commit', False)

        # Пробрасываем в env флаг `unified_merge_commit`, нужно чтобы КМ последнего merge коммита не дублировался в `Publish` коммит.
        # @see https://st.yandex-team.ru/FEI-23809
        if unified_merge_commit is True:
            os.environ['unified_merge_commit'] = 'true'

        # При `unified_merge_commit: False` изменения от `afterMerge` комманд должны быть в отдельном коммите, вне merge-коммита пулл-реквеста.
        # Для этого создаём merge-коммит до запуска команд из `afterMerge`.
        # @see https://st.yandex-team.ru/FEI-22718
        if unified_merge_commit is False:
            logging.debug('Creating merge commit before execution afterMerge commands')

            self.scripts.run_js(
                'script/merge-queue/local-merge.js',
                ref_name,
                base_ref,
                repo_params,
                additional_params,
                work_dir=project_dir,
            )

        try:
            self.exec_after_merge_commands()
        except Exception as e:
            logging.exception('Exception occurred during after-merge phase')
            self.Context.merge_queue_task_exception = e.message
            self.mq_manager.merge_error_handler(e)

            raise

        # При `unified_merge_commit: True` изменения от `afterMerge` комманд должны быть в merge-коммите пулл-реквеста.
        # Для этого создаём merge-коммит после запуска команд из `afterMerge`.
        # @see https://st.yandex-team.ru/FEI-22718
        if unified_merge_commit is True:
            logging.debug('Creating merge commit after execution afterMerge commands')

            self.scripts.run_js(
                'script/merge-queue/local-merge-commit.js',
                ref_name,
                'HEAD',
                base_ref,
                repo_params,
                additional_params,
                work_dir=project_dir,
            )

        self.scripts.run_js(
            'script/merge-queue/push.js',
            base_ref,
            work_dir=project_dir,
        )

        self.Context.merged_pr.append(strip_ref_prefix(ref_name))

    def on_finish(self, prev_status, status):
        super(SandboxCiMerge, self).on_finish(prev_status, status)

        with self.memoize_stage.save_merged_pr():
            self.Parameters.merged_pr = self.Context.merged_pr
