# -*- coding: utf-8 -*-

import logging
import json
import traceback
import time
import os
import re
import requests
import copy

from contextlib import nested

from sandbox import sdk2

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt

from sandbox.common.errors import TaskFailure
from sandbox.common.utils import singleton_property
from sandbox.common.utils import get_task_link

from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import prioritizer, list_utils
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.sandbox_ci_merge import SandboxCiMerge
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Debug
from sandbox.projects.sandbox_ci.utils.github import GitHubApi
from sandbox.projects.sandbox_ci.utils.task_history import TaskHistory
from sandbox.projects.sandbox_ci.managers.profiler import ACTION_PROFILE_HISTORY_STATUS_FILTER
from sandbox.projects.sandbox_ci.managers.arc.context import create_arc_context
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.process import CustomOsEnviron
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sdk2.helpers.misc import MemoizeStage
from sandbox.projects.common.yasm import push_api
from sandbox.projects.common.vcs.arc import ArcCommandFailed
from sandbox.projects.sandbox_ci.task import ProjectPathsMixin


SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0'
RAMDRIVE_SIZE = 15 * 1024

class MemoizeStageOnBaseCommit(MemoizeStage):
    def __init__(self, task, stage_name):
        name = '{}_{}'.format(stage_name, task.Context.project_git_base_commit)
        super(MemoizeStageOnBaseCommit, self).__init__(task, name)


class MemoizeStageOnBaseCommitCreater(object):
    def __init__(self, task):
        self.__task = task

    def __getattr__(self, name):
        return MemoizeStageOnBaseCommit(self.__task, name)

    def __getitem__(self, name):
        return MemoizeStageOnBaseCommit(self.__task, name)


class SandboxCiMergeQueue(TasksResourceRequirement, ManagersTaskMixin, ProjectPathsMixin, sdk2.Task):
    """Очередь для влития Pull Request'ов в дефолтную ветку проекта"""

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL
        environments = (
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
            PipEnvironment('retrying', use_wheel=True, version="1.3.3"),
        )
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

    class Parameters(parameters.CommonParameters):
        kill_timeout = 3 * 3600

        project_github_repository = parameters.ProjectGitHubRepositoryParameters

        with sdk2.parameters.Group('Pull Request') as pull_request_block:
            pull_request_number = sdk2.parameters.Integer(
                'Номер пулл-реквеста',
                required=True,
            )
            pull_request_head_commit = parameters.project_git_head_commit()

        with sdk2.parameters.Group('Build') as build_block:
            success_build_uuid = sdk2.parameters.String(
                'uuid успешно завершенного билда',
                default=''
            )
            success_build_task_id = sdk2.parameters.String(
                'task id успешно завершенного билда',
                default=''
            )
            should_skip_rebase_and_check = sdk2.parameters.Bool(
                'Параметр, который отвечает за то, нужно ли запускать проверки перед вливанием',
                default=False
            )

        with sdk2.parameters.Group(u'GitHub → Arcanum') as arcanum_migration_block:
            pull_request_provider = sdk2.parameters.String(
                'Провайдер пулл-реквеста',
                default='github',
                choices=[
                    ('GitHub', 'github'),
                    ('Arcanum', 'arcanum'),
                ],
                description='Из какой системы был отправлен пулл-реквест',
            )
            pull_request_merge_provider = parameters.pull_request_merge_provider()

        with sdk2.parameters.Group('Автор merge & rebase') as author_block:
            author_name = sdk2.parameters.String(
                'Имя git-пользователя',
                default='robot-merge-queue',
                required=True,
                description='Используется, чтобы указать автора команд git rebase и git merge',
            )
            author_email = sdk2.parameters.String(
                'Почта git-пользователя',
                default='robot-merge-queue@yandex-team.ru',
                required=True,
                description='Используется, чтобы указать автора команд git rebase и git merge',
            )

        with sdk2.parameters.Group('Merge Queue') as mq_block:
            ssh_key_vault_name = sdk2.parameters.String(
                'Имя переменной в vault-е (ssh key)',
                default='robot-merge-queue.id_rsa',
                required=True,
                description='Имя переменной в vault-е, в которй лежит приватный SSH ключ пользователя',
            )
            github_api_token_vault_name = sdk2.parameters.String(
                'Имя переменной в vault-е (github api token)',
                default='robot-merge-queue.github_api_token',
                required=True,
                description='Имя переменной в vault-е, в которй лежит github api token пользователя',
            )
            config = sdk2.parameters.String(
                'Конфигурация Merge Queue',
                multiline=True,
                required=True,
                description='Содержимое конфигурационного файла проекта (.config/merge-queue.json)',
            )
            refs_to_merge = parameters.refs_to_merge()

        with sdk2.parameters.Group('arc') as arc_block:
            use_arc = sdk2.parameters.Bool(
                'Use arc',
                description='Использовать arc для чекаута',
                default=False
            )

        with sdk2.parameters.Group('Отправка статистики') as stats_block:
            need_send_profile_stat = sdk2.parameters.Bool(
                'Отправлять результаты профилирования',
                default=True,
                description='Время выполнения каждого этапа отправляется в StatFace сервис',
            )

        with sdk2.parameters.Group('Webhook') as webhook_block:
            webhook_urls = sdk2.parameters.List('Urls')

        scripts_sources = parameters.ScriptsSourcesParameters
        environment = parameters.EnvironmentParameters

        node_js = parameters.NodeJsParameters

        with sdk2.parameters.Output():
            merge_fail_reason = sdk2.parameters.String('Fail reason')
            mq_error_code = sdk2.parameters.String('Error code id')

    class Context(sdk2.Task.Context):
        pull_request = None  # Github payload с Pull Request'ом
        project_task_id = None
        project_git_url = None
        project_git_base_ref = None
        project_git_base_commit = None
        project_git_head_ref = None
        project_git_head_commit = None
        project_git_tree_hash = None
        project_pull_request_labels = []
        # Ошибки, которые не влияют на влитие PR'а
        noncritical_errors = []
        # Критичная ошибка, отправляется в сервис merge-queue
        merge_queue_task_exception = ''
        # Тип ошибки, отправляются в сервис merge-queue
        merge_queue_error_name = ''
        # Код ошибки, отправляется в сервис merge-queue
        merge_queue_error_code = None
        # Непрошедшие обязательные проверки (завершились с любым статусом, кроме `success`)
        unsatisfied_checks = []
        # Упавшие обязательные проверки (завершились со статусом `failure` или `error`)
        failed_checks = []
        arc_tree_hash = None
        # Отключает создание файла с отладочным выводом содержимого ramdrive в файл ramdrive_usage.yaml
        __do_not_dump_ramdrive = True

    def on_save(self):
        super(SandboxCiMergeQueue, self).on_save()
        self.Parameters.priority = prioritizer.get_priority(self)

        try:
            self.Parameters.use_arc = self.use_arc
        except Exception as error:
            logging.exception('Cannot update environ parameter value: {}'.format(error))

    def on_prepare(self):
        self.init_environ()

        with self.profiler.actions.clone_scripts('Cloning scripts'):
            self.scripts.sync_resource()

        with self.memoize_stage.init_parameters:
            self.Context.project_git_head_commit = self.Parameters.pull_request_head_commit

            with self.profiler.actions.init_parameters('Initializing parameters'):
                self.init_project_source_parameters()

        with self.memoize_stage.start:
            self.send_webhooks(ctt.Status.PREPARING)

    @in_case_of('arcanum_pr_provider', 'arcanum_init_project_source_parameters')
    def init_project_source_parameters(self):
        pull_request = self.pull_request

        self.check_head_ref(pull_request)

        self.Context.project_git_url = pull_request['base']['repo']['ssh_url']
        self.Context.project_git_base_ref = pull_request['base']['ref']
        self.Context.project_git_base_commit = self.get_base_ref_sha()
        self.Context.project_git_head_ref = 'pull/{}'.format(pull_request['number'])
        self.Context.project_git_head_commit = pull_request['head']['sha']
        self.Context.project_pull_request_labels = self.pull_request_labels

        try:
            # Добавляем arcadia_path для балкового вливания нескольких пулл-реквестов из разных VCS.
            # Нужно для того, чтобы выполнить трансплантацию PR-а из arcanum в github
            pair = self.find_arcadia_github_pair(self.Parameters.project_github_owner, self.Parameters.project_github_repo)
            self.Context.arcadia_path = pair.get('arcadia', {}).get('path')
        except Exception as error:
            logging.debug('Cannot find pair for owner {} and repo {}. Error: {}'.format(
                self.Parameters.project_github_owner, self.Parameters.project_github_repo, error
            ))

        self.Context.save()

        logging.debug('Defined git url: {}'.format(self.Context.project_git_url))
        logging.debug('Defined base ref: {}'.format(self.Context.project_git_base_ref))
        logging.debug('Defined base commit: {}'.format(self.Context.project_git_base_commit))
        logging.debug('Defined head ref: {}'.format(self.Context.project_git_head_ref))
        logging.debug('Defined head commit: {}'.format(self.Context.project_git_head_commit))
        logging.debug('Defined labels: {}'.format(', '.join(self.Context.project_pull_request_labels)))

    def arcanum_init_project_source_parameters(self):
        owner = self.Parameters.project_github_owner
        repo = self.Parameters.project_github_repo
        review_request_id = self.Parameters.pull_request_number

        pair = self.find_arcadia_github_pair(owner, repo)

        if self.Parameters.pull_request_merge_provider == "github":
            self.Context.project_git_base_ref = pair.get('github', {}).get('branch')

        if self.Parameters.pull_request_merge_provider == "arcanum":
            self.Context.project_git_base_ref = "trunk"

        self.Context.project_git_head_ref = 'review/{}'.format(review_request_id)

        self.Context.project_git_url = self.format_git_url(owner, repo)
        self.Context.arcadia_path = pair.get('arcadia', {}).get('path')

        self.Context.save()

        logging.debug('Defined git url: {}'.format(self.Context.project_git_url))
        logging.debug('Defined arcadia path: {}'.format(self.Context.arcadia_path))
        logging.debug('Defined base ref: {}'.format(self.Context.project_git_base_ref))
        logging.debug('Defined head ref: {}'.format(self.Context.project_git_head_ref))

    def find_arcadia_github_pair(self, owner, repo):
        try:
            pair = self.scripts.run_js(
                'script/merge-queue/find-arcadia-github-pair.js',
                {
                    'owner': owner,
                    'repo': repo,
                },
            )

            logging.debug('Pair for owner: {owner} and repo: {repo}: {pair}'.format(
                owner=owner,
                repo=repo,
                pair=pair,
            ))

            return pair
        except SandboxSubprocessError:
            message = 'Failed to find Arcadia <-> GitHub pair for {owner}/{repo}'.format(
                owner=owner,
                repo=repo,
            )
            self.Context.merge_queue_task_exception = message

            # Вывод ошибки в описание таски и проброс в failReason нас интересуют только в случае Арканума
            if self.arcanum_pr_provider:
                self.mq_manager.merge_error_handler(Exception(message))

            raise TaskFailure(message)

    @property
    def arcanum_pr_provider(self):
        return self.Parameters.pull_request_provider == 'arcanum'

    @property
    def pull_request(self):
        logging.debug('Getting pull request info')
        pull_request_info = self.get_pull_request_info()

        logging.debug('Updated pull request info: {}'.format(pull_request_info))
        self.Context.pull_request = pull_request_info

        return pull_request_info

    @property
    def total_execution_time(self):
        if self.Context.first_execution_started_at is ctm.NotExists:
            return 0

        return int(time.time() - self.Context.first_execution_started_at)

    def get_pull_request_info(self):
        return GitHubApi().get_pr_info(
            self.Parameters.project_github_owner,
            self.Parameters.project_github_repo,
            self.Parameters.pull_request_number,
        )

    def format_git_url(self, owner, repo):
        return 'git@github.yandex-team.ru:{owner}/{repo}.git'.format(
            owner=owner,
            repo=repo
        )

    # TODO: use ci script
    # @see FEI-6427
    def get_base_ref_sha(self):
        logging.debug('Getting base commit')
        ref_path = 'repos/{owner}/{repo}/git/refs/heads/{ref}'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
            ref=self.Context.project_git_base_ref,
        )
        git_ref = GitHubApi().request(ref_path)

        sha = git_ref['object']['sha']
        logging.debug('Got base commit: {}'.format(sha))
        return sha

    def on_execute(self):
        from retrying import retry

        def should_retry_create_arc_context(exception):
            return isinstance(exception, ArcCommandFailed)

        @retry(wait_exponential_multiplier=2000, stop_max_attempt_number=3, retry_on_exception=should_retry_create_arc_context)
        def create_arc_context_with_retry(*args, **kwargs):
            return create_arc_context(*args, **kwargs)

        self._create_arc_context = create_arc_context_with_retry

        self.execute()

    def build_execute_contexts(self):
        custom_env = {'GITHUB_API_TOKEN': self.vault.read(self.Parameters.github_api_token_vault_name)}

        # явно переопределяем в env переменную GITHUB_API_TOKEN(по умолчанию robot-serp-bot) для MQ (robot-merge-queue)
        # потому что все скрипты под капотом используют переменную GITHUB_API_TOKEN, например: getRequiredChecksContexts
        contexts = (CustomOsEnviron(custom_env),)

        if self.use_arc:
            project_config = self.mq_manager.get_project_config()
            arc_config = project_config.get('arc', {})

            extra_params = []
            if arc_config.get('no_threads', False):
                extra_params.append('--no-threads')
            if arc_config.get('use_vfs2', False):
                extra_params.extend(['--vfs-version', '2'])

            arc_apply_attrs = arc_config.get('apply_attrs', {})

            mount_options = dict(
                mount_point=self.arc_mount_path,
                store_path=self.arc_store_path,
                object_store_path=str(self.arc_object_store_path),
                extra_params=extra_params,
                arc_apply_attrs=arc_apply_attrs,
            )
            contexts += (self._create_arc_context(**mount_options),)

        return contexts

    def __should_skip_rebase_and_check(self):
        return self.Parameters.should_skip_rebase_and_check or not self.config.get('strict', True)

    @in_case_of('arcanum_pr_provider', 'arcanum_execute')
    def execute(self):
        contexts = self.build_execute_contexts()

        with nested(*contexts):
            self.check_pull_request()

            if self.__should_skip_rebase_and_check():
                self.merge()
                self.__check_merge_task()
                return

            self.rebase()
            self.add_tags()

            # Rebase + force push не запускает Sandbox-задачу по веб-хукам (настраивается в конфиге проекта).
            # Запускаем обязательные проверки непосредственно в этой задаче.
            # @see FEI-6442
            if not self.__is_merged():
                self.run_required_checks()

            with self.memoize_stage_on_base_commit.check_required_checks:
                self.check_required_checks()

            # Если за время выполнения проверок другой PR был влит в обход MQ, нужно перезапустить обязательные проверки.
            #
            # https://st.yandex-team.ru/FEI-16411
            #if not self.__is_merged() and self.base_ref_was_changed():
            #    return self.restart()

            # Проверяем, что за время выполнения проверок PR не был изменён или закрыт.
            self.check_pull_request()
            self.merge()
            self.__check_merge_task()

    def arcanum_execute(self):
        contexts = self.build_execute_contexts()

        with nested(*contexts):
            # all checks for "arcanum" in arcanum-merge-queue-cli
            # @see https://github.yandex-team.ru/search-interfaces/infratest/tree/master/packages/arcanum-merge-queue-cli#arcanum-merge-queue-cli
            # @see https://st.yandex-team.ru/FEI-23135
            if self.Parameters.pull_request_merge_provider == "github":
                self.check_review_request()

            strict_mode = not self.__should_skip_rebase_and_check()
            poll_mode = self.config.get('poll', False)

            if poll_mode:
                # Сейчас нет поддержки поллинга, внедрим в FEI-17314.
                logging.debug('The polling is not yet supported. Falling back to non-strict mode.')
                strict_mode = False

            if strict_mode:
                with self.memoize_stage_on_base_commit.rebase:
                    # all checks for "arcanum" in arcanum-merge-queue-cli
                    # @see https://github.yandex-team.ru/search-interfaces/infratest/tree/master/packages/arcanum-merge-queue-cli#arcanum-merge-queue-cli
                    # @see https://st.yandex-team.ru/FEI-21945
                    if self.Parameters.pull_request_merge_provider == "github":
                        # В арке нельзя пушить в чужие ветки, но чекаутить их можно.
                        # Локально чекаутим и ребейзим ревью-реквест на trunk чтобы выявить мердж конфликты.
                        self.rebase_review_request()

                if not self.__is_merged():
                    self.run_required_checks()

                with self.memoize_stage_on_base_commit.report_checks:
                    self.review_request_report_checks()

            self.merge()
            self.__check_merge_task()

    def check_review_request(self):
        if self.__is_merged():
            return

        try:
            review_request = self.review_request

            self.assert_request_state(review_request)
            self.assert_request_head(review_request)
        except AssertionError as e:
            self.Context.merge_queue_task_exception = e.message
            self.mq_manager.merge_error_handler(e)
            raise TaskFailure(self.Context.merge_queue_task_exception)

    @property
    def review_request(self):
        review_request_id = self.Parameters.pull_request_number
        fields = [
            'id',
            'state',
            'vcs(type,name,from_branch)',
            'checks(system,type,required,satisfied,status,system_check_uri)',
            'active_diff_set(id,status,arc_branch_heads(from_id,to_id,merge_id))',
        ]

        logging.debug('Fetching review request {}'.format(review_request_id))

        review_request_info = self.scripts.run_js(
            'script/merge-queue/fetch-arcanum-review.js',
            str(review_request_id),
            {'fields': ','.join(fields)}
        )

        logging.debug('Review request info: {}'.format(review_request_info))

        self.Context.review_request = review_request_info

        return review_request_info

    @staticmethod
    def assert_request_state(review_request):
        rr_id = review_request.get('id')
        actual = review_request.get('state')
        expected = 'open'

        logging.debug('Asserting review request {id} state, actual state is "{actual}", expected "{expected}"'.format(
            id=rr_id, actual=actual, expected=expected,
        ))

        assert actual == expected, 'review request {id} state is "{actual}", expected "{expected}"'.format(
            id=rr_id, actual=actual, expected=expected,
        )

    def assert_request_head(self, review_request):
        rr_id = review_request.get('id')
        actual = review_request.get('active_diff_set', {}).get('arc_branch_heads', {}).get('from_id')
        expected = self.Context.project_git_head_commit

        logging.debug('Asserting review request {id} head, actual commit is "{actual}", expected "{expected}"'.format(
            id=rr_id,
            actual=actual,
            expected=expected,
        ))

        assert actual == expected, 'review request {id} head commit is "{actual}", expected "{expected}"'.format(
            id=rr_id, actual=actual, expected=expected,
        )

    def rebase_review_request(self):
        git_url = self.Context.project_git_url
        params = {
            'use-arc': True,
            'dump-hashes': True,
            'base-ref': self.Context.project_git_base_ref,
            'merge-ref': self.Context.project_git_head_ref
        }

        # ключ нужен, потому что sha коммита в arc берется из git (по гитовому sha)
        with Debug('*'), GitRetryWrapper(), self.vault.ssh_key(self.Parameters.ssh_key_vault_name):
            try:
                hashes = self.scripts.run_js(
                    'script/prepare-working-copy.js',
                    git_url,
                    self.working_path('mirror'),
                    params,
                    log_prefix='rebase-review-request',
                )
                logging.debug('Review request rebase info: {}'.format(hashes))
                self.Context.arc_tree_hash = hashes.get('merged', hashes['base'])['treeHash']
            except SandboxSubprocessError as e:
                self.mq_manager.merge_error_handler(e)

                raise

    def add_tags(self):
        try:
            project_tags = self.tags_manager.get_project_tags(
                str(self.project_dir),
                self.Parameters.project_github_owner,
                self.Parameters.project_github_repo,
                self.Context.project_git_base_commit
            )

            tags = map(lambda tag: tag.lower(), project_tags + self.Parameters.tags)

            logging.debug('Adding tags: {}'.format(tags))

            self.Parameters.tags = list(set(tags))
        except Exception as error:
            logging.exception('Cannot add tags {}'.format(error))

    def restart(self):
        logging.debug('Task was restarted')
        self.Context.restarted = True
        self.send_webhooks(ctt.Status.EXECUTING)
        self.execute()

    def check_pull_request(self):
        if self.__is_merged():
            return

        pull_request = self.pull_request

        self.check_pull_request_state(pull_request)
        self.check_head_ref(pull_request)

    def check_pull_request_state(self, pull_request):
        state = pull_request['state']

        if state == 'open':
            logging.debug('Pull request {} is open'.format(self.pull_request_id))
            return

        self.Context.merge_queue_task_exception = 'pull request {pr_id} is {state}'.format(
            pr_id=self.pull_request_id,
            state=state,
        )

        raise TaskFailure(self.Context.merge_queue_task_exception)

    def check_head_ref(self, pull_request):
        head_sha = pull_request['head']['sha']

        logging.debug('head_sha is {}'.format(head_sha))
        logging.debug('project_git_head_commit is {}'.format(self.Context.project_git_head_commit))

        if self.Context.project_git_head_commit == head_sha:
            logging.debug('Pull request branch is not changed')
            return

        self.Context.merge_queue_task_exception = 'pull request {} has been updated'.format(self.pull_request_id)

        raise TaskFailure(self.Context.merge_queue_task_exception)

    def base_ref_was_changed(self):
        """
        Возвращает True, если базовая ветка была изменена.
        Обновляет base_sha в контексте MQ-задачи на актуальный.
        """
        base_sha = self.get_base_ref_sha()

        if self.Context.project_git_base_commit != base_sha:
            logging.info('Base branch is updated ({old_commit} → {new_commit})'.format(
                old_commit=self.Context.project_git_base_commit,
                new_commit=base_sha,
            ))

            self.Context.project_git_base_commit = base_sha
            return True

        logging.debug('Base branch is not changed')
        return False

    @singleton_property
    def config(self):
        try:
            return json.loads(self.Parameters.config)
        except ValueError:
            self.Context.merge_queue_task_exception = 'could not parse MQ config'
            raise TaskFailure(self.Context.merge_queue_task_exception)

    @singleton_property
    def use_arc(self):
        use_arc = self.Parameters.use_arc

        logging.debug('Parameter use_arc value: {}'.format(use_arc))

        try:
            # В текущем классе переопределено свойство `config` и используется в других тасках.
            # @see FEI-15532
            project_config = self.mq_manager.get_project_config()

            use_arc = project_config.get('use_arc', None)

            if use_arc is None:
                use_arc = self.Parameters.use_arc

        except Exception as error:
            logging.exception('Cannot get project config: {}'.format(error))

        logging.debug('Use arc to checkout: {}'.format(use_arc))

        return use_arc

    @singleton_property
    def pull_request_id(self):
        return '{owner}/{repo}#{number}'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
            number=self.Parameters.pull_request_number,
        )

    @singleton_property
    def memoize_stage_on_base_commit(self):
        """
        Обертка над memoize_stage в sdk2.Task, которая позволяет привязать стадию к project_git_base_commit.

        Пример, когда может использоваться:
        в пулл-реквест пушат, пока таска находится в wait_task,
        соответственно, предыдущая стадия должна быть инвалидирована
        """
        return MemoizeStageOnBaseCommitCreater(self)

    @singleton_property
    def pull_request_labels(self):
        """
        Возвращает список лэйблов пулл-реквеста

        :rtype: list of str
        """
        logging.debug('Getting pull-request labels')
        path = 'repos/{owner}/{repo}/issues/{number}/labels'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
            number=self.Parameters.pull_request_number,
        )
        res = GitHubApi().request(path)

        labels = map(lambda label: label['name'], res)
        logging.debug('Got pull-request labels: {}'.format(labels))

        return labels

    def run_required_checks(self):
        """
        Дожидается запущенную прогревочную задачу и переиспользует, если она в статусе SUCCESS,
        либо запускает проектную задачу с нуля в том случае, если прогревочная задача не найдена или
        завершилась в статусе FAILURE.

        Если в MQ-конфиге нет информации о проектной задаче, ожидает прокраски обязательных проверок
        для HEAD-коммита пулл-реквеста.
        """
        if not self.config.get('projectTask'):
            logging.warning('Could not run project task, project task config is not found')

            if self.config.get('poll'):
                self.poll_required_checks(self.Context.project_git_head_commit)

            return None

        with self.profiler.actions.run_required_checks('Running required checks'):
            logging.debug('Waiting required checks for {base_sha}...{head_sha}'.format(
                base_sha=self.Context.project_git_base_commit,
                head_sha=self.Context.project_git_head_commit,
            ))

            with self.memoize_stage_on_base_commit.cache_task_run:
                cache_task = self.find_project_task_cache()

                if cache_task:
                    logging.debug('Cache task found, id={}'.format(cache_task['id']))
                    try:
                        self.__inc_priority(cache_task)
                        logging.debug('Priority for cache task with id {} was increased to {}'.format(cache_task['id'], self.Parameters.priority))
                    except Exception as e:
                        logging.warning('Could not increase priority for task {}'.format(cache_task['id']))
                        logging.warning(e)

                    self.set_info(self.get_info_message_about_project_task(cache_task['id'], True), do_escape=False)
                    self.wait_project_task(cache_task)

            with self.memoize_stage_on_base_commit.cache_task_check:
                self.register_cache_waiting_action()

                cache_task = self.get_project_task()

                if self.has_cache_been_used(cache_task):
                    self.register_cache_task_hit()
                else:
                    cache_fail_reason = self.get_cache_fail_reason(cache_task)

                    logging.debug('Cache not used. Reason: "{}"'.format(cache_fail_reason))

                    self.register_cache_task_miss(cache_fail_reason)

                    logging.debug('Project task is about to start')
                    project_task = self.run_project_task()
                    self.set_info(self.get_info_message_about_project_task(project_task['id'], False), do_escape=False)
                    self.wait_project_task(project_task)

    def poll_required_checks(self, commit_sha):
        with Debug('serp:*'), GitRetryWrapper(), self.vault.ssh_key(self.Parameters.ssh_key_vault_name):
            owner = self.Parameters.project_github_owner
            repo = self.Parameters.project_github_repo
            commit_id = '{owner}/{repo}#{commit_sha}'.format(
                owner=owner,
                repo=repo,
                commit_sha=commit_sha
            )

            logging.debug('Polling required checks for {}'.format(commit_id))

            try:
                self.scripts.run_js(
                    'script/merge-queue/poll-required-checks.js',
                    {
                        'owner': owner,
                        'repo': repo,
                        'sha': commit_sha,
                        'attempt-delay': self.config.get('poll', {}).get('attemptDelay'),
                        'max-retries': self.config.get('poll', {}).get('maxRetries')
                    },
                    work_dir=self.project_dir,
                )
            except SandboxSubprocessError as e:
                message = 'Failed polling required checks for {}'.format(commit_id)
                self.Context.merge_queue_task_exception = message
                self.mq_manager.merge_error_handler(Exception(message))

                raise

    def register_cache_waiting_action(self):
        """
        Хэлпер для регистрации времени ожидания прогревочной задачи.
        Фильтрует полный набор статусов прогревочной задачи до набора статусов, который имеет смысл отслеживать. После
        фильтрации находит разницу между текущим статусом и тем, что был указан в качестве интересующего (WAIT_TASK).
        """
        history = self.server.task[self.id].audit.read()
        history_filtered = filter(lambda state: state.get('status') in ACTION_PROFILE_HISTORY_STATUS_FILTER, history)
        history_tracker = TaskHistory(history_filtered)

        self.profiler.register_action(
            'waiting_for_cache_task',
            description='Waiting for cache task',
            duration=history_tracker.get_transition_times_from(ctt.Status.WAIT_TASK),
        )

    def get_info_message_about_project_task(self, task_id, cache_found):
        """
        Формирует сообщение об ожидаемых задачах

        :param task_id: идентификатор таски
        :type task_id: int
        :param cache_found: нашлась ли прогревочная таска
        :type cache_found: bool
        :rtype: str
        """
        config = self.config.get('projectTask', {})
        task_type = 'cache' if cache_found else 'project'
        task_link = get_task_link(task_id)

        # TODO: FEI-14453 оторвать ссылку на кэши по related_task после полного переключения на MQ на монге
        return ''.join([
            u'Waiting {2} task: <a href="{3}">{1}#{0}</a>, ',
            u'<a href=\'/tasks?type={1}&input_parameters={{"related_task":{4}}}\'>cache tasks by related task</a>, ',
            u'<a href=\'/tasks?type={1}&input_parameters={{"pull_request_number":{5}}}\'>cache tasks by PR number</a>',
        ]).format(task_id, config['type'], task_type, task_link, self.id, self.Parameters.pull_request_number)

    def has_cache_been_used(self, cache_task):
        """
        Проверяет удалось ли, в итоге, использовать кэш.

        Зависит от нескольких факторов:
        - наличия прогревочной таски;
        - успешность прохождения обязательных проверок в прогревочной таске;
        - совпадает ли tree hash для MQ и прогревочной таски.

        :param cache_task: Экземпляр SANDBOX_CI_<PROJECT_NAME>
        :type cache_task: sandbox.projects.sandbox_ci.task.BaseMetaTask.BaseMetaTask
        :rtype: bool
        """
        if not cache_task:
            logging.debug('Cache task not found by id={}'.format(self.Context.project_task_id))
            return False

        if not self.use_arc and cache_task.use_arc:
            logging.debug(
                'cache task were built using arc, but MQ were started without use_arc parameter, could not compare '
                'tree hashes'
            )
            return False

        arc_tree_hash_equals = cache_task.Parameters.project_tree_hash == self.Context.arc_tree_hash
        git_tree_hash_equals = cache_task.Parameters.project_tree_hash == self.Context.project_git_tree_hash

        if self.use_arc and cache_task.use_arc and not arc_tree_hash_equals:
            logging.debug(
                'Arc tree hash of cache task ({}) differs from MQ ({})'.format(
                    cache_task.Parameters.project_tree_hash,
                    self.Context.arc_tree_hash,
                ),
            )
            return False

        if not cache_task.use_arc and not git_tree_hash_equals:
            logging.debug(
                'Tree hash of cache task ({}) differs from MQ ({})'.format(
                    cache_task.Parameters.project_tree_hash,
                    self.Context.project_git_tree_hash,
                ),
            )
            return False

        return self.check_passed_required_checks(cache_task)

    def check_passed_required_checks(self, cache_task):
        """
        Проверяет статусы обязательных проверок в указанной таске.

        :param cache_task: Экземпляр SANDBOX_CI_<PROJECT_NAME>
        :type cache_task: sandbox.projects.sandbox_ci.task.BaseMetaTask.BaseMetaTask
        :rtype: bool
        """
        required_checks = self.config.get('requiredChecks')

        if not required_checks:
            return False

        github_statuses = cache_task.Context.github_statuses
        logging.debug('Got github statuses of cache_task: {}'.format(github_statuses))

        if not github_statuses:
            return False

        for status in github_statuses:
            state = status['state']
            context = status['context']

            if state != 'success' and context in required_checks:
                logging.debug('{context} is required to be success but {state} found'.format(
                    context=context.encode('utf-8'),
                    state=state,
                ))
                return False

        logging.debug('All required checks in cache task passed: {}'.format(required_checks))
        return True

    def get_cache_fail_reason(self, cache_task):
        """
        Возвращает причину того, почему кэш не был переиспользован.

        :param cache_task: Экземпляр SANDBOX_CI_<PROJECT_NAME>
        :type cache_task: sandbox.projects.sandbox_ci.task.BaseMetaTask.BaseMetaTask
        :rtype: str
        """
        if not cache_task:
            return 'Not found'

        if not cache_task.Context.github_statuses:
            return 'There is no github statuses in cache task'

        if cache_task.status == ctt.Status.FAILURE:
            failed_statuses = self.get_failed_github_statuses(cache_task)

            return 'Failed steps: ' + ', '.join(failed_statuses) if failed_statuses else 'Failed'

        if cache_task.Parameters.project_tree_hash != self.Context.project_git_tree_hash:
            return 'Tree hash does not match'

        return 'Broken'

    def get_failed_github_statuses(self, project_task):
        """
        Возвращает список FAILURE-статусов для переданной задачи.

        :param project_task: инстанс SANDBOX_CI_<PROJECT_NAME>
        :type project_task: sandbox.projects.sandbox_ci.task.BaseMetaTask.BaseMetaTask
        :rtype: list of str
        """
        github_statuses = project_task.Context.github_statuses
        if not github_statuses:
            return []

        return [status['context'].encode('utf8') for status in github_statuses if status['state'] == 'failure']

    def register_cache_task_miss(self, description=None):
        """
        Хэлпер для регистрации отсутствия кэша.

        :param description: Краткое описание того, почему кэш не был переиспользован
        :type description: str
        """
        self.cache_profiler.miss('merge_queue_task_cache', 'task', description)

    def register_cache_task_hit(self, description=None):
        """
        Хэлпер для регистрации присутствия кэша.

        :param description: Краткое описание того, почему кэш был переиспользован
        :type description: str
        """
        self.cache_profiler.hit('merge_queue_task_cache', 'task', description)

    def wait_project_task(self, project_task):
        project_task_id = project_task['id']

        logging.debug('Waiting task #{}'.format(project_task_id))
        self.Context.project_task_id = project_task_id
        # Максимальное время ожидания проектной задачи, равное удвоенному времени её жизни (накладные расходы Sandbox)
        wait_timeout = project_task['kill_timeout'] * 2

        raise sdk2.WaitTask(project_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, timeout=wait_timeout)

    def find_project_task_cache(self):
        """
        Находит прогревочную проектную задачу, созданную в MQ-сервисе.

        :return: инстанс SANDBOX_CI_<PROJECT_NAME>
        :rtype: sandbox.projects.sandbox_ci.task.BaseMetaTask.BaseMetaTask
        """
        project_task_config = self.config.get('projectTask')
        project_task_parameters_config = project_task_config['custom_fields']
        tree_hash = self.Context.project_git_tree_hash

        logging.debug('Searching cache for project task by tree hash {}'.format(tree_hash))

        input_parameters = {}
        for field in project_task_parameters_config:
            input_parameters[field['name']] = field['value']

        query = sdk2.Task.find(
            type=project_task_config['type'],
            project_github_owner=self.Parameters.project_github_owner,
            project_github_repo=self.Parameters.project_github_repo,
            input_parameters=dict(
                pull_request_number=self.Parameters.pull_request_number,
                project_git_base_ref=input_parameters['project_git_base_ref'],
                project_build_context=input_parameters['project_build_context'],
            ),
            children=True,
        )
        project_task_cache = next(iter(query.order(-sdk2.Task.id).limit(1)), None)
        logging.debug('Cache task for PR {} is {}'.format(self.Parameters.pull_request_number, project_task_cache))

        if not project_task_cache:
            query = sdk2.Task.find(
                type=project_task_config['type'],
                input_parameters=dict(
                    related_task=self.id,
                    project_git_base_ref=input_parameters['project_git_base_ref'],
                    project_build_context=input_parameters['project_build_context'],
                ),
                children=True,
            )
            project_task_cache = next(iter(query.order(-sdk2.Task.id).limit(1)), None)
            logging.debug('Cache task for related task {} is {}'.format(self.id, project_task_cache))

        if project_task_cache:
            logging.info('Found project task cache: {}'.format(project_task_cache))

            # Если на момент поиска прогревочной задачи та находится в DRAFT-те, то скорее всего она там и останется
            if project_task_cache.status in ctt.Status.Group.DRAFT:
                logging.info('Cache task is in "DRAFT" status. Run project task')
                return None

            # Имитируем результат `server.task` (так создаётся проектная задача в `run_project_task`)
            return dict(id=project_task_cache.id, kill_timeout=project_task_cache.Parameters.kill_timeout)
        else:
            logging.info('Cache for project task is not found')

    def run_project_task(self):
        config = self.config

        logging.debug('MQ config: {}'.format(config))

        project_task_config = config['projectTask']

        custom_fields = {field['name']: field['value'] for field in project_task_config['custom_fields']}

        refs_names = list_utils.map_prop(self.Parameters.refs_to_merge, 'ref')

        custom_fields.update(
            project_git_url=self.Context.project_git_url,
            project_github_owner=self.Parameters.project_github_owner,
            project_github_repo=self.Parameters.project_github_repo,
            project_git_base_ref=self.Context.project_git_base_ref,
            project_git_base_commit=self.Context.project_git_base_commit,
            # Первый merge_ref в списке указывает на head (@see FEI-5174); нужно оторвать этот костыль (@see FEI-16602)
            project_git_merge_ref=[list_utils.last(refs_names)] + refs_names,
            project_git_merge_commit=self.Context.project_git_head_commit,
            project_github_commit=self.Context.project_git_head_commit,
            # Не устанавливаем ограничение на использование семафоров для всех тасок, кроме прогревочных (FEI-7494)
            external_config={'merge-queue.tests.limited_grid': False},
        )

        labels_config = config.get('labels', {})
        logging.debug('MQ config has the following labels: {}'.format(labels_config))

        for label in self.Context.project_pull_request_labels:
            if label not in labels_config:
                continue

            label_config = labels_config[label]

            label_custom_fields = {field['name']: field['value'] for field in label_config.get('custom_fields', [])}

            logging.debug('Label {label} has custom fields: {fields}'.format(
                label=label,
                fields=label_custom_fields,
            ))

            custom_fields.update(label_custom_fields)

        task_type = project_task_config['type']
        task_parameters = dict(
            owner=project_task_config['owner'],
            priority=self.Parameters.priority,
            tags=project_task_config.get('tags', []),
            notifications=project_task_config.get('notifications', []),
            description=self.Parameters.description,
            custom_fields=[{'name': name, 'value': value} for (name, value) in custom_fields.items()],
        )

        logging.debug('Starting project task {type} with params {parameters}'.format(
            type=task_type,
            parameters=task_parameters,
        ))

        return self.meta.run_task(task_type_name=task_type, parameters=task_parameters)

    def get_project_task(self):
        """
        :return: Инстанс проектной задачи SANDBOX_CI_<PROJECT_NAME>
        :rtype: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask or None
        """
        project_task_id = self.Context.project_task_id

        if not project_task_id:
            logging.debug('Project task id is not defined')
            return None

        logging.debug('Searching project task by id {}'.format(project_task_id))
        return sdk2.Task.find(id=project_task_id, children=True).first()

    def merge(self):
        try:
            with self.memoize_stage.mk_merge_subtask:
                merge_task = self.meta.create_subtask(
                    task_type=SandboxCiMerge,
                    description='merge step for {}'.format(self.pull_request_id),
                    ssh_key_vault_name=self.Parameters.ssh_key_vault_name,
                    github_api_token_vault_name=self.Parameters.github_api_token_vault_name,
                    project_github_repo=self.Parameters.project_github_repo,
                    refs_to_merge=self.__refs_to_merge,
                    pull_request_merge_provider=self.Parameters.pull_request_merge_provider,
                    scripts_last_resource=self.Parameters.scripts_last_resource,
                    scripts_resource=self.Parameters.scripts_resource,
                )
                self.Context.merge_task_id = merge_task.id

                raise sdk2.WaitTask(
                    tasks=self.meta.start_subtasks([merge_task]),
                    statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
                )
        except Exception as e:
            logging.exception('Exception occurred during merge phase')
            self.Context.merge_queue_task_exception = e.message
            raise

    @property
    def __refs_to_merge(self):
        """
        Временный костыль (который уйдет, когда откажемся от таски SANDBOX_CI_MERGE_QUEUE в FEI-17629), связанный с тем, что
        project_git_head_commit может поменяться после rebase по основной ветке,
        поэтому необходимо актуализировать это значение в refs_to_merge для соответствующего project_git_head_ref
        """
        refs_to_merge = copy.deepcopy(self.Parameters.refs_to_merge)

        for ref_to_merge in refs_to_merge:
            if ref_to_merge['ref'] == self.Context.project_git_head_ref:
                ref_to_merge['head_sha'] = self.Context.project_git_head_commit

        return refs_to_merge

    def __get_merge_task(self):
        return sdk2.Task.find(id=self.Context.merge_task_id).limit(1).first()

    def __is_merged(self):
        if not self.__has_merge_task():
            return False

        merge_task = self.__get_merge_task()

        if merge_task:
            return merge_task.status in ctt.Status.Group.FINISH or merge_task.status in ctt.Status.Group.BREAK

        return False

    def __has_merge_task(self):
        return self.Context.merge_task_id is not ctm.NotExists

    def __check_merge_task(self):
        if not self.__has_merge_task():
            return

        merge_task = sdk2.Task.find(id=self.Context.merge_task_id).limit(1).first()

        if self.meta.is_any_failed([merge_task]):
            self.__copy_from_ctx(merge_task, ['merge_queue_task_exception', 'merge_queue_error_name', 'merge_queue_error_code'])

            raise TaskFailure('Could not merge, see logs of subtask for more details \n {}'.format(self.Context.merge_queue_error_name))

    def __copy_from_ctx(self, task, props):
        for prop in props:
            value = getattr(task.Context, prop, ctm.NotExists)

            if value is not ctm.NotExists:
                setattr(self.Context, prop, value)

    def check_required_checks(self):
        """Проверяет успешность обязательных проверок в проектной задаче."""
        with self.profiler.actions.check_required_checks('Check required checks'):
            project_task = self.get_project_task()

            if project_task:
                project_task.meta.report_github_statuses(self.Context.project_git_head_commit, force=True)

                if project_task.status == ctt.Status.SUCCESS:
                    logging.debug('Project task succeeded: no need to check github required checks')
                    return

            self.check_github_required_checks(self.Context.project_git_head_commit)

    def check_github_required_checks(self, commit_sha):
        """
        Проверяет github-статусы обязательных проверок для указанного коммита.

        :param commit_sha: хеш коммита
        :type commit_sha: str
        """
        with Debug('serp:*'), GitRetryWrapper():
            res = self.scripts.run_js(
                'script/merge-queue/get-required-checks-info.js',
                {
                    'owner': self.Parameters.project_github_owner,
                    'repo': self.Parameters.project_github_repo,
                    'sha': commit_sha,
                },
            )

            logging.debug('Got required checks info: {}'.format(res))

            failed_checks = self.Context.failed_checks = res['failedChecks']
            unsatisfied_checks = self.Context.unsatisfied_checks = res['unsatisfiedChecks']
            not_passed_checks = failed_checks or unsatisfied_checks

            if not_passed_checks:
                message = 'Failed project required checks: {}'.format(
                    '; '.join(map(lambda arg: arg.encode('utf-8'), not_passed_checks)),
                )
                self.Context.merge_queue_task_exception = message
                self.mq_manager.merge_error_handler(Exception(message))

                raise TaskFailure(message)

    def review_request_report_checks(self):
        """
        @TODO: Объединить с check_required_checks в рамках FEI-17124.
        """
        with self.profiler.actions.check_required_checks('Check required checks'):
            project_task = self.get_project_task()

            if project_task:
                review_request = self.review_request

                project_task.meta.report_arcanum_checks(
                    review_request_id=review_request.get('id'),
                    diff_set_id=review_request.get('active_diff_set', {}).get('id'),
                    force=True,
                )

                if project_task.status == ctt.Status.SUCCESS:
                    logging.debug('Project task succeeded: no need to check Arcanum required checks')
                    return

            self.review_request_validate_required_checks()

    def review_request_validate_required_checks(self):
        """Проверяет успешность обязательных проверок в проектной задаче."""
        review_request = self.review_request
        review_request_id = review_request.get('id')
        checks = review_request.get('checks')
        required_checks = self.config.get('requiredChecks')

        logging.debug('Review request {rr_id} checks: {checks}, required checks: {required_checks}'.format(
            rr_id=review_request_id,
            checks=checks,
            required_checks=required_checks,
        ))

        def get_check_type(check):
            return check.get('type')

        def create_check(check_type, satisfied=False):
            return dict(type=check_type, satisfied=satisfied)

        def is_unsatisfied_required_check(check):
            is_required = get_check_type(check) in required_checks
            is_unsatisfied = not check.get('satisfied')

            return is_required and is_unsatisfied

        existing_check_types = map(get_check_type, checks)
        logging.debug('Existing check types: {}'.format(existing_check_types))

        missing_check_types = set(required_checks) - set(existing_check_types)
        logging.debug('Missing check types: {}'.format(missing_check_types))

        all_checks = checks + map(create_check, missing_check_types)
        logging.debug('All checks: {}'.format(all_checks))

        unsatisfied_checks = filter(is_unsatisfied_required_check, all_checks)
        logging.debug('Unsatisfied checks: {}'.format(unsatisfied_checks))

        if unsatisfied_checks:
            unsatisfied_check_types = map(get_check_type, unsatisfied_checks)

            self.Context.unsatisfied_checks = unsatisfied_check_types
            message = 'Failed project required checks: {}'.format(
                '; '.join(unsatisfied_check_types)
            )

            self.Context.merge_queue_task_exception = message
            self.mq_manager.merge_error_handler(Exception(message))

            raise TaskFailure(message)

    def __inc_priority(self, task):
        task_stat = json.loads(requests.get('{}/task/{}'.format(SANDBOX_API_URL, task.get('id'))).text)

        if ctt.Priority.make(task_stat['priority']) >= self.Parameters.priority:
            return

        sandbox_auth_token = self.vault.read('env.SANDBOX_AUTH_TOKEN')
        headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'OAuth {}'.format(sandbox_auth_token)
        }
        data = json.dumps({'id': task.get('id')})
        requests.put('{}/batch/tasks/increase_priority'.format(SANDBOX_API_URL), data=data, headers=headers)

        return self.__inc_priority(task)

    def clone_project(self):
        # Предотвращает повторное клонирование проекта, если текущее состояние актуально
        if self.is_working_copy_up_to_date():
            logging.debug('Skip repeat cloning repository')
            return

        self.mq_manager.clone_project()

        if self.use_arc:
            self.mq_manager.arc_clone_project()

    def is_working_copy_up_to_date(self):
        """
        :rtype: bool
        """
        def debug(msg):
            logging.debug('is_working_copy_up_to_date: {}'.format(msg))

        if not os.path.isdir(self.project_git_dir):
            debug('first project clone')
            return False

        current_tree_hash = self.project_git_exec('rev-parse', 'HEAD^{tree}')

        if self.Context.project_git_tree_hash != current_tree_hash:
            debug('tree hash changed (actual={}, expected={})'.format(current_tree_hash, self.Context.project_git_tree_hash))
            return False

        debug('working copy is up to date')
        return True

    def rebase(self):
        with self.memoize_stage_on_base_commit.rebase:
            try:
                with self.profiler.actions.clone_project('Cloning project'):
                    self.clone_project()

                with self.profiler.actions.git_rebase('Rebasing remote head branch'):
                    self.git_rebase()

                logging.debug('Waiting for GitHub to sync fork with base...')
                self.wait_sync_commit()
            except Exception as e:
                logging.exception('Exception occurred during rebase phase')
                self.Context.merge_queue_task_exception = e.message
                raise

    def git_rebase(self):
        with Debug('*'), GitRetryWrapper(), self.vault.ssh_key(self.Parameters.ssh_key_vault_name):
            try:
                self.scripts.run_js(
                    'script/merge-queue/rebase-and-force-push.js',
                    self.Context.project_git_head_ref,
                    self.Context.project_git_base_ref,
                    {
                        'owner': self.Parameters.project_github_owner,
                        'repo': self.Parameters.project_github_repo,
                        'login': self.Parameters.author_name,
                        'email': self.Parameters.author_email,
                    },
                    work_dir=self.project_dir,
                )
            except SandboxSubprocessError as e:
                self.mq_manager.merge_error_handler(e)

                raise

            commit_sha = self.project_git_exec('rev-parse', 'HEAD')
            tree_hash = self.project_git_exec('rev-parse', 'HEAD^{tree}')

            logging.info('Head branch is updated after rebase & force push: {old_commit} → {new_commit}'.format(
                old_commit=self.Context.project_git_head_commit,
                new_commit=commit_sha,
            ))

            self.Context.project_git_head_commit = commit_sha
            self.Context.project_git_tree_hash = tree_hash

            return commit_sha

    def wait_sync_commit(self):
        """
        Синхронизация изменений форка в базовый репозиторий происходит не мгновенно.
        Ждём, иначе не получится выставить и проверить статусы.
        """
        commit_path = 'repos/{owner}/{repo}/git/commits/{commit}'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
            commit=self.Context.project_git_head_commit,
        )

        max_retries = self.config.get('syncCommitRetries', 10)

        GitHubApi().request(commit_path, max_retries=max_retries)
        self.wait_head_commit_changed(max_retries)

    def wait_head_commit_changed(self, max_retries):
        """
        Даже после синхронизации коммита, ручка get_pr_info может возвращать старые данные.
        Ждем пока head_commit не изменится на хэш ребейзнутого коммита.
        """
        retry = 0

        while retry <= max_retries:
            logging.debug('Waiting for GitHub to change head commit, retry {}'.format(retry))
            head_sha = self.pull_request['head']['sha']

            if head_sha == self.Context.project_git_head_commit:
                break

            time.sleep(10)
            retry += 1

        if retry >= max_retries:
            logging.debug('Head commit has not changed to {} after {} retries'.format(self.Context.project_git_head_commit, retry))

    def on_finish(self, prev_status, status):
        self.profiler.register_wait_actions()
        self.send_profiler_report()
        self.send_status_signal(status)

        super(SandboxCiMergeQueue, self).on_finish(prev_status, status)

    # копипаста из BaseTask
    # TODO: добавить хелпер/менеджер для работы с noncritical_errors
    def send_profiler_report(self):
        if not self.Parameters.need_send_profile_stat:
            logging.debug('Skip sending profile data to statface')
            return

        # catch statface errors, see https://st.yandex-team.ru/FEI-5011
        import socket
        from statface_client import StatfaceClientError
        from requests.exceptions import HTTPError, ConnectionError
        try:
            full_repo_name = '{}/{}'.format(self.Parameters.project_github_owner, self.Parameters.project_github_repo)
            report_path = '{}/merge-queue/{}'.format(self.profiler.statface_report_prefix, full_repo_name)
            report_name = '{} actions profile'.format(full_repo_name)
            custom_data = dict(project_github_commit=self.Context.project_git_head_commit)

            self.profiler.send_report(report_path, report_name, custom_data)
        except (StatfaceClientError, HTTPError, ConnectionError, socket.error):
            error_traceback = traceback.format_exc()
            self.Context.noncritical_errors.append(error_traceback)
            logging.exception('Failed to send profile data')

    def on_success(self, prev_status):
        # Явно указываем статус, потому что задача находится в статусе `FINISHING`
        status = ctt.Status.SUCCESS

        self.send_webhooks(status)

        super(SandboxCiMergeQueue, self).on_success(prev_status)

    def on_failure(self, prev_status):
        # Явно указываем статус, потому что задача находится в статусе `FINISHING`
        status = ctt.Status.FAILURE

        self.Parameters.merge_fail_reason = json.dumps(self.fail_reason)
        self.Parameters.mq_error_code = self.mq_error_code
        self.send_webhooks(status)

        super(SandboxCiMergeQueue, self).on_failure(prev_status)

    def on_break(self, prev_status, status):
        self.Parameters.merge_fail_reason = json.dumps(self.fail_reason)
        self.Parameters.mq_error_code = self.mq_error_code
        self.send_webhooks(status)
        self.send_status_signal(status)

        super(SandboxCiMergeQueue, self).on_break(prev_status, status)

    @property
    def fail_reason(self):
        """
        Возвращает информацию информацию о причинах падения задачи.

        :rtype: dict
        """
        return {
            'merge_conflict': self.mq_manager.is_merge_conflict(self.Context.merge_queue_error_name),
            'unsatisfied_checks': self.clarify_failed_checks(self.Context.unsatisfied_checks[:]),
            'task_error_name': self.Context.merge_queue_error_name,
            'task_error_code': self.Context.merge_queue_error_code,
            'task_exception': self.Context.merge_queue_task_exception,
        }

    @property
    def mq_error_code(self):
        """
        Код ошибки при merge

        :rtype tp.Optional[str]
        """
        return self.Context.merge_queue_error_code

    def clarify_failed_checks(self, failed_checks):
        """
        Уточняет список упавших проверок.

        :param failed_checks: Список неуспешных проверок
        :rtype failed_checks: list
        :rtype: list
        """
        project_task = self.get_project_task()

        if not project_task:
            return []

        for child_task in iter(project_task.find()):
            child_failed_dependency_ids = self.get_failed_dependency_ids(child_task)

            if child_task.github_context in failed_checks and child_failed_dependency_ids:
                failed_checks.remove(child_task.github_context)
                failed_checks.extend(self.get_github_contexts_by_id(child_failed_dependency_ids))

        return list(set(failed_checks))

    @staticmethod
    def get_failed_dependency_ids(task):
        if task.Context.wait_tasks_statuses is ctm.NotExists:
            return []

        getId = lambda str: str.split(':')[-1].strip('>')

        return [getId(k) for k, v in task.Context.wait_tasks_statuses.items() if v != 'SUCCESS']

    def get_github_contexts_by_id(self, task_ids):
        contexts = []

        for task_id in task_ids:
            task_github_context = self.get_task_github_context(task_id)
            contexts.append(task_github_context)

        return contexts

    @staticmethod
    def get_task_github_context(task_id):
        task = next(iter(sdk2.Task.find(id=task_id, children=True).limit(1)))
        return task.github_context

    def send_status_signal(self, status):
        """
        :param status: статус задачи
        :type status: str
        """
        if getattr(self.Context, 'copy_of', False):
            return

        # Yasm tag values are limited: https://wiki.yandex-team.ru/golovan/userdocs/tagsandsignalnaming
        sanitize = lambda x: re.sub(r'[^a-zA-Z0-9\-_]', '_', x)
        repo = self.Context.project_git_url or self.format_git_url(
            self.Parameters.project_github_owner,
            self.Parameters.project_github_repo
        )
        build_context = 'merge'

        try:
            push_api.push_signals(
                signals={
                    'status_{}_mmmm'.format(status.lower()): 1,
                    'execution_time_min_hgram': [self.total_execution_time / 60, 1]
                },
                tags={
                    'itype': 'sandboxci',
                    'prj': 'sandboxci',
                    'build_context': build_context,
                    'service_name': sanitize(repo),
                    'task_type': str(self.type),
                }
            )

            self.zeroline_reporter.report_task({
                'build_id': (self.Parameters.success_build_uuid or self.id),
                'task_type': str(self.type),
                'task_id': self.id,
                'parent_id': (self.parent.id if self.parent else None),
                'project': repo,
                'build_context': build_context,
                'status': status,
                'duration': self.total_execution_time,
            })
        except Exception:
            logging.exception('Exception while sending status signal to yasm')

    def send_webhooks(self, status):
        """
        Отправляет веб-хуки задачи.

        :param status: текущий статус задачи
        :type status: str
        """
        logging.debug('Sending webhooks to service with status {}'.format(status))
        if self.Parameters.webhook_urls:
            self.webhook.send(status, self.Parameters.webhook_urls)

    def init_environ(self):
        task_environ = dict(
            SANDBOX_TASK_LOG_PATH=self.log_path()
        )

        envs = (
            env.from_vault(self),
            self.Parameters.environ,
            task_environ,
        )

        env.merge_and_export(envs)

        env.log(self)

    @sdk2.header()
    def header(self):
        report = {}

        profiler_actions = self.Context.profiler_actions
        if profiler_actions:
            report.update(**self.task_reports.actions_profile(profiler_actions))

        return report
