# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure

from sandbox.projects.MollyRun import MollyRun
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import BaseTask


class SandboxCiMollyRun(BaseTask):
    """Задача для запуска сканирования веб-сервисов на уязвимости (XSS, CSRF, RCE, SSRF)"""

    class Context(BaseTask.Context):
        molly_run_task_id = None
        report_links = []

    class Parameters(BaseTask.Parameters):
        platform = parameters.platform()
        ref = sdk2.parameters.String(
            'Build branch',
            required=True,
            description='Название собираемой ветки в GitHub',
        )

        with sdk2.parameters.Group('Molly') as molly_block:
            target_uri = sdk2.parameters.String(
                'Target uri',
                description='URI вашего сервиса, который нужно сканировать',
                required=True,
            )

            auth_profile = sdk2.parameters.String(
                'Authentication profile UID',
                description='ID аутентификационного профиля',
                required=True,
                default='260be436-7e94-472a-af43-cdde5b234679'
            )

            user_agent = sdk2.parameters.String(
                'User-Agent',
                description='Значение заголовка User-Agent, которое будет использоваться при сканировании'
            )

            qs_params = sdk2.parameters.String(
                'QueryString params',
                description='Дополнительные GET параметры, которые будут добавлены к каждому запросу в сервис',
            )

            palmsync_task = sdk2.parameters.Task(
                'Palmsync task',
                description='ID Sandbox таски, в которой были сгенерированы ресурсы с урлами фичей для каждой платформы',
                required=False,
                default=None,
            )

            request_samples_resource = sdk2.parameters.String(
                'Sandbox resource id containing HTTP request samples',
                description='ID ресурса с урлами фичей',
                required=False,
            )

            ignore_time_limit = sdk2.parameters.Bool(
                'Ignore scan time limit',
                description='Не завершать таску сканирования по таймауту',
                default=True,
                required=False,
            )

            abc_id = sdk2.parameters.String(
                'ABC id',
                description='Идентификатор сервиса в ABC',
                default='https://abc.yandex-team.ru/services/serp/',
            )

            users = sdk2.parameters.String(
                'Notify users',
                description='Cписок логинов пользователей (через запятую), которых нужно оповестить при проблемах',
            )

            target = sdk2.parameters.String(
                'Aggregation Target',
                description='Идентификатор, по которому будет происходить агрегация False Positive-ов',
            )

            st_queue = sdk2.parameters.String(
                'Tracker Queue',
                description='Очередь в ST для автоматического заведения тикетов об уязвимостях',
            )

            rps = sdk2.parameters.String(
                'RPS limit',
                description='Ограничение по RPS при сканировании',
            )

            severity = sdk2.parameters.Integer(
                'Minimum severity level to mark target as vulnerable',
                description='Минимальный уровень критичности найденных уязвимостей, при которых status будет меняться',
                default=50
            )

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

    @classmethod
    def format_github_context(self, description):
        return u'[Sandbox CI] MOLLY: {}'.format(description)

    @property
    def github_context(self):
        return self.format_github_context(self.Parameters.platform)

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    def get_samples_resource_id(self):
        """
        request_sample_resource может определяться двумя способами:
        - напрямую указан в параметрах таски;
        - находится в выходных параметрах таски palmsync.

        :rtype: int or None
        """
        request_sample_resource = self.Parameters.request_samples_resource

        if not self.Parameters.palmsync_task:
            return request_sample_resource

        task = self.Parameters.palmsync_task
        return task.Parameters.resource_map.get(self.Parameters.platform, None)

    def run_molly_task(self):
        task_parameters = dict(
            target_uri=self.Parameters.target_uri,
            auth_profile=self.Parameters.auth_profile,
            qs_params=self.Parameters.qs_params,
            abc_id=self.Parameters.abc_id,
            users=self.Parameters.users,
            target=self.Parameters.target,
            st_queue=self.Parameters.st_queue,
            rps=self.Parameters.rps,
            severity=self.Parameters.severity,
            ignore_time_limit=self.Parameters.ignore_time_limit,
            request_samples_resource=self.get_samples_resource_id(),
        )

        logging.debug('Starting MOLLY_RUN task with parameters: {}'.format(task_parameters))

        return MollyRun(self, **task_parameters).enqueue()

    def run_and_wait_molly_task(self):
        """ Запускаем и ожидаем выполнения переданной задачи. """
        molly_run_task = self.run_molly_task()
        molly_run_task_id = molly_run_task.id

        logging.debug('Waiting task #{}'.format(molly_run_task_id))

        self.Context.molly_run_task_id = molly_run_task_id

        raise sdk2.WaitTask(molly_run_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def get_molly_run_task(self):
        return sdk2.Task[self.Context.molly_run_task_id]

    def execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            self.run_and_wait_molly_task()

        molly_run_task = self.get_molly_run_task()
        molly_run_task_id = self.Context.molly_run_task_id

        if not molly_run_task:
            raise TaskFailure('Something went wrong: task with id #{} not found'.format(molly_run_task_id))
        elif molly_run_task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Child task with id #{id} was finished with not correct status {status}'.format(
                id=molly_run_task_id,
                status=molly_run_task.status,
            ))

        report_url = molly_run_task.Parameters.report
        # Были ли обнаружены уязвимости с критичностью больше, чем заданная в конфигурации запуска
        status = molly_run_task.Parameters.status

        report_status = ctt.Status.FAILURE if status else ctt.Status.SUCCESS

        self.Context.report_links.append({
            'Status': self.task_reports.status_badge(report_status),
            'Report': self.task_reports.description(report_url, 'molly_report', None, None),
        })

        if status:
            raise TaskFailure('Vulnerabilities were found, for more details see molly run report.')

    @sdk2.header()
    def header(self):
        report = super(SandboxCiMollyRun, self).header()

        if self.Context.report_links:
            report_object = {'<h3 id="checks-reports">Checks reports</h3>': self.Context.report_links}
            report.update(**report_object)

        return report

    def before_end(self, status):
        """
        Вызывается перед завершением таски
        :param status: статус задачи (SUCCESS, FAILED)
        :type status: sandbox.common.types.task
        """
        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)

    def on_before_end(self, status):
        """ Должны зарепорить в релизный тикет статус текущей задачи, если это релизная сабтаска. """
        super(SandboxCiMollyRun, self).on_before_end(status)
        self.before_end(status)
