# -*- coding: utf-8 -*-

import logging
import re

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny.client import NannyApiException
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.utils.release import remove_release_branch_prefix

NANNY_API_URL = 'http://nanny.yandex-team.ru/'


class SandboxCiNannyEnsureDeploy(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """Проверка успешности деплоя ресурса в Nanny"""

    name = 'SANDBOX_CI_NANNY_ENSURE_DEPLOY'
    github_context = '[Sandbox CI] Проверка деплоя в Nanny'

    class Parameters(sdk2.Parameters):
        _container = parameters.environment_container()

        release_id = sdk2.parameters.String(
            'Nanny release id',
            required=True,
            description='ID релиза в Nanny, который нужно проверять',
        )
        semaphore_name = sdk2.parameters.String(
            'Name of semaphore',
            default=None,
            description='Название семафора, которого нужно дождаться, прежде чем начинать поллинг',
        )
        poll_interval = sdk2.parameters.Integer(
            'Poll interval',
            default=300,
            description='Интервал между попытками проверить состояние деплоя, в секундах',
        )
        attempts = sdk2.parameters.Integer(
            'Amount of attempts',
            default=12,
            description='Количество попыток. По истечению попыток таск перейдет в статус Failure',
        )
        failed_statuses = sdk2.parameters.List(
            "Failed statuses",
            default=["DEPLOY_FAILED", "ROLLED_BACK"],
            description="Статусы тикетов, при которых релиз будет считаться неудачным",
        )

        with sdk2.parameters.Group('Трекер') as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        dependencies = parameters.DependenceParameters

    class Context(sdk2.Context):
        failed_tasks = []
        passed_attempts = 0

    def on_enqueue(self):
        self.task_dependencies.wait_tasks(fail_on_deps_fail=True)
        self.task_dependencies.wait_output_parameters(timeout=(self.Parameters.kill_timeout * 2))

        if self.Parameters.semaphore_name:
            with self.memoize_stage.semaphore:
                self.Requirements.semaphores = ctt.Semaphores(
                    acquires=[
                        ctt.Semaphores.Acquire(name=self.Parameters.semaphore_name, weight=1, capacity=1)
                    ],
                    release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
                )

    def on_execute(self):
        try:
            release = self.get_release()
            tickets = self.get_tickets()
        except NannyApiException as e:
            logging.debug('Error in fetching release info: {}'.format(e.message))
            self.wait()

        if self.is_release_deployed(release, tickets):
            self.on_deploy_success()
        elif self.is_release_failed(tickets):
            self.on_deploy_fail(tickets)
        else:
            self.wait()

    def get_release(self):
        logging.debug('fetching release')
        return self.nanny_client.get_release(self.Parameters.release_id)

    def get_tickets(self):
        logging.debug('fetching tickets for release')
        req = {
            'query': {
                'releaseId': self.Parameters.release_id
            }
        }
        return self.nanny_client.filter_tickets(req)

    @singleton_property
    def nanny_client(self):
        oauth_token = self.vault.read('env.NANNY_TOKEN')
        return nanny.NannyClient(api_url=NANNY_API_URL, oauth_token=oauth_token)

    def is_release_deployed(self, release, tickets):
        if self.is_release_failed(tickets):
            return False

        return self.is_release_closed(release)

    def is_release_failed(self, tickets):
        failed_statuses = filter(self.is_failed, tickets)
        return len(failed_statuses) > 0

    def is_failed(self, ticket):
        status = ticket['status']['status']
        failed_statuses = self.Parameters.failed_statuses

        logging.debug('Checking is ticket "{id}" with status "{status}" are in {statuses}'.format(
            id=ticket['id'],
            status=status,
            statuses=failed_statuses,
        ))

        return status in failed_statuses

    def is_release_closed(self, release):
        return release['status']['status'] == 'CLOSED'

    def on_deploy_success(self):
        logging.debug('release {} successfully deployed'.format(self.Parameters.release_id))

        comment = '✅ Деплой (({release_url} {release_id})) завершился успешно.'.format(
            release_url=self.release_url,
            release_id=self.Parameters.release_id,
        )

        self.log_templates_deployed_release_step()
        self.add_comment(comment)

    def on_deploy_fail(self, tickets):
        failed_tickets = filter(self.is_failed, tickets)
        failed_tickets_wiki_urls = map(self.get_ticket_wiki_url, failed_tickets)
        statuses = map(self.format_status_with_wiki, self.Parameters.failed_statuses)

        urls = '\n\t'.join(failed_tickets_wiki_urls)
        reason = 'некоторые тикеты завершились с одним из статусов: {statuses}.'.format(
            statuses=', '.join(statuses),
        )
        reason += '\n\t{urls}'.format(urls=urls)

        self.raise_error(reason)

    def get_ticket_wiki_url(self, ticket):
        url = self.get_ticket_url(ticket)

        return '(({url} {id})) — {status}'.format(
            url=url,
            id=ticket['id'],
            status=self.format_status_with_wiki(ticket['status']['status']),
        )

    def get_ticket_url(self, ticket):
        return '{nanny_url}/ui/#/t/{id}'.format(
            nanny_url=NANNY_API_URL,
            id=ticket['id'],
        )

    def format_status_with_wiki(self, status):
        return '%%{status}%%'.format(status=status)

    def raise_error(self, reason):
        logging.debug('deploy of release {} failed'.format(self.Parameters.release_id))

        comment = '❗Деплой (({release_url} {release_id})) не удался: {reason}'.format(
            release_url=self.release_url,
            release_id=self.Parameters.release_id,
            reason=reason,
        )
        self.add_comment(comment)

        raise TaskFailure(reason)

    @singleton_property
    def release_url(self):
        return '{nanny_url}/ui/#/r/{release_id}'.format(
            nanny_url=NANNY_API_URL,
            release_id=self.Parameters.release_id,
        )

    def add_comment(self, text):
        issue_key = self.Parameters.send_comment_to_issue
        comment_data = dict(issue_key=issue_key, text=text)

        if not issue_key or not text:
            return logging.debug('skip sending comment "{text}" to issue {issue_key}'.format(**comment_data))

        logging.debug('send comment to issue {issue_key}: {text}'.format(**comment_data))

        try:
            self.release.startrek_client.add_comment(**comment_data)
        except Exception as e:
            logging.exception('failed to add comment: {}'.format(e.message))

    def log_templates_deployed_release_step(self):
        try:
            issue_key = self.Parameters.send_comment_to_issue

            if not issue_key:
                return

            self.metrics_logger.log_release_steps([dict(
                project='{}/{}'.format(self.Context.project_github_owner, self.Context.project_github_repo),
                version=remove_release_branch_prefix(branch=self.parent.ref),
                issue=issue_key,
                step='templates_deployed',
            )])
        except Exception:
            logging.exception('Exception while logging release step')

    def wait(self):
        if self.Context.passed_attempts > self.Parameters.attempts:
            return self.raise_error('время ожидания истекло.')

        logging.debug('deploy not finished yet, waiting')

        self.Context.passed_attempts += 1

        raise sdk2.WaitTime(self.Parameters.poll_interval)
