# -*- coding: utf-8 -*-

from sandbox import sdk2
import os

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants

from sandbox.common.utils import classproperty

RAMDRIVE_SIZE = 30 * 1024


class SandboxCiPalmsyncBaseTask(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    lifecycle_steps = {
        'npm_install': 'npm ci',
        'custom_command': '{custom_command}',
    }

    class Requirements(BaseTask.Requirements):
        disk_space = 35 * 1024
        cores = 1
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

        # https://st.yandex-team.ru/SANDBOX-6042 – проблема c mount sqsh-образов на мультислотах
        # class Caches(sdk2.Requirements.Caches):
        #     pass

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_tree_hash = parameters.project_tree_hash()
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()
        project_name = sdk2.parameters.String('Project name', required=True)
        testpalm_project_name = sdk2.parameters.String('Testpalm project name')
        custom_opts = sdk2.parameters.String('Опции, которые будут добавлены при запуске Palmsync')
        custom_command = sdk2.parameters.String('Кастомная команда перед запуском Palmsync', multiline=True)
        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    @property
    def lifecycle(self):
        return super(SandboxCiPalmsyncBaseTask, self).lifecycle.update_vars(
            project=self.testpalm_project_name,
            run_custom_opts=self.Parameters.custom_opts,
            custom_command=self.Parameters.custom_command,
        )

    @classproperty
    def task_label(self):
        return 'palmsync'

    @property
    def project_name(self):
        return str(self.Parameters.project_name)

    @property
    def testpalm_project_name(self):
        if self.Parameters.testpalm_project_name:
            return str(self.Parameters.testpalm_project_name)

        return self.project_name

    @property
    def int_resources_path(self):
        return self.working_path('.yandex-int')

    @property
    def int_logs_path(self):
        return self.int_resources_path / 'logs'

    def on_save(self):
        super(BaseTask, self).on_save()

        setattr(self.Context, '__do_not_dump_ramdrive', True)  # use setattr to avoid mangling

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir, unsquash=True)
        self.prepare()
        self._install_dependencies()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.execute_custom_command()
            self.test()

    def execute_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            with Node(self.Parameters.node_js_version), self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                self.prepare()
                self.execute_custom_command()
                self.test()

    def execute_custom_command(self):
        if self.Parameters.custom_command:
            with self.profile_action(actions_constants['CUSTOM_COMMAND'], 'Executing custom command'):
                self.lifecycle('custom_command')

    def prepare(self):
        self.set_environments()

    def set_environments(self):
        os.environ['YANDEX_INT_RESOURCES_PATH'] = str(self.int_resources_path)
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)
        os.environ['hermione_muted_tests_stability_index_output_enabled'] = 'false'

    def make_reports(self):
        self.artifacts.create_report(
            resource_path=self.int_logs_path,
            type='yandex-int-logs',
            task_id=self.id,
            status=ctt.Status.SUCCESS,
            ttl=30,
        )
