# -*- coding: utf-8 -*-

from sandbox import sdk2
import os
import json
import logging

from sandbox.projects.sandbox_ci.utils.context import Debug
from sandbox.projects.sandbox_ci.sandbox_ci_palmsync.SandboxCiPalmsyncBaseTask import SandboxCiPalmsyncBaseTask
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants

from sandbox.sandboxsdk import process
from sandbox.projects.common import solomon
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.utils import classproperty
from copy import deepcopy

TESTPALM_CLI_PACKAGE = '@yandex-int/si.ci.testpalm-cli@2.4'

SOLOMON_PUSH_COMMON_PARAMETERS = {
    'project': 'fei',
    'cluster': 'sandbox',
    'service': 'palmsync',
}


class SandboxCiPalmsyncSynchronize(SandboxCiPalmsyncBaseTask):
    lifecycle_steps = deepcopy(SandboxCiPalmsyncBaseTask.lifecycle_steps)
    lifecycle_steps.update({
        'sync': 'node --max-old-space-size=4096 $(npm bin)/palmsync synchronize --project={project} {run_custom_opts}',
    })

    fail_on_deps_fail = False

    class Parameters(SandboxCiPalmsyncBaseTask.Parameters):
        ignore_semaphores_requirement = sdk2.parameters.Bool('Игнорировать требования по семафорам (только для тестирования)', default=False)

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Resources') as output_resource_map:
                resource_map = sdk2.parameters.Dict('Список ресурсов c урлами фичей для всех платформ, в формате { platform: id }')

    @property
    def skip_lfs_checkout(self):
        return self.use_arc or not self.use_overlayfs

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Синхронизация тестовых сценариев'

    @classproperty
    def report_description(self):
        return 'testpalm urls'

    def on_save(self):
        super(SandboxCiPalmsyncSynchronize, self).on_save()

        if self.use_git_in_overlayfs_mode:
            self.set_ramdrive_size(40 * 1024)

        self.__set_semaphores()

    def __set_semaphores(self):
        if self.Parameters.ignore_semaphores_requirement:
            return

        semaphore_name = self.type.name + '_' + self.project_name.upper()

        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name=semaphore_name, capacity=1)
            ],
        )

    def test(self):
        try:
            with self.profile_action(actions_constants['SYNC'], 'Running palmsync synchronize'), Debug('testpalm-api', depth=Debug.INFINITY):
                if self.lifecycle('sync', wait=False).wait():
                    raise TaskFailure('palmsync synchronize error, check sync.out.txt')

                self.check_and_report_errors(self.testpalm_project_name)
        finally:
            self.make_reports()

    def make_reports(self):
        testpalm_urls_dir = 'testpalm-urls'
        resource_map_dict = {}

        for platform in ['desktop', 'touch-pad', 'touch-phone']:
            resource = self.artifacts.create_report(
                resource_path='{}/{}.json'.format(testpalm_urls_dir, platform),
                type='{}-{}'.format(testpalm_urls_dir, platform),
                status=ctt.Status.SUCCESS,
                add_to_context=True
            )

            if resource:
                resource_map_dict[platform] = resource.id

        logging.debug('{} set to output parameter with name "resource_map"'.format(resource_map_dict))

        if resource_map_dict:
            self.Parameters.resource_map = resource_map_dict

        super(SandboxCiPalmsyncSynchronize, self).make_reports()

    def set_environments(self):
        super(SandboxCiPalmsyncSynchronize, self).set_environments()

        os.environ.update({
            'NODE_EXTRA_CA_CERTS': '/etc/ssl/certs/YandexInternalRootCA.pem',
            'NPM_CONFIG_REGISTRY': 'http://npm.yandex-team.ru',
            'NPM_CONFIG_USER_AGENT': 'npm/6.2.0 (verdaccio yandex canary)',
            'TESTPALM_OAUTH_TOKEN': self.vault.read('env.TESTPALM_OAUTH_TOKEN')
        })

    def check_and_report_errors(self, testpalm_project):
        errors_count = 0

        try:
            testcases = self.fetch_test_cases(testpalm_project)
            errors_count = self.calc_testcase_errors(testcases)
        finally:
            logging.debug('found errors: {}'.format(errors_count))

            if errors_count > 0:
                self._send_solomon_sensors([{
                    'labels': {'sensor': 'palmsync_synchronize_{}'.format(testpalm_project)},
                    'kind': 'IGAUGE',
                    'value': errors_count
                }])

    def fetch_test_cases(self, project):
        cmd = [
            'npx',
            TESTPALM_CLI_PACKAGE,
            'fetch-test-cases',
            project,
            '--apply-definitions',
            '--include=attributes',
        ]

        p = process.run_process(
            cmd,
            log_prefix='testpalm_testcases',
            work_dir=str(self.project_dir),
            outputs_to_one_file=False,
        )

        with open(p.stdout_path, 'r') as out_file:
            out = out_file.read()

        return json.loads(out.strip())

    def calc_testcase_errors(self, testcases):
        errors_count = 0
        for testcase in testcases:
            errors_count += len(testcase.get('attributes').get('palmsync_synchronization_errors', []))

        return errors_count

    def _send_solomon_sensors(self, sensors):
        try:
            solomon.push_to_solomon_v2(
                token=self.vault.read('env.SOLOMON_OAUTH_TOKEN'),
                params=SOLOMON_PUSH_COMMON_PARAMETERS,
                sensors=sensors
            )
            logging.debug('Sensors pushed: {}'.format(sensors))
        except Exception as error:
            logging.exception('Cannot send sensors ({sensors}): {error}'.format(
                sensors=sensors,
                error=error
            ))
