# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.utils import classproperty

from sandbox.projects import resource_types
from sandbox.projects.sandbox_ci.task import BaseTask
from sandbox.projects.sandbox_ci.managers.s3mds import S3Manager
from sandbox.projects.sandbox_ci.utils.static import compress_static


class SandboxCiStaticS3(BaseTask):
    """Таска для загрузки статики через S3 API в MDS"""

    name = 'SANDBOX_CI_STATIC_S3'  # Для корректного имени задачи (по умолчанию получается SANDBOX_CI_STATIC_S_3)

    class Requirements(BaseTask.Requirements):
        cores = 1

        environments = BaseTask.Requirements.environments.default + (
            PipEnvironment('boto3', '1.4.4', use_wheel=True),
        )

        class Caches(BaseTask.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        static_resource = sdk2.parameters.Resource(
            'Static resource',
            resource_type=resource_types.REPORT_STATIC_PACKAGE,
            required=True,
        )

        project_name = sdk2.parameters.String(
            'Project name',
            required=True,
        )

        bucket_name = sdk2.parameters.String(
            'Bucket name (default: serp-static-testing)'
        )

        freeze_path = sdk2.parameters.String(
            'freeze_path',
            description='Path to dir with frozen files (you can use {version} placeholder)',
            required=True
        )

        key_prefix = sdk2.parameters.String(
            'Key prefix (default: equal freeze_path value)'
        )

        compress_needed = sdk2.parameters.Bool(
            'Create .gz and .br versions before upload (default: false)',
            default=False,
        )

        recursive_update = sdk2.parameters.Bool(
            'Rewrite all files inside directory if presents in S3 (default: true)',
            default=True,
        )

        vault_key_id = sdk2.parameters.String(
            'Name of secret which is Key id for S3 (default: env.SEARCH_INTERFACES_S3_ACCESS_KEY_ID)',
            default='env.SEARCH_INTERFACES_S3_ACCESS_KEY_ID',
        )

        vault_secret_key = sdk2.parameters.String(
            'Name of secret which is Access key for S3 (default: env.SEARCH_INTERFACES_S3_SECRET_ACCESS_KEY)',
            default='env.SEARCH_INTERFACES_S3_SECRET_ACCESS_KEY',
        )

        vault_group = sdk2.parameters.String(
            'Name of group for Key id and Access key secrets (default: SANDBOX_CI_SEARCH_INTERFACES)',
            default='SANDBOX_CI_SEARCH_INTERFACES'
        )

    @property
    def project_name(self):
        return str(self.Parameters.project_name)

    @property
    def freeze_path(self):
        return str(self.Parameters.freeze_path)

    @property
    def key_prefix(self):
        if self.Parameters.key_prefix:
            return str(self.Parameters.key_prefix)

        return self.freeze_path

    @property
    def s3_bucket_name(self):
        if self.Parameters.bucket_name:
            return str(self.Parameters.bucket_name)

        return 'serp-static-testing'

    @property
    def resource(self):
        return self.Parameters.static_resource

    @property
    def compress_needed(self):
        return self.Parameters.compress_needed

    @property
    def recursive_update(self):
        return self.Parameters.recursive_update

    @classproperty
    def task_label(self):
        return 'static-s3'

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Деплой статики (S3 API в MDS)'

    def get_freeze_path(self):
        freeze_path = self.project_path(self.freeze_path)

        return str(freeze_path)

    def upload_to_mds(self):
        key_id = self.vault.read(self.Parameters.vault_key_id, self.Parameters.vault_group)
        access_key = self.vault.read(self.Parameters.vault_secret_key, self.Parameters.vault_group)
        client = S3Manager(key_id, access_key, self.s3_bucket_name)
        freeze_path = self.get_freeze_path()

        client.upload_dir(freeze_path, self.key_prefix, self.recursive_update)

    def prepare(self):
        with self.profiler.actions.unpack_artifacts('Unpacking static artifact'):
            self.artifacts.unpack_build_artifacts([self.resource], self.project_dir, 1)

    def compress(self):
        logging.info('Creating .gz and .br versions')
        compress_static(self.get_freeze_path())

    def execute(self):
        self.prepare()
        if self.compress_needed:
            self.compress()
        self.upload_to_mds()
