# -*- coding: utf-8 -*-

import logging
import json

from sandbox import sdk2

from sandbox.common.types import task as ctt

from sandbox.projects.common.nanny import nanny

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.decorators.skip_subtask import skip_subtask
from sandbox.projects.sandbox_ci.task.skip_validator import SandboxCiSkipValidator
from sandbox.projects.sandbox_ci.sandbox_ci_yappy_deploy import SandboxCiYappyDeploy
from sandbox.projects.sandbox_ci.sandbox_ci_web4_yappy_cleanup import SandboxCiWeb4YappyCleanup
from sandbox.projects.sandbox_ci.sandbox_ci_ab_experiments import SandboxCiAbExperiments
from sandbox.projects.sandbox_ci.sandbox_ci_molly_run import SandboxCiMollyRun
from sandbox.projects.sandbox_ci.sandbox_ci_nanny_ensure_deploy import SandboxCiNannyEnsureDeploy
from sandbox.projects.sandbox_ci.sandbox_ci_web4_build import SandboxCiWeb4Build
from sandbox.projects.sandbox_ci.sandbox_ci_web4_docs import SandboxCiWeb4Docs
from sandbox.projects.sandbox_ci.sandbox_ci_web4_hermione_e2e import SandboxCiWeb4HermioneE2E
from sandbox.projects.sandbox_ci.sandbox_ci_hermione import SandboxCiHermione
from sandbox.projects.sandbox_ci.sandbox_ci_testpalm_suite_runner import SandboxCiTestpalmSuiteRunner
from sandbox.projects.sandbox_ci.sandbox_ci_testpalm_suite_runner.utils import should_start_testpalm_suite_runner_task
from sandbox.projects.sandbox_ci.sandbox_ci_palmsync.synchronize import SandboxCiPalmsyncSynchronize
from sandbox.projects.sandbox_ci.sandbox_ci_palmsync.validate import SandboxCiPalmsyncValidate
from sandbox.projects.sandbox_ci.sandbox_ci_web4_manual_test_run import SandboxCiWeb4ManualTestRun
from sandbox.projects.sandbox_ci.sandbox_ci_web4_unit import SandboxCiWeb4Unit
from sandbox.projects.sandbox_ci.sandbox_ci_web4_linters import SandboxCiWeb4Linters
from sandbox.projects.sandbox_ci.sandbox_ci_blockstat_test import SandboxCiBlockstatTest
from sandbox.projects.sandbox_ci.sandbox_ci_expflags import SandboxCiExpflags
from sandbox.projects.sandbox_ci.sandbox_ci_testids_for_flags import SandboxCiTestidsForFlags
from sandbox.projects.sandbox_ci.sandbox_ci_test_cost_reporter import SandboxCiTestCostReporter
from sandbox.projects.sandbox_ci.sandbox_ci_test_changes_collector import SandboxCiTestChangesCollector
from sandbox.projects.sandbox_ci.pulse.subtasks import create_pulse_subtasks
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_parameters
from sandbox.projects.sandbox_ci.task import ManualRunMetaTaskMixin
from sandbox.projects.sandbox_ci.infratest.assert_metrics_stats import InfratestAssertMetricsStats
from sandbox.projects.sandbox_ci.constants import MailingLists


class SandboxCiWeb4(nanny.ReleaseToNannyTask2, ManualRunMetaTaskMixin, SandboxCiWeb4Build):
    """Автосборка Серпа (serp/web4)"""

    name = 'SANDBOX_CI_WEB4'  # Для обратной совместимости с sdk1 задачей

    class Parameters(SandboxCiWeb4Build.Parameters):
        with SandboxCiWeb4Build.Parameters.project_build_block() as project_build_block:
            selective_checks = parameters.selective_checks()

        with sdk2.parameters.Group('Hermione') as hermione_block:
            hermione_platforms = parameters.hermione_platforms

        with sdk2.parameters.Group('Pulse') as pulse_block:
            pulse_should_report_to_stat = parameters.pulse_should_report_to_stat()
            pulse_should_run_ticount = parameters.pulse_should_run_ticount()
            fail_on_limits_exceed = pulse_parameters.fail_on_limits_exceed()
            send_email_on_limits_exceed = pulse_parameters.send_email_on_limits_exceed()
            pulse_platforms = parameters.pulse_platforms

        with sdk2.parameters.Group('Molly') as molly_block:
            molly_should_run = sdk2.parameters.Bool('Run tests', default=False)
            molly_platforms = parameters.molly_platforms

        manual_test_runs = parameters.ManualTestRunsParameters

        yappy_beta = parameters.DeployYappyParameters

        assessors = parameters.AssessorsParameters

    manual_run_task_type = SandboxCiWeb4ManualTestRun

    project_name = 'web4'

    def on_enqueue(self):
        super(SandboxCiWeb4, self).on_enqueue()
        if self.is_release:
            self.create_nanny_testing_release_semaphore()
        elif self.is_dev:
            self.acquire_dev_build_semaphore()
        elif self.Parameters.project_build_context == 'dev_production':
            self.acquire_dev_build_semaphore('dev_production')

    @property
    def github_context(self):
        return u'[Sandbox CI] Автосборка'

    @property
    def use_overlayfs_in_palmsync_sync(self):
        return self.use_overlayfs and self.project_conf.get('use_overlayfs_in_palmsync_sync', False)

    def is_mq_cache_semaphore_required(self):
        return super(SandboxCiWeb4Build, self).is_mq_cache_semaphore_required()

    def get_build_artifacts_for_manual_run(self, artifact_types):
        return self.get_build_artifacts_for_testpalm_task(artifact_types)

    def get_beta_urls_param(self):
        # В dev запускаются e2e-тесты на покоммитные dev-беты
        # В dev_production тесты не запускаются, обновляется единая dev-бета
        # В релизной сборке тестирование производится на приёмке
        beta_suffix = self.Parameters.beta_suffix
        overwrite_quota = self.Parameters.overwrite_quota
        beta_domain = 'https://renderer-{}-{}.hamster.yandex.ru/'.format(
            self.project_name,
            beta_suffix
        )
        e2e_beta_domain = beta_domain

        if self.is_dev:
            # В payload нет сокращенного sha, не задать в sandbox-ci.yml
            beta_suffix += '-{}'.format(self.Parameters.project_github_commit[0:7])
            overwrite_quota = 'renderer-serp-vip'
            e2e_beta_domain = 'https://{}-{}.hamster.yandex.ru/'.format(
                overwrite_quota,
                beta_suffix
            )

        if self.is_release:
            if '-exp/' in self.Parameters.project_git_base_ref:
                beta_domain = 'https://webtemplates-exp.hamster.yandex.ru'
            elif '-exp2/' in self.Parameters.project_git_base_ref:
                beta_domain = 'https://webtemplates-exp2.hamster.yandex.ru'
            elif '-exp3/' in self.Parameters.project_git_base_ref:
                beta_domain = 'https://webtemplates-exp3.hamster.yandex.ru'
            elif '-exp4/' in self.Parameters.project_git_base_ref:
                beta_domain = 'https://webtemplates-exp4.hamster.yandex.ru'
            else:
                beta_domain = 'https://webtemplates.hamster.yandex.ru'
            e2e_beta_domain = beta_domain
        res = dict()
        res['beta_suffix'] = beta_suffix
        res['overwrite_quota'] = overwrite_quota
        res['beta_domain'] = beta_domain
        res['e2e_beta_domain'] = e2e_beta_domain
        return res

    def declare_subtasks(self):
        beta_urls_param = self.get_beta_urls_param()
        overwrite_quota = beta_urls_param['overwrite_quota']
        beta_suffix = beta_urls_param['beta_suffix']
        beta_domain = beta_urls_param['beta_domain']
        e2e_beta_domain = beta_urls_param['e2e_beta_domain']

        nanny_ensure_testing_deploy_task = self.create_nanny_ensure_testing_deploy_subtask(wait_tasks=[self])

        deploy_wait_parameters = {self.id: 'is_static_uploaded,is_artifacts_ready'}
        deploy_yappy_subtask = self.create_deploy_yappy_subtask(
            wait_output_parameters=deploy_wait_parameters,
            suffix=beta_suffix,
            overwrite_quota=overwrite_quota,
        )

        create_ab_experiment = self.create_ab_experiment_testid(
            deploy_subtask=deploy_yappy_subtask if not self.is_release else nanny_ensure_testing_deploy_task,
            yappyBeta=True if not self.is_release else False,
            experiment_beta_slot='' if not self.is_release else "renderer-web-templates.hamster.yandex.ru:9080",
        )

        unit_tests_subtask = self.create_unit_tests_subtask()

        subtasks = [
            deploy_yappy_subtask,
            create_ab_experiment,
            self.create_expflags_subtask(),
            self.create_linters_subtask(),
            self.create_palmsync_validate_subtask(),
            unit_tests_subtask,
        ]

        issues_info = self.startrek.get_issues_info()
        issue_keys = issues_info['keys']
        if self.config.is_enabled('tests', 'skip-validator'):
            subtasks.append(self.create_skip_validator_subtask(issue_keys=issue_keys))

        test_tasks = []
        wait_tasks = []

        hermione_task = self.create_hermione_task(
            task_type=SandboxCiHermione,
            tool='hermione',
            platforms=self.Parameters.hermione_platforms,
            wait_output_parameters=self.static_output_parameters_to_wait,
            issue_keys=issue_keys,
            beta_domain=beta_domain,
        )
        if hermione_task:
            test_tasks.append(hermione_task)
            wait_tasks.append(hermione_task)

        hermione_e2e_task = self.create_hermione_task(
            task_type=SandboxCiWeb4HermioneE2E,
            tool='hermione-e2e',
            platforms=self.Parameters.hermione_platforms,
            wait_tasks=[deploy_yappy_subtask] if not self.is_release else [nanny_ensure_testing_deploy_task],
            issue_keys=issue_keys,
            beta_domain=e2e_beta_domain,
            hermione_base_url=e2e_beta_domain,
            hermione_config_path='.hermione-e2e.conf.js',
        )

        if hermione_e2e_task:
            test_tasks.append(hermione_e2e_task)
            if self.is_dev and not self.is_release:
                # В dev запускаются e2e-тесты на покоммитные dev-беты, их нужно погасить
                subtasks.append(self.create_yappy_cleanup_subtask(
                    wait_tasks=[deploy_yappy_subtask, hermione_e2e_task],
                    suffix=beta_suffix,
                    overwrite_quota=overwrite_quota,
                ))

        subtasks += test_tasks

        if self.config.is_enabled('tests', 'pulse'):
            subtasks += self.create_pulse_subtasks(
                self,
                platforms=self.Parameters.pulse_platforms,
                shoot_dynamic=True,
                shoot_static=True,
                use_assets_json=True,
                use_assets_json_v2=True,
                waitable=(not self.Parameters.is_release),
                pr_number=self.pr_number,
            )

        # ожидание всех тестовых задач нужно для выгрузки урлов в testpalm, которые во время прогона тестов собираются в ресурс, а после используются в задаче palmsync
        palmsync_subtask = self.create_palmsync_synchronize_subtask()
        if palmsync_subtask:
            subtasks.append(palmsync_subtask)
            subtasks.append(self.create_testids_for_flags_subtask(wait_tasks=[palmsync_subtask]))

        # нужно для релизов, @see FEI-8047
        if self.Parameters.molly_should_run:
            for platform in self.Parameters.molly_platforms:
                params = self.config.get_deep_value(['tests', 'molly', 'platforms', platform], {})

                # params['target_uri'] может быть предустановлен в Genisys и имеет формат {beta_domain}/some/path/?foo=bar
                target_uri = params['target_uri'].format(beta_domain=beta_domain) if params['target_uri'] else beta_domain
                params['target_uri'] = target_uri

                if palmsync_subtask:
                    params['wait_output_parameters'] = {palmsync_subtask.id: 'resource_map'}
                    params['palmsync_task'] = palmsync_subtask.id

                molly_run_task = self.create_molly_run_subtask(
                    platform,
                    wait_tasks=[deploy_yappy_subtask] if not self.is_release else [nanny_ensure_testing_deploy_task],
                    **params
                )

                if molly_run_task:
                    subtasks.append(molly_run_task)

        if create_ab_experiment:
            wait_tasks.append(create_ab_experiment)

        manual_run_task = self.create_manual_run_subtask(
            issues_info=issues_info,
            palmsync_base_url=beta_domain,
            changed_files=self.changed_files_filename,
            skip_make_drafts=True,
            skip_make_runs=True,
            notifications=self.build_release_critical_notifications(),
            project_hash=self.Parameters.project_hash,
            use_overlayfs=self.use_overlayfs_in_palmsync_sync,
        )
        if manual_run_task:
            subtasks.append(manual_run_task)
            subtasks.append(self.create_testids_for_flags_subtask(wait_tasks=[manual_run_task]))

        subtasks.append(nanny_ensure_testing_deploy_task)

        testpalm_suite_runner = self.create_testpalm_suite_runner_subtask(
            wait_tasks=[manual_run_task, nanny_ensure_testing_deploy_task],
            wait_output_parameters=self.static_output_parameters_to_wait,
        )

        subtasks.append(testpalm_suite_runner)

        if self.Parameters.blockstat_tests:
            templates = self.get_registered_artifact_id('web4-micropackage')

            for platform in self.Parameters.pulse_platforms:
                subtasks.append(self.create_blockstat_test_subtask(
                    task_type=SandboxCiBlockstatTest,
                    platform=platform,
                    templates=templates,
                    is_apphost=True,
                ))

        subtasks.append(self.create_docs_subtask(wait_tasks=hermione_task))
        subtasks.append(self.create_test_cost_reporter_subtask(wait_tasks=[testpalm_suite_runner, hermione_task, hermione_e2e_task]))
        subtasks.append(self.create_assert_metrics_stats_subtask(wait_tasks=[hermione_task]))
        subtasks.append(self.create_test_changes_collector_subtask())

        return subtasks

    @skip_subtask(SandboxCiWeb4Build, u'Не менялся код для данной задачи')
    def create_build_subtask(self, **params):
        return self.meta.declare_subtask(
            task_type=SandboxCiWeb4Build,
            project_tree_hash=None,
            **params
        )

    @skip_subtask(SandboxCiWeb4Linters, u'Не менялся код для данной проверки')
    def create_linters_subtask(self, wait_tasks=None):
        if self.config.is_enabled('tests', 'linters'):
            return self.meta.declare_subtask(
                task_type=SandboxCiWeb4Linters,
                wait_tasks=wait_tasks,
                description=self.Parameters.description,
                project_git_base_ref=self.Parameters.project_git_base_ref,
                project_git_base_commit=self.Parameters.project_git_base_commit,
                project_git_merge_ref=self.Parameters.project_git_merge_ref,
                project_git_merge_commit=self.Parameters.project_git_merge_commit,
                use_arc=self.use_arc,
            )

        return self.meta.skip_step(
            github_context=SandboxCiWeb4Linters.github_context,
            description=u'Линтеры отключены в Genisys',
            label='linters',
        )

    @skip_subtask(SandboxCiYappyDeploy, u'Не менялся код для данной задачи')
    def create_deploy_yappy_subtask(self, wait_output_parameters, suffix, overwrite_quota):
        if self.config.is_enabled('deploy', 'yappy'):
            return self.meta.declare_subtask(
                task_type=SandboxCiYappyDeploy,
                description=self.Parameters.description,
                waitable=bool(self.config.get_deep_value(['deploy', 'yappy', 'waitable'], True)),
                use_overlayfs=self.use_overlayfs,
                project_hash=self.Parameters.project_hash,
                build_artifacts_resources=self.get_build_artifacts(['web4']),
                templates_resources_ids=[
                    self.get_registered_artifact_id('web4-micropackage'),
                    self.get_registered_artifact_id('web4-lowload-micropackage'),
                ],
                wait_output_parameters=wait_output_parameters,
                suffix=suffix,
                keep_forever=self.Parameters.keep_beta_forever,
                do_find_dev_beta_slot=self.Parameters.do_find_dev_beta_slot,
                webhook_urls=self.Parameters.deploy_webhook_urls,
                overwrite_quota=overwrite_quota,
            )

        return self.meta.skip_step(
            github_context=SandboxCiYappyDeploy.github_context,
            description=u'Деплой отключён в Genisys',
            label='yappy',
        )

    def create_yappy_cleanup_subtask(self, wait_tasks, suffix, overwrite_quota):
        return self.meta.declare_subtask(
            task_type=SandboxCiWeb4YappyCleanup,
            description=self.Parameters.description,
            waitable=False,
            use_overlayfs=self.use_overlayfs,
            project_hash=self.Parameters.project_hash,
            build_artifacts_resources=self.get_build_artifacts(['web4']),
            wait_tasks=wait_tasks,
            suffix=suffix,
            overwrite_quota=overwrite_quota,
        )

    @skip_subtask(SandboxCiAbExperiments, u'Не менялся код для данной задачи')
    def create_ab_experiment_testid(self, deploy_subtask=None, yappyBeta=True, experiment_beta_slot=''):
        if not deploy_subtask:
            return self.meta.skip_step(
                github_context=SandboxCiAbExperiments.github_context,
                description=u'Создание test-id невозможно без деплоя беты',
                label='ab_experiment_testid',
            )

        if not self.config.is_enabled('deploy', 'ab_experiment'):
            return self.meta.skip_step(
                github_context=SandboxCiAbExperiments.github_context,
                description=u'Создание test-id эксперимента отключено в Genisys',
                label='ab_experiment_testid',
            )

        domain_suffix = '{}-{}'.format(self.project_name, self.Parameters.beta_suffix)

        return self.meta.declare_subtask(
            task_type=SandboxCiAbExperiments,
            description=self.Parameters.description,
            webhook_urls=self.Parameters.ab_experiments_webhook_urls,
            build_artifacts_resources=self.get_registered_artifact_ids(['web4']) if not self.use_overlayfs else [],
            use_overlayfs=self.use_overlayfs,
            config_path=self.Parameters.ab_experiments_config_path,
            experiment_title=domain_suffix,
            experiment_domain_prefix=domain_suffix,
            experiment_beta_slot=experiment_beta_slot,
            project_hash=self.Parameters.project_hash,
            wait_output_parameters={deploy_subtask.id: 'beta_slot'} if yappyBeta else {},
            use_slot_from_yappy=yappyBeta,
            wait_tasks=[] if yappyBeta else deploy_subtask
        )

    @skip_subtask(SandboxCiWeb4Unit, u'Не менялся код для данной проверки')
    def create_unit_tests_subtask(self, wait_tasks=None):
        return self.meta.declare_subtask(
            task_type=SandboxCiWeb4Unit,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            build_artifacts_resources=self.get_build_artifacts(['unit-tests']),
            reusable=True,
            ref=self.ref,
            project_base_hash=self.Parameters.project_base_hash,
            project_hash=self.Parameters.project_hash,
            use_overlayfs=self.use_overlayfs
        )

    def _get_platforms_to_run(self, platforms, task_type, tool):
        test_platforms = []

        for test_platform in platforms:
            test_label = '{}.{}'.format(tool, test_platform)

            if not self.config.is_enabled('tests', tool.replace('-', '_')):
                self.meta.skip_step(
                    github_context=task_type.format_github_context(test_platform),
                    description=u'{}-тесты отключены'.format(tool),
                    label=test_label,
                )
            elif self.need_to_skip_check(test_label):
                self.meta.skip_step(
                    github_context=task_type.format_github_context(test_platform),
                    description=u'Не менялся код для данной платформы',
                    label=test_label,
                    reason='not modified',
                )
            else:
                test_platforms.append(test_platform)

        return test_platforms

    @in_case_of('use_overlayfs', 'get_build_artifacts_in_overlayfs_mode')
    @in_case_of('sqsh_artifacts', 'get_build_artifacts_in_sqsh_mode')
    def get_build_artifacts_to_run_tests(self, artifacts):
        return self.get_registered_artifact_ids(artifacts)

    @in_case_of('use_overlayfs_in_palmsync_sync', 'get_build_artifacts_in_overlayfs_mode')
    @in_case_of('sqsh_artifacts', 'get_build_artifacts_in_sqsh_mode')
    def get_build_artifacts_for_testpalm_task(self, artifacts):
        return self.get_registered_artifact_ids(artifacts)

    def get_build_artifacts_in_sqsh_mode(self, *args, **kwargs):
        return self.get_registered_artifact_ids(['web4.sqsh'])

    @in_case_of('use_overlayfs', 'get_build_artifacts_in_overlayfs_mode')
    def get_build_artifacts(self, artifacts):
        return self.get_registered_artifact_ids(artifacts)

    def get_build_artifacts_in_overlayfs_mode(self, *args, **kwargs):
        return self.get_registered_artifact_ids(['web4.squashfs'])

    def get_registered_artifact_ids(self, artifact_types):
        return map(self.get_registered_artifact_id, artifact_types)

    def get_build_artifact_resources_to_run_tests(self):
        """
        remove me after https://st.yandex-team.ru/FEI-14115
        """
        return map(self.get_registered_artifact_id, ['web4.sqsh'] if self.config.get_deep_value(['sqsh_artifacts'], False) else ['web4', 'features', 'hermione'])

    def create_hermione_task(self, task_type, platforms, tool='hermione', **params):
        test_platforms = self._get_platforms_to_run(platforms=platforms, task_type=task_type, tool=tool)
        # в конфиге генезиса секция называется hermione_e2e
        config_tool = tool.replace('-', '_')

        if len(test_platforms):
            return self.meta.declare_subtask(
                task_type=task_type,
                tool=tool,
                additional_tags=[tool.upper()],
                platforms=test_platforms,
                waitable=bool(self.config.get_deep_value(['tests', config_tool, 'waitable'], True)),
                sqsh_build_artifacts_resources=self.get_build_artifacts_to_run_tests(['web4', 'features', 'hermione']),
                build_artifacts_resources=self.get_build_artifacts(['hermione']),
                reusable=False,
                ref=self.ref,
                changed_files=self.changed_files_filename,
                project_base_hash=self.Parameters.project_base_hash,
                project_base_tree_hash=self.Parameters.project_base_tree_hash,
                project_hash=self.Parameters.project_hash,
                selective_run=bool(self.config.get_deep_value(['tests', config_tool, 'selective'], False)),
                html_reporter_use_sqlite=bool(self.config.get_deep_value(['tests', config_tool, 'html_reporter_use_sqlite'], False)),
                use_overlayfs=self.use_overlayfs,
                **params
            )

    @skip_subtask(SandboxCiPalmsyncValidate, u'Не менялся код для данной проверки')
    def create_palmsync_validate_subtask(self, **params):
        return self.meta.declare_subtask(
            task_type=SandboxCiPalmsyncValidate,
            description=self.Parameters.description,
            build_artifacts_resources=self.get_build_artifacts_to_run_tests(['web4', 'features', 'hermione']),
            project_name=self.project_name,
            project_base_hash=self.Parameters.project_base_hash,
            project_hash=self.Parameters.project_hash,
            reusable=True,
            waitable=bool(self.config.get_deep_value(['tests', 'palmsync_validate', 'waitable'], True)),
            notifications=self.build_release_critical_notifications(),
            use_overlayfs=self.use_overlayfs,
            **params
        )

    def create_molly_run_subtask(self, platform, wait_tasks=None, **params):
        """
        :param platform: на какой платформе запускать сервис
        :type platform: str
        :param wait_tasks: список зависимых тасок
        :type wait_tasks: list of sdk2.Task
        :rtype: sdk2.Task
        """
        return self.meta.declare_subtask(
            task_type=SandboxCiMollyRun,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            ref=self.ref,
            platform=platform,
            project_base_hash=self.Parameters.project_git_base_commit,
            waitable=bool(self.config.get_deep_value(['tests', 'molly', 'waitable'], True)),
            additional_tags=[platform],
            **params
        )

    def create_palmsync_synchronize_subtask(self):
        if not self.config.is_enabled('deploy', 'testpalm'):
            return

        return self.meta.create_subtask(
            task_type=SandboxCiPalmsyncSynchronize,
            description=self.Parameters.description,
            build_artifacts_resources=self.get_build_artifacts_for_testpalm_task(['web4', 'features', 'hermione']),
            project_name=self.project_name,
            testpalm_project_name=self.Parameters.testpalm_base_project_name,
            environ=self.Parameters.environ,
            project_hash=self.Parameters.project_hash,
            use_overlayfs=self.use_overlayfs_in_palmsync_sync,
        )

    def create_skip_validator_subtask(self, **params):
        return self.meta.declare_subtask(
            task_type=SandboxCiSkipValidator,
            description=self.Parameters.description,
            build_artifacts_resources=self.get_build_artifacts_to_run_tests(['web4', 'features', 'hermione']),
            project_name=self.project_name,
            project_hash=self.Parameters.project_hash,
            use_overlayfs=self.use_overlayfs,
            **params
        )

    def create_docs_subtask(self, wait_tasks=None):
        if self.config.is_enabled('deploy', 'serpdocs'):
            return self.meta.declare_subtask(
                task_type=SandboxCiWeb4Docs,
                wait_tasks=wait_tasks,
                build_artifacts_resources=self.get_build_artifacts(['web4', 'docs']),
                project_hash=self.Parameters.project_hash,
                project_git_merge_ref=self.Parameters.project_git_merge_ref,
                project_git_base_commit=self.Parameters.project_git_base_commit,
                project_git_merge_commit=self.Parameters.project_git_merge_commit,
                use_overlayfs=self.use_overlayfs,
            )

        return self.meta.skip_step(
            github_context=SandboxCiWeb4Docs.github_context,
            description=u'Сборка документации отключена',
            label='deploy.serpdocs',
        )

    def create_nanny_testing_release_semaphore(self):
        with self.memoize_stage.nanny_testing_semaphore:
            self.set_semaphore(
                capacity=1,
                name=self.nanny_testing_release_semaphore_name,
                release=(ctt.Status.RELEASED, ctt.Status.Group.BREAK, ctt.Status.FAILURE),
            )

    def create_nanny_ensure_testing_deploy_subtask(self, wait_tasks=None):
        if not self.is_release:
            return None

        issue_key = None
        if hasattr(self.Parameters, 'send_comment_to_issue'):
            issue_key = self.Parameters.send_comment_to_issue

        return self.meta.declare_subtask(
            task_type=SandboxCiNannyEnsureDeploy,
            release_id=self.nanny_testing_release_id,
            semaphore_name=self.nanny_testing_release_semaphore_name,
            poll_interval=300,
            attempts=12,
            failed_statuses=["DEPLOY_FAILED", "CANCELLED", "ROLLED_BACK"],
            issue_key=issue_key,
            waitable=False,
            wait_tasks=wait_tasks,
            notifications=self.build_release_critical_notifications(),
        )

    @property
    def nanny_testing_release_semaphore_name(self):
        return '{name}_{id}_testing_resource_release'.format(name=self.project_name, id=self.id)

    @property
    def nanny_testing_release_id(self):
        # Nanny генерирует id релиза в таком формате:
        # '{nanny_release_type}-{sandbox_task_id}-{sandbox_task_release_status}'
        return 'SANDBOX_RELEASE-{id}-TESTING'.format(id=self.id)

    def create_testpalm_suite_runner_subtask(self, wait_tasks, wait_output_parameters):
        should_start = should_start_testpalm_suite_runner_task(
            is_release=self.is_release,
            force_run=self.Parameters.force_assessors_run,
            config_path=self.Parameters.assessors_config_path,
            testpalm_project_suffix=self.testpalm_project_suffix,
            testpalm_base_project_name=self.Parameters.testpalm_base_project_name,
            ticket_id=self.Parameters.send_comment_to_issue,
        )

        if not should_start:
            return None

        return self.meta.declare_subtask(
            task_type=SandboxCiTestpalmSuiteRunner,
            wait_tasks=wait_tasks,
            wait_output_parameters=wait_output_parameters,
            waitable=False,
            description=self.Parameters.description,
            build_artifacts_resources=self.get_registered_artifact_ids(['web4']) if not self.use_overlayfs else [],
            use_overlayfs=self.use_overlayfs,
            config_path=self.Parameters.assessors_config_path,
            testpalm_project_suffix=self.testpalm_project_suffix,
            testpalm_base_project_name=self.Parameters.testpalm_base_project_name,
            notifications=self.build_release_critical_notifications(),
            use_arc=self.use_arc,
            project_name=self.project_name,
            project_hash=self.Parameters.project_hash,
            ticket_id=self.Parameters.send_comment_to_issue,
        )

    def create_testids_for_flags_subtask(self, wait_tasks):
        if not self.Parameters.testids_for_flags:
            return None

        config_path = self.working_path('web4/.config/testids-for-flags.json')
        config_exists = config_path.is_file()
        config_path = str(config_path)

        config = None

        if config_exists:
            logging.debug('testids-for-flags config exists, reading from {path}'.format(path=config_path))
            with open(config_path, 'r') as f:
                config = json.load(f)
                logging.debug('testids-for-flags config readed: {config}'.format(config=config))
        else:
            logging.debug('testids-for-flags config does not found by path {path}'.format(path=config_path))

        return self.meta.declare_subtask(
            task_type=SandboxCiTestidsForFlags,
            wait_tasks=wait_tasks,
            waitable=bool(self.config.get_deep_value(['tests', 'testids_for_flags', 'waitable'], False)),
            description=self.Parameters.description,
            project_name=self.project_name,
            testpalm_project_name=self.Parameters.testpalm_project_name,
            config=config,
            # Удалить после FEI-17530
            ignore_script_errors=True,
        )

    def create_expflags_subtask(self, wait_tasks=None):
        is_upload = self.Parameters.upload_created_exp_flags
        is_update = self.Parameters.update_changed_exp_flags
        deleted_exp_flags = self.Parameters.deleted_exp_flags

        if not is_upload and not is_update and not deleted_exp_flags:
            return self.meta.skip_step(
                github_context=SandboxCiExpflags.github_context,
                description=u'Синхронизация флагов не требуется',
                label='expflags.sync',
            )

        service = self.config.get_deep_value(['expflags', 'service'], [])
        component = self.config.get_deep_value(['expflags', 'component'], [])
        pattern = self.config.get_deep_value(['expflags', 'pattern'])
        handler = self.config.get_deep_value(['expflags', 'handler'])
        source = 'web4'
        validation = {'requireVteam': True, 'validateValuesType': True}

        return self.meta.declare_subtask(
            task_type=SandboxCiExpflags,
            wait_tasks=wait_tasks,
            waitable=True,
            description=self.Parameters.description,
            project_name=self.project_name,
            expflags_resource=self.get_registered_artifact_id('expflags'),
            expflags_pattern=pattern,
            expflags_options={'validation': validation},
            expflags_tags=['{}-pr-{}'.format(self.project_name, self.pr_number)],
            upload_exp_flags=is_upload,
            update_exp_flags=is_update,
            deleted_exp_flags=deleted_exp_flags,
            deleted_exp_flags_handler=handler,
            flag_template={'service': service, 'component': component, 'source': source},
            source=source,
        )

    def create_test_cost_reporter_subtask(self, wait_tasks=None):
        if not self.is_release:
            return None

        return self.meta.declare_subtask(
            task_type=SandboxCiTestCostReporter,
            waitable=False,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            target_task_id=self.id,
            release_version=self.Parameters.project_git_base_ref,
            required_task_type='SANDBOX_CI_WEB4',
            mode='release',
        )

    def create_pulse_subtasks(self, *args, **params):
        return create_pulse_subtasks(
            *args,
            notifications=self.build_release_critical_notifications(mailing_list=MailingLists.PULSE_MON),
            **params
        )

    def create_assert_metrics_stats_subtask(self, wait_tasks=None):
        if not self.config.is_enabled('tests', 'hermione_send_metrics_coverage'):
            return None

        return self.meta.create_subtask(
            task_type=InfratestAssertMetricsStats,
            waitable=False,
            wait_tasks=wait_tasks,
            description=self.Parameters.description,
            project_name=self.project_name
        )

    def create_test_changes_collector_subtask(self):
        if not self.config.is_enabled('tests', 'changes_collector'):
            return None

        return self.meta.create_subtask(
            task_type=SandboxCiTestChangesCollector,
            use_overlayfs=self.use_overlayfs,
            waitable=False,
            project_hash=self.Parameters.project_hash,
            build_artifacts_resources=self.get_build_artifacts_to_run_tests(['web4', 'features', 'hermione']),
            test_types=['hermione', 'hermione-e2e'],
        )

    def get_nanny_webhook_urls(self, additional_parameters):
        release_type = additional_parameters['release_status']

        if release_type == 'testing':
            return [
                'https://webhook-handler.si.yandex-team.ru/v1/sandbox/release-check-static?genisys-section={}'.format(
                    'search-interfaces.releases.{}'.format(self.project_name)
                )
            ]

        if release_type == 'stable':
            return ['https://webhook-handler.si.yandex-team.ru/v1/nanny/release-request-tickets-status-change']

    def get_nanny_webhook_type(self, additional_parameters):
        return 'RELEASE_WITH_TICKETS'

    def get_nanny_duplicate_policy_type(self, additional_parameters):
        return 'IGNORE'

    def on_release(self, additional_parameters):
        """
        :param additional_parameters: введенные данные из формы релизов
        :type additional_parameters: dict
        """
        release_type = additional_parameters['release_status']

        tags = set(self.Parameters.tags + [release_type])

        logging.debug('Releasing task in {release_type} with parameters: {parameters} and tags: {tags}'.format(
            release_type=release_type,
            parameters=additional_parameters,
            tags=tags,
        ))

        self.Parameters.tags = list(tags)

        SandboxCiWeb4Build.on_release(self, additional_parameters)
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

    @classmethod
    def is_test_task(cls, task):
        return isinstance(task, SandboxCiWeb4Unit) or super(SandboxCiWeb4, cls).is_test_task(task)
