# -*- coding: utf-8 -*-

import traceback
import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.common.utils import classproperty, singleton_property
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.resource_types import REPORT_STATIC_PACKAGE

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.resources.template_packages import WebMicroPackage, WebLowloadMicroPackage, WebExp1MicroPackage, WebExp2MicroPackage, WebExp3MicroPackage, WebExp4MicroPackage
from sandbox.projects.sandbox_ci.task import BaseMetaTask
from sandbox.projects.sandbox_ci.utils import flow, static
from sandbox.projects.sandbox_ci.utils.squashfs import common_mksquashfs_options
from sandbox.projects.sandbox_ci.managers.arc.arc_cli import arc_changed_files_cmd, arc_deleted_files_cmd
from sandbox.projects.sandbox_ci.managers.artifacts import ArtifactCacheStatus
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.managers import s3mds


class SandboxCiWeb4Build(BaseMetaTask):
    """Автосборка Серпа (serp/web4)"""

    name = 'SANDBOX_CI_WEB4_BUILD'  # Для обратной совместимости с sdk1 задачей

    project_name = 'web4'

    class Requirements(BaseMetaTask.Requirements):
        disk_space = 80 * 1024
        cores = 32

        environments = BaseMetaTask.Requirements.environments.default + (
            PipEnvironment('boto3', '1.4.4', use_wheel=True),
        )
        kill_timeout = 2400

    class Parameters(BaseMetaTask.Parameters):
        with sdk2.parameters.Group('Флаги') as flags_input_block:
            compute_changed_exp_flags = sdk2.parameters.Bool(
                'Найти затронутые флаги',
                default=False
            )
            upload_created_exp_flags = sdk2.parameters.Bool(
                'Загрузить добавленные флаги',
                description=u'Новые флаги из проектной папки expflags будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )
            update_changed_exp_flags = sdk2.parameters.Bool(
                'Обновить измененные флаги',
                description=u'Изменения в проектных флагах в папке expflags будут отправлены в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )
            deprecate_deleted_exp_flags = sdk2.parameters.Bool(
                'Пометить удаленные флаги как устаревшие',
                description=u'Удаленные флаги в папке expflags будут помечены как устаревшие в flag storage админки экспериментов. Подробнее https://nda.ya.ru/3UWbTQ',
                default=False
            )

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Флаги') as flags_output_block:
                changed_exp_flags = sdk2.parameters.List('Затронутые флаги')
                deleted_exp_flags = sdk2.parameters.List('Удаленные флаги')

            with sdk2.parameters.Group('Build') as output_build_block:
                is_static_uploaded = sdk2.parameters.Bool(
                    'Is static uploaded',
                    description='Была ли загружена статика',
                )

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'configure': 'npm run ci:config',
        'build': 'YPROJECT={build_platform} npm run ci:build',
        'save-git-changed-files': 'git diff --name-only --no-renames HEAD~1 > {changed_files_filename}',
        'save-git-deleted-files': 'git diff --name-only --diff-filter=D HEAD~1 > {deleted_files_filename}',
        'save-arc-changed-files': arc_changed_files_cmd() + ' > {changed_files_filename}',
        'save-arc-deleted-files': arc_deleted_files_cmd() + ' > {deleted_files_filename}',
        'print-changed-files': 'cat {changed_files_filename}',
        'print-deleted-files': 'cat {deleted_files_filename}',
        'changed-exp-flags': 'CHANGED_FILES="{changed_files_filename}" npm run ci:expflags',
        'ammo-counters': 'CHANGED_FILES="{changed_files_filename}" npm run ci:ammo-counter',
        'artifacts-src': 'EXCLUDE_ARTIFACTS={exclude_artifacts} ARTIFACTS_DIR={working_dir} npm run ci:artifacts:src',
        'artifacts-blt': 'EXCLUDE_ARTIFACTS={exclude_artifacts} ARTIFACTS_DIR={working_dir} npm run ci:artifacts:blt',
        'artifacts-sqsh': 'mksquashfs . {working_dir}/web4.sqsh -e .git ' + ' '.join(common_mksquashfs_options),
        'artifacts-squashfs': 'mksquashfs {build_results_dir} {working_dir}/web4.squashfs ' + ' '.join(common_mksquashfs_options),
    }

    @property
    def use_overlayfs(self):
        return self.Parameters.use_overlayfs or self.project_conf.get('use_overlayfs', False)

    def on_save(self):
        super(SandboxCiWeb4Build, self).on_save()

        try:
            use_overlayfs = self.use_overlayfs
            self.Parameters.use_overlayfs = use_overlayfs
        except Exception as error:
            logging.exception('Can not update parameter value: {}'.format(error))

    def is_mq_cache_semaphore_required(self):
        return False

    @singleton_property
    @in_case_of('use_overlayfs', 'lifecycle_artifacts_map_in_overlayfs_mode')
    @in_case_of('sqsh_artifacts', 'lifecycle_artifacts_map_in_sqsh_artifacts_mode')
    def lifecycle_artifacts_map(self):
        self.lifecycle.update_vars(exclude_artifacts='')

        return {
            'artifacts-src': [
                'features',
                'hermione',
                'docs',
                'expflags',
            ],
            'artifacts-blt': [
                'web4',
                'web4-micropackage',
                'web4-lowload-micropackage',
                'web4-static',
                'unit-tests',
            ],
        }

    def lifecycle_artifacts_map_in_sqsh_artifacts_mode(self):
        self.lifecycle.update_vars(exclude_artifacts='features')

        return {
            'artifacts-src': [
                'hermione',
                'docs',
                'expflags',
            ],
            'artifacts-blt': [
                'web4',
                'web4-micropackage',
                'web4-lowload-micropackage',
                'web4-static',
                'unit-tests',
            ],
            'artifacts-sqsh': [
                {
                    'type': 'web4.sqsh',
                    'relative_path': 'web4.sqsh',
                },
            ]
        }

    def lifecycle_artifacts_map_in_overlayfs_mode(self):
        self.lifecycle.update_vars(exclude_artifacts='web4,features,hermione,docs,unit-tests')

        return {
            'artifacts-src': [
                'expflags',
            ],
            'artifacts-blt': [
                'web4-micropackage',
                'web4-lowload-micropackage',
                'web4-static',
            ],
            'artifacts-squashfs': [
                {
                    'type': 'web4.squashfs',
                    'relative_path': 'web4.squashfs',
                },
            ],
        }

    @property
    def artifacts_resource_types(self):
        """
        :return: Тип и названия ресурсов.
        :rtype: dict
        """
        resource_types = {
            'web4-micropackage': self.get_micropackage_resource_type(),
            'web4-static': REPORT_STATIC_PACKAGE,
        }

        lowload_resource_type = self.get_lowload_micropackage_resource_type()
        if lowload_resource_type:
            resource_types['web4-lowload-micropackage'] = lowload_resource_type

        return resource_types

    def get_micropackage_resource_type(self):
        """
        В TEMPLATES_DEST_NAME передаётся название папки которая создаётся при сборке проекта в tartifacts.
        По имени папки можно опреледить какой должен быть ресурс у микропакета.
        @see SERP-60456, FEI-11080
        :return: Тип ресурса микропакета
        :rtype: sdk2.Resource
        """
        templates_id = self.get_templates_id()
        resource_type = {
            'web4': WebMicroPackage,
            'web3_exp': WebExp1MicroPackage,
            'web_exp': WebExp2MicroPackage,
            'web_exp3': WebExp3MicroPackage,
            'web_exp4': WebExp4MicroPackage,
        }.get(templates_id)

        if resource_type is None:
            raise Exception('There is resource type associated with TEMPLATES_DEST_NAME="{id}"'.format(
                id=templates_id,
            ))

        logging.debug('Resource type associated with TEMPLATES_DEST_NAME="{id} is {type}'.format(
            id=templates_id,
            type=resource_type,
        ))

        return resource_type

    def get_lowload_micropackage_resource_type(self):
        templates_id = self.get_templates_id()
        resource_type = {
            'web4': WebLowloadMicroPackage,
        }.get(templates_id)

        logging.debug('Lowload resource type associated with TEMPLATES_DEST_NAME="{id} is {type}'.format(
            id=templates_id,
            type=resource_type,
        ))

        return resource_type

    def get_templates_id(self):
        return self.environ.get('TEMPLATES_DEST_NAME', 'web4')

    @classproperty
    def task_label(self):
        return 'build'

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Сборка'

    @property
    @common.utils.singleton
    def lifecycle(self):
        return super(SandboxCiWeb4Build, self).lifecycle.update_vars(
            changed_files_filename=self.changed_files_filename,
            deleted_files_filename=self.deleted_files_filename,
            changed_exp_flags=self.changed_exp_flags_filename,
            build_platform=self.Parameters.build_platform
        )

    @property
    def changed_files_filename(self):
        # имя файла указано в паттернах tartifacts
        return 'changed-files.log'

    @property
    def changed_files_filepath(self):
        return '{}/{}'.format(self.project_dir, self.changed_files_filename)

    @property
    def deleted_files_filename(self):
        return 'deleted-files.log'

    @property
    def deleted_files_filepath(self):
        return '{}/{}'.format(self.project_dir, self.deleted_files_filename)

    @property
    def changed_exp_flags_filename(self):
        return 'changed-exp-flags.txt'

    @property
    def ammo_counters_filenames(self):
        return {'touch-phone': 'ammo-counter.touch-phone.txt',
                'desktop': 'ammo-counter.desktop.txt'}

    @property
    def changed_exp_flags_filepath(self):
        return '{}/{}'.format(self.project_dir, self.changed_exp_flags_filename)

    @property
    def ammo_counters_filepaths(self):
        return {platform: '{}/{}'.format(self.project_dir, filename)
                for platform, filename in self.ammo_counters_filenames.iteritems()}

    def _build(self, cache_status):
        super(SandboxCiWeb4Build, self)._build(cache_status)

        if cache_status is ArtifactCacheStatus.FOUND:
            self.Parameters.is_static_uploaded = True

    def configure(self):
        """
        Запускает шаг `configure` из `lifecycle_steps`, на котором проект конфигурируется и скачивается blockstat.
        Загрузить словарь (файл blockstat.dict.json) из ресурсов Сендбокса, если ресурс не найден загрузить из statface.

        Важно: флаг `-B` был специально удален из шага, чтобы принудительно не загружать словарь из statface!
        @see FEI-6315
        """
        with self.profile_action(actions_constants['BLOCKSTAT'], 'Downloading'):
            self.dependencies.blockstat_install()

        self.prepare_build_environ()

        super(SandboxCiWeb4Build, self).configure()

    def prepare_sources(self):
        super(SandboxCiWeb4Build, self).prepare_sources()

        vcs = 'arc' if self.use_arc else 'git'

        self.lifecycle('save-{}-changed-files'.format(vcs), check=False)
        self.lifecycle('save-{}-deleted-files'.format(vcs), check=False)

        self.lifecycle('print-changed-files', check=False)
        self.lifecycle('print-deleted-files', check=False)

        if self.Parameters.compute_changed_exp_flags:
            self.set_changed_exp_flags_parameter()

        if self.Parameters.deprecate_deleted_exp_flags:
            self.set_deleted_exp_flags_parameter()

        if self.Parameters.predict_ammo_counter:
            self.set_ammo_counters_parameter()

    @in_case_of('use_overlayfs', 'build_in_overlayfs_mode')
    @in_case_of('sqsh_artifacts', 'build_in_sqsh_artifacts_mode')
    def build(self):
        self.__build(build_artifact_lifecycle_steps=['artifacts-blt'])

    def build_in_sqsh_artifacts_mode(self):
        self.__build(build_artifact_lifecycle_steps=['artifacts-sqsh', 'artifacts-blt'])

    def build_in_overlayfs_mode(self):
        self.__build(build_artifact_lifecycle_steps=['artifacts-squashfs', 'artifacts-blt'])

    def __build(self, build_artifact_lifecycle_steps):
        def mkBuildArtifacts():
            super(SandboxCiWeb4Build, self).build()

            tasks_to_create_artifacts = map(lambda lifecycle_step: lambda: self.create_artifacts(lifecycle_step), build_artifact_lifecycle_steps)
            flow.parallel(apply, tasks_to_create_artifacts + [self._create_build_cache_artifact, self.upload_static, self.upload_storybook])

        flow.parallel(apply, [
            mkBuildArtifacts,
            lambda: self.create_artifacts('artifacts-src'),
        ])

    def create_artifacts(self, lifecycle_step=None):
        if not lifecycle_step:
            logging.debug('All artifacts are already built')
            return

        super(SandboxCiWeb4Build, self).create_artifacts(lifecycle_step)

    def set_changed_exp_flags_parameter(self):
        changed_files = self.get_changed_files()
        changed_exp_flags = self.get_changed_exp_flags(changed_files)

        self.Parameters.changed_exp_flags = changed_exp_flags

    def set_deleted_exp_flags_parameter(self):
        self.Parameters.deleted_exp_flags = self.get_deleted_exp_flags()

    def set_ammo_counters_parameter(self):
        self.Parameters.ammo_counters = self.get_ammo_counters()

    def get_changed_files(self):
        """
        :rtype: list of str
        """
        logger = logging.getLogger(name='get_changed_files')

        if not os.path.exists(self.changed_files_filepath):
            logger.debug('Can not find file {}'.format(self.changed_files_filepath))
            return []

        with open(self.changed_files_filepath, 'r') as file:
            content = file.read().strip()

            if content == '':
                logger.debug('No changed files')
                return []

            return content.split('\n')

    def get_changed_exp_flags(self, changed_files):
        """
        :param changed_files: Список изменившихся файлов
        :type changed_files: list of str
        :rtype: list of str
        """
        logger = logging.getLogger(name='get_changed_exp_flags')

        changed_files_as_str = ' '.join(map(lambda s: '"{}"'.format(s), changed_files))

        if changed_files_as_str == '':
            logger.debug('No changed exp flags')
            return []

        self.lifecycle('changed-exp-flags', check=False)

        if not os.path.exists(self.changed_exp_flags_filepath):
            logger.debug('Can not find file {}'.format(self.changed_exp_flags_filepath))
            return []

        with open(self.changed_exp_flags_filepath, 'r') as file:
            content = file.read().strip()

            if content == '':
                logger.debug('No changed flags')
                return []

            return content.split('\n')

    def get_ammo_counters(self):
        """
        :rtype: str on None
        """
        logger = logging.getLogger(name='get_ammo_counters')

        self.lifecycle('ammo-counters', check=False)

        result = {}
        for platform, filepath in self.ammo_counters_filepaths.iteritems():
            if not os.path.exists(filepath):
                logger.debug('Can not find file {}'.format(filepath))
                continue

            with open(filepath, 'r') as file:
                content = file.read().strip()

                if content == '':
                    logger.debug('No ammo counter for platform {}'.format(platform))
                    continue

                result[platform] = content

        return result


    def get_deleted_exp_flags(self):
        logger = logging.getLogger(name='get_deleted_exp_flags')

        if not os.path.exists(self.deleted_files_filepath):
            logger.debug('Can not find file {}'.format(self.deleted_files_filepath))
            return []

        with open(self.deleted_files_filepath, 'r') as file:
            exp_flags_path_prefix = 'expflags/'
            exp_flag_file_extension = '.json'

            get_flag_name = lambda file_path: file_path.strip().replace(exp_flags_path_prefix, '').replace(exp_flag_file_extension, '')
            starts_with_prefix = lambda file_path: file_path.strip().startswith(exp_flags_path_prefix)

            flags = [
                get_flag_name(file_path)
                for file_path in file.readlines()
                if starts_with_prefix(file_path)
            ]

            if not flags:
                logger.debug('No deleted flags')

            return flags

    def upload_static(self):
        if self.need_to_skip_check('static'):
            logging.info('Skip static uploading')
            self.Parameters.is_static_uploaded = False
            return

        with self.profile_action(actions_constants['DEPLOY_STATIC'], 'Deploy static'):
            freeze_path = os.path.join(self.project_name, 'freeze')
            abs_freeze_path = str(self.working_path(freeze_path))
            is_release_mode = self.is_release or self.Parameters.project_build_context == 'dev_production'

            if is_release_mode:
                bucket_name = 'web4static'
                key_prefix = '_/v2'
            else:
                bucket_name = 'serp-static-testing'
                key_prefix = freeze_path

            self.Parameters.is_static_uploaded = s3mds.upload_dir(
                freeze_path=abs_freeze_path,
                bucket_name=bucket_name,
                key_prefix=key_prefix,
                should_compress=is_release_mode,
                key_id=self.vault.read('env.SEARCH_INTERFACES_S3_ACCESS_KEY_ID'),
                access_key=self.vault.read('env.SEARCH_INTERFACES_S3_SECRET_ACCESS_KEY')
            )

    def upload_storybook(self):
        try:
            self._upload_storybook()
        except Exception as err:
            logging.error('Unable to deploy storybook:\n{}'.format(traceback.format_exc()))

    def _upload_storybook(self):
        build_context = self.Parameters.project_build_context
        create_id = {
            'pull-request': lambda: self.review_request_number or self.pr_number,
            'release': lambda: self.release_version,
            'dev': lambda: 'dev',
        }.get(build_context)

        if create_id is None:
            logging.info('Build context={context}. Step "Deploy storybook" has been skipped.'.format(context=build_context))
            return

        target_id = create_id()

        if self.need_to_skip_check('storybook'):
            logging.info('Selectivity. Step "Deploy storybook" has been skipped.')
            return

        with self.profile_action(actions_constants['DEPLOY_STORYBOOK'], 'Deploy storybook'):
            storybook_path = os.path.join(self.project_name, '.build/components-examples')
            abs_storybook_path = str(self.working_path(storybook_path))
            s3_path = os.path.join(self.project_name, target_id)
            bucket_name = 'static-storybook'

            s3mds.upload_dir(
                freeze_path=abs_storybook_path,
                bucket_name=bucket_name,
                key_prefix=s3_path,
                should_compress=False,
                key_id=self.vault.read('env.SEARCH_INTERFACES_S3_ACCESS_KEY_ID'),
                access_key=self.vault.read('env.SEARCH_INTERFACES_S3_SECRET_ACCESS_KEY')
            )

            logging.info('Storybook deployed to={to}'.format(to=s3_path))
