# -*- coding: utf-8 -*-
import logging
import posixpath
import re
from os import listdir
from functools import reduce
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr, task as ctt, misc as ctm
from sandbox.sandboxsdk import errors


from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import OverlayfsMixin, PrepareWorkingCopyMixin, BaseTask
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils import request
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants

RAMDRIVE_SIZE = 15 * 1024


class SandboxCiWeb4Docs(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """ Генерация документации Серпа (serp/web4) """

    name = 'SANDBOX_CI_WEB4_DOCS'

    project_name = 'web4'

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_hash = parameters.project_hash()
        project_git_merge_ref = parameters.project_git_merge_ref()
        project_git_base_commit = parameters.project_git_base_commit()
        project_git_merge_commit = parameters.project_git_merge_commit()

        with sdk2.parameters.Group('Serpdocs') as serpdocs_block:
            serpdocs_url = sdk2.parameters.String(
                'Serpdocs updater url',
                required=True,
                description='Адрес, на который будут отправляться данные для документации',
                default='https://serpdocs.si.yandex-team.ru/updater/update',
            )
            serpdocs_build_folder = sdk2.parameters.String(
                'Serpdocs build folder',
                required=True,
                description='Директория, в которой будет собираться документация',
                default='docs-build',
            )

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    class Requirements(BaseTask.Requirements):
        cores = 1
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'make_docs': 'npm run ci:build:docs'
    }

    github_context = u'[Sandbox CI] Генерация документации'

    report_description = 'Docs generation report'

    hermione_templar_logs = []
    files_to_send = []

    def on_save(self):
        super(SandboxCiWeb4Docs, self).on_save()

        if self.use_git_in_overlayfs_mode:
            self.set_ramdrive_size(25 * 1024)

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self._install_dependencies()
        self.get_test_reports()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.docs()

    def execute_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            with Node(self.Parameters.node_js_version), self._overlayfs(lower_dirs=[self.project_sources_dir],
                                         resources=self.Parameters.build_artifacts_resources,
                                         target_dir=self.project_dir):
                self.get_test_reports()
                self.docs()

    def docs(self):
        try:
            self.make_docs()
            for log in self.hermione_templar_logs:
                self.parse_logs(log)
        except TaskFailure as error:
            self.make_reports(self.find_docs_files(), ctt.Status.FAILURE)
            raise error

        files = self.find_docs_files()
        data = {
            'pull_request': self.Parameters.project_git_merge_ref,
            'base_commit': self.Parameters.project_git_base_commit,
            'merge_commit': self.Parameters.project_git_merge_commit
        }

        self.make_reports(files, ctt.Status.SUCCESS)
        self.send_post(
            url=self.Parameters.serpdocs_url,
            files=files,
            data=data,
        )

    def get_test_reports(self):
        wait_tasks_ids = self.meta.flatten_task_ids(self.Parameters.wait_tasks)
        logging.debug('getting resources from tasks {}'.format(wait_tasks_ids))
        artifact_resources = reduce(self.get_resource, wait_tasks_ids, [])

        self.hermione_templar_logs = artifact_resources

    def get_resource(self, resources, task_id):
        logging.debug('getting resource from task {}'.format(task_id))
        resource = self.artifacts.get_last_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            task_id=str(task_id),
            type='yandex-int-logs',
            state=ctr.State.READY,
        )

        if (isinstance(resource, sdk2.Resource)):
            test_task = sdk2.Task[task_id]
            platform = test_task.Parameters.platform
            path = self.artifacts.sync_build_artifacts([resource])
            path_to_rr_log = posixpath.join(str(path[0]), 'rr')
            logging.debug('logs path for {} is {}'.format(platform, path_to_rr_log))
            resources.append({
                'path': path_to_rr_log,
                'platform': platform,
            })

        return resources

    def make_docs(self):
        try:
            with self.profile_action(actions_constants['RUN'], 'Running docs generation'):
                self.lifecycle('make_docs')
        except errors.SandboxSubprocessError:
            raise TaskFailure('Docs generation error, check make_docs.out.txt')

    def send_post(self, url, files, data):
        """
        :param url: URL загрузчика документации
        :type url: str
        :type files: list of str
        :param data: объект с пулл-реквестом и хэшем коммита
        :type data: dict
        :rtype: dict
        """
        try:
            logging.debug('sending {} from {}'.format(files, self.project_path(self.Parameters.serpdocs_build_folder)))
            res = request.send_request(
                method='post',
                url=url,
                data=data,
                files=self.prepare_post_files(files),
                timeout=120,
            )

            logging.debug('response for {url} with code {code} is: {text}'.format(
                url=url,
                code=res.status_code,
                text=res.text,
            ))

            return res
        except Exception as error:
            logging.exception('request to {url} failed with {error}'.format(
                url=url,
                error=error
            ))
            raise TaskFailure('docs sending error: {}, check debug.log'.format(error.message))

    def parse_logs(self, log):
        if not log['path'] or not log['platform']:
            return None

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.lifecycle_steps.update({'parse_logs': self.generate_parse_logs_step(log)})

            try:
                self.lifecycle('parse_logs')
            except TaskFailure as error:
                raise error

    def generate_parse_logs_step(self, log):
        return 'PLATFORM={} node ./tools/serpdocs/blocks-usage.js {}'.format(log['platform'], log['path'])

    def find_docs_files(self):
        files = listdir(str(self.project_path(self.Parameters.serpdocs_build_folder)))

        return filter(lambda file: file == 'serpdocs.json' or re.search('^[-a-z]+\-blocks\-usage\.json$', file), files)

    def prepare_post_files(self, file_names):
        result = {}
        for file_name in file_names:
            file_path = self.project_path(posixpath.join(self.Parameters.serpdocs_build_folder, file_name))
            result[file_name.replace('.json', '')] = file_path.open()

        return result

    def make_reports(self, files, status):
        for file_name in files:
            file_path = self.project_path(posixpath.join(self.Parameters.serpdocs_build_folder, file_name))

            self.artifacts.create_report(
                resource_path=file_path,
                type='docs-json',
                status=status,
                add_to_context=True
            )
