# -*- coding: utf-8 -*-

from sandbox.projects.release_machine import rm_notify as rm_notify

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.constants import MailingLists
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.resources import WebMicroPackage
from sandbox.projects.sandbox_ci.sandbox_ci_hermione import SandboxCiHermione
from sandbox.projects.sandbox_ci.sandbox_ci_web4_hermione_e2e import SandboxCiWeb4HermioneE2E
from sandbox.projects.sandbox_ci.pulse.pulse_shooter_custom import PulseShooterCustom
from sandbox.projects.sandbox_ci.task.experiments import BaseExperimentsTask


@rm_notify.notify2()
class SandboxCiWeb4Experiments(BaseExperimentsTask):
    """
    Мета-задача для запуска тестов web4 с экспериментальными флагами
    """
    name = 'SANDBOX_CI_WEB4_EXPERIMENTS'

    project_name = u'web4'

    class Parameters(BaseExperimentsTask.Parameters):
        with BaseExperimentsTask.Parameters.repo_version_block() as repo_version_block:
            project_hash_for_tests = parameters.CommitHash('Project commit hash for tests', default='')

    @property
    @in_case_of('use_overlayfs', 'build_artifact_types_in_overlayfs_mode')
    def build_artifact_types(self):
        return ['web4.sqsh']

    def build_artifact_types_in_overlayfs_mode(self):
        return ['web4.squashfs']

    @property
    def pulse_platforms(self):
        platforms = []
        for platform in self.Parameters.platforms:
            if platform == 'desktop':
                platforms.append('desktop')
            if platform == 'touch-phone':
                platforms.append('touch')
        return platforms

    def create_subtasks(self, resources):
        subtasks = []

        for tool in self.Parameters.tools:
            env = getattr(self.Parameters, self._get_env_param_name(tool), None)
            task_type = self._get_task_type(tool)

            if tool == 'hermione-e2e':
                subtasks.append(self.create_hermione_e2e_subtask(task_type, resources, self.Parameters.platforms, env))
            elif tool == 'hermione':
                subtasks.append(self.create_hermione_subtask(task_type, resources, self.Parameters.platforms, env))
            elif tool == 'pulse-shooter':
                subtasks += self.create_pulse_subtasks()

        return subtasks

    @staticmethod
    def _get_task_type(tool):
        if tool == 'hermione':
            return SandboxCiHermione
        if tool == 'hermione-e2e':
            return SandboxCiWeb4HermioneE2E

    def create_hermione_e2e_subtask(self, task_type, resources, platforms, env, **kwargs):
        return self.meta.create_subtask(
            tags=['flags'],
            platforms=platforms,

            hermione_base_url=self.Parameters.hermionee2e_base_url,
            project_name=self.project_name,

            html_reporter_use_sqlite=bool(self.config.get_deep_value(['tests', 'hermione_e2e', 'html_reporter_use_sqlite'], False)),
            sqsh_build_artifacts_resources=resources,

            **self._get_common_task_params(task_type, resources, env, **kwargs)
        )

    def create_hermione_subtask(self, task_type, resources, platforms, env, **kwargs):
        return self.meta.create_subtask(
            tags=['flags'],
            platforms=platforms,

            html_reporter_use_sqlite=bool(self.config.get_deep_value(['tests', 'hermione', 'html_reporter_use_sqlite'], False)),
            sqsh_build_artifacts_resources=resources,

            **self._get_common_task_params(task_type, resources, env, **kwargs)
        )

    def create_pulse_subtasks(self, **kwargs):
        build_micropackage_artifact = self.get_build_artifact('web4-micropackage', WebMicroPackage)
        common_query_params = ['exp_flags=test_tool=pulse']

        common_params = dict(
            family='pulse',
            task_type=PulseShooterCustom,
            tags=['flags'],
            project=self.project_name,

            use_soy=True,
            soy_operation_priority=self.Parameters.pulse_shooter_priority,
            base_query_params=common_query_params + ['no-tests=1'],
            build_actual_ammo=True,
            actual_query_params=common_query_params + self.Parameters.pulse_shooter_query_params,

            apphost_mode=True,
            requests_number=10000,
            access_log_threshold=9900,
            plan_threshold=9000,
            request_limit=15000,

            base_templates_package=build_micropackage_artifact,
            actual_templates_package=build_micropackage_artifact,

            notifications=self.build_release_critical_notifications(mailing_list=MailingLists.PULSE_MON),
            waitable=False,

            **kwargs
        )

        subtasks = []
        for platform in self.pulse_platforms:
            subtasks.append(self.meta.create_subtask(platform=platform, **common_params))

        return subtasks

    def _get_common_task_params(self, task_type, resources, env, **kwargs):
        return super(SandboxCiWeb4Experiments, self)._get_common_task_params(
            task_type=task_type,
            resources=resources,
            env=env,

            # Overlayfs
            use_overlayfs=self.use_overlayfs,
            project_hash=self.Parameters.project_hash_for_tests,

            **kwargs
        )
