# -*- coding: utf-8 -*-

import os.path

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.utils import classproperty
from sandbox.common.types import task as ctt

from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask
from sandbox.projects.sandbox_ci.parameters import CommitHash
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.sandbox_ci_web4 import SandboxCiWeb4
from sandbox.projects.sandbox_ci.sandbox_ci_web4_generate_baobab_counters.resources import SANDBOX_CI_WEB4_BAOBAB_COUNTERS


class SandboxCiWeb4GenerateBaobabCounters(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """ Генерация ресурса с баобаб-счетчиками Cерпа """
    name = 'SANDBOX_CI_WEB4_GENERATE_BAOBAB_COUNTERS'

    class Parameters(BaseTask.Parameters):
        with sdk2.parameters.Group('Generate from') as generate_from:
            gen_git_ref = sdk2.parameters.String('Git ref', required=True, default='dev')
            gen_git_commit = CommitHash('Commit')
            gen_tags = sdk2.parameters.List('Tags', sdk2.parameters.String, default=['DEV'])
            gen_debug_mode = sdk2.parameters.Bool('Debug mode', default=False)

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    project_name = 'web4'

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Генерация списка баобаб-счетчиков'

    @classproperty
    def task_label(self):
        return 'generate-baobab-counters'

    def execute(self):
        web4_task = self._get_referenced_web4_task()
        resources = self._get_compatible_source_resources(web4_task)
        project_hash = web4_task.Parameters.project_hash

        if self.Parameters.use_overlayfs:
            with self.prepare_working_copy_context(dict(commit=project_hash), dict(checkout_params=web4_task.Parameters.git_checkout_params)):
                with self._overlayfs(lower_dirs=[self.project_sources_dir], resources=resources, target_dir=self.project_dir):
                    self._generate_counters()
        else:
            self._download_sources(resources, self.project_dir)
            self._generate_counters()

    def _get_referenced_web4_task(self):
        ref = self.Parameters.gen_git_ref
        commit = self.Parameters.gen_git_commit

        input_parameters = dict(
            project_github_owner=self.Parameters.project_github_owner,
            project_github_repo=self.Parameters.project_github_repo,
            project_git_base_ref=ref,
        )

        if commit:
            input_parameters['project_git_base_commit'] = commit

        task = sdk2.Task.find(
            type=SandboxCiWeb4,
            tags=self.Parameters.gen_tags,
            status=ctt.Status.SUCCESS,
            children=False,
            input_parameters=input_parameters
        ).order(-sdk2.Task.id).limit(1).first()

        if not task:
            raise TaskFailure('Referenced task ({}, {}) not found.'.format(ref, commit))

        # it is not in input_parameters query just to simplify debugging
        if task.Parameters.use_overlayfs != self.Parameters.use_overlayfs:
            raise TaskFailure('Task #{} has non-matching use_overlayfs param ({}).'.format(task.id, task.Parameters.use_overlayfs))

        return task

    def _get_compatible_source_resources(self, web4_task):
        resource_type = 'web4.squashfs' if web4_task.Parameters.use_overlayfs else 'web4.sqsh'

        resource = sdk2.Resource.find(
            type=SANDBOX_CI_ARTIFACT,
            task=web4_task,
            attrs=dict(type=resource_type)
        ).order(-sdk2.Resource.id).limit(1).first()

        if resource is None:
            return []

        return [resource]

    def _generate_counters(self):
        resource_path = os.path.join(os.getcwd(), 'baobab-counters')

        run_process(
            ['./tools/collect-counters > {}'.format(resource_path)],
            shell=True,
            work_dir=self.project_dir,
            log_prefix='collect-counters-script',
            outputs_to_one_file=False,
        )

        resource = sdk2.ResourceData(SANDBOX_CI_WEB4_BAOBAB_COUNTERS(
            self, self.Parameters.description, resource_path, debug=self.Parameters.gen_debug_mode
        ))

        resource.ready()
