# -*- coding: utf-8 -*-

import logging
import os

from sandbox import common
from sandbox import sdk2

from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import classproperty

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.task import OverlayfsMixin, PrepareWorkingCopyMixin, BaseTask
from sandbox.projects.sandbox_ci.utils import flow
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.managers import LifecycleManager
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants


REPORT_RESOURCES_PATHS = {
    'priv-report': {
        'resource_paths': ['priv-test_desktop.html', 'priv-test_touch-phone.html', 'priv-test_touch-pad.html'],
        'root_path': None,
    },
    'client-report': {
        'resource_paths': ['karma-report'],
        'root_path': None,
    },
    'hermione-helpers': {
        'resource_paths': ['hermione-helpers-tests.html'],
        'root_path': None,
    },
    'commonjs-tests': {
        'resource_paths': ['renderer-test.html'],
        'root_path': None,
    },
}

RAMDRIVE_SIZE = 15 * 1024


class SandboxCiWeb4Unit(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """
    Модульные тесты Серпа (serp/web4)
    """
    name = 'SANDBOX_CI_WEB4_UNIT'

    class Requirements(BaseTask.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)
        kill_timeout = 900

    # https://st.yandex-team.ru/FEI-15349 – пробуем не запускать unit-тесты на мультислотах из-за деградации во времени
    #     cores = 4

    #     class Caches(sdk2.Requirements.Caches):
    #         pass

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()
        project_tree_hash = parameters.project_tree_hash()
        ref = sdk2.parameters.String('Build Branch')

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    project_name = 'web4'

    lifecycle_steps = {
        'npm_install': 'npm ci'
    }

    unit_tests = {
        'priv-report': 'npm run ci:test:priv',
        'client-report': 'npm run ci:test:ibem',
        'hermione-helpers': 'npm run ci:test:helpers',
        'commonjs-tests': 'npm run ci:test:common'
    }

    @classproperty
    def task_label(self):
        return 'unit-tests'

    @classmethod
    def format_github_context(cls):
        return u'[Sandbox CI] Unit-тесты'

    @classproperty
    def github_context(self):
        return self.format_github_context()

    @property
    @common.utils.singleton
    def run_test_process(self):
        return LifecycleManager(self, self.unit_tests, self.project_dir)

    def on_save(self):
        super(SandboxCiWeb4Unit, self).on_save()

        if self.use_git_in_overlayfs_mode:
            self.set_ramdrive_size(25 * 1024)

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self.set_environments()
        self._install_dependencies()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.test()

    def execute_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            self.set_environments()

            with Node(self.Parameters.node_js_version), self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                self.test()

    def set_environments(self):
        os.environ['MOCHA_HTML_REPORTER'] = 'true'
        os.environ['MOCHA_COVERAGE_REPORT'] = 'true'

    def test(self):
        with self.profile_action(actions_constants['TEST'], 'Running unit tests'):
            failed_unit_tests = []

            def _generate_report((report_type, report_desc)):
                process = self.run_test_process(report_type, check=False)
                is_failed = bool(process.wait())

                if is_failed:
                    failed_unit_tests.append(report_type)

                report_status = ctt.Status.FAILURE if is_failed else ctt.Status.SUCCESS

                for res_path in report_desc['resource_paths']:
                    resource_path = self.project_dir / res_path
                    self.artifacts.create_report(
                        resource_path=resource_path,
                        add_to_context=True,
                        resource_type=SANDBOX_CI_ARTIFACT,
                        type=res_path,
                        status=report_status,
                        root_path=report_desc['root_path'] or '',
                        public=True,
                    )

                out_text_resource_path = report_type + '.out.txt'
                self.artifacts.create_report(
                    resource_path=self.artifacts.duplicate_artifact_from_task_log(out_text_resource_path),
                    type=out_text_resource_path,
                    add_to_context=True,
                    status=report_status,
                    public=True,
                )

            flow.parallel(_generate_report, list(REPORT_RESOURCES_PATHS.iteritems()))

        self.register_coverage_report()

        if failed_unit_tests:
            raise TaskFailure('{} failed, see reports'.format(', '.join(failed_unit_tests)))

    def register_coverage_report(self):
        coverage_commonjs_test_resource = self.artifacts.create_report(
            resource_path=self.project_dir / 'coverage-commonjs-test',
            add_to_context=True,
            resource_type=SANDBOX_CI_ARTIFACT,
            type='coverage-commonjs-test',
            status=ctt.Status.SUCCESS,
            root_path='index.html',
            public=True,
        )

    def on_before_end(self, status):
        super(SandboxCiWeb4Unit, self).on_before_end(status)

        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)
