# -*- coding: utf-8 -*-
import subprocess
import json

from sandbox import sdk2
from sandbox.common.utils import classproperty

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants


class SandboxCiYappyDeploy(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """Таск деплоя бет в Yappy"""
    class Requirements(BaseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_hash = parameters.project_hash()
        webhook_urls = BaseTask.Parameters.webhook_urls(default=[])

        with sdk2.parameters.Group('Параметры деплоя') as deploy_block:
            templates_resources_ids = sdk2.parameters.List('Пакет с шаблонами', required=True)
            suffix = sdk2.parameters.String('Суффикс для беты (pull-1234)', required=True)
            config_path = sdk2.parameters.String('Путь до файла с конфигом', default='./.config/renderer-betas/config.json')
            keep_forever = sdk2.parameters.Bool('Отключить TTL (вечная бета)', default=False)
            do_find_dev_beta_slot = sdk2.parameters.Bool('Найти слот, занимаемый бетой для dev', default=False)
            overwrite_quota = sdk2.parameters.String('Заменить квоту по умолчанию на указанную', default='')

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

        with sdk2.parameters.Output():
            beta_name = sdk2.parameters.String('Поддомен беты на hamster')
            beta_slot = sdk2.parameters.String('Endpoint контейнера с бетой (для srcrwr)')
            dev_beta_name = sdk2.parameters.String('Поддомен dev-ветки на hamster')
            dev_beta_slot = sdk2.parameters.String('Endpoint контейнера с бетой dev-ветки')

    lifecycle_steps = {
        'npm_install': 'npm ci'
    }

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Деплой в Yappy'

    @classproperty
    def task_label(self):
        return 'yappy-deploy'

    @property
    def project_name(self):
        return self.Parameters.project_github_repo

    def on_enqueue(self):
        super(SandboxCiYappyDeploy, self).on_enqueue()

        self.ensure_static_is_uploaded()
        self.task_dependencies.ensure_output_parameter('is_artifacts_ready', True, 'Template packages are not ready')

    def execute(self):
        if self.Parameters.use_overlayfs:
            with self.prepare_working_copy_context():
                with self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                    self._install_dependencies()
                    self.deploy()
        else:
            self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
            self._install_dependencies()
            self.deploy()

    def deploy(self):
        with self.profile_action(actions_constants['DEPLOY_YAPPY'], 'Running yappy deploy'):
            # package version is specified in projects' package.json
            cmd = 'npx @yandex-int/renderer-betas-cli create -c "{config_path}" --suffix "{suffix}" --all-slots'.format(
                config_path=self.Parameters.config_path,
                suffix=self.Parameters.suffix,
            )

            for res_id in self.Parameters.templates_resources_ids:
                cmd += ' --resource {}'.format(res_id)

            if self.Parameters.keep_forever:
                cmd += ' --keep-forever'

            if self.Parameters.overwrite_quota:
                cmd += ' --overwrite-quota {}'.format(self.Parameters.overwrite_quota)

            json_out = self._run_proc_and_get_stdout(cmd, 'create-beta')
            self.Parameters.beta_slot = ';'.join(json_out.get('slot'))
            self.Parameters.beta_name = json_out.get('betaName')

            if self.Parameters.do_find_dev_beta_slot:
                self.find_dev_beta_slot()

    def find_dev_beta_slot(self):
        cmd = 'npx @yandex-int/renderer-betas-cli get-beta-info -c "{config_path}" --suffix "{suffix}" --all-slots'.format(
            config_path=self.Parameters.config_path,
            suffix='dev',
        )

        json_out = self._run_proc_and_get_stdout(cmd, 'get-beta-info')
        self.Parameters.dev_beta_slot = ';'.join(json_out.get('slot'))
        self.Parameters.dev_beta_name = json_out.get('betaName')

    def _run_proc_and_get_stdout(self, cmd, log_prefix):
        with Node(self.Parameters.node_js_version), self.vault.ssh_key():
            proc = run_process(
                [cmd],
                shell=True,
                work_dir=self.project_dir,
                log_prefix=log_prefix,
                outputs_to_one_file=False,
                stdout=subprocess.PIPE,
            )

            stdout, _ = proc.communicate()
            return json.loads(stdout.strip())
