# -*- coding: utf-8 -*-

import os
import logging
import shutil
import re
import traceback
from contextlib import nested
from urlparse import urlparse
from copy import copy

from sandbox import sdk2

from sandbox.projects.sandbox_ci.managers.arc.context import create_arc_context
from sandbox.projects.sandbox_ci.managers.arc.arc_cli import arc_push
from sandbox.common.types import misc as ctm, task as ctt
from sandbox.common.errors import TaskFailure, TaskStop, TaskError
from sandbox.common.utils import NullContextmanager, singleton_property
from sandbox.sandboxsdk.errors import SandboxSubprocessError

from sandbox.projects.sandbox_ci.utils import flow, env

from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.utils.metrics import extract_pwc_metrics
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node, GitWithoutLfsProcessing, Debug, Overlayfs, DaemonProcesses
from sandbox.projects.sandbox_ci.utils.ref_formatter import strip_ref_prefix
from sandbox.projects.sandbox_ci.utils.fs import rmrf
from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, BaseTask
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import testpalm
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT, SANDBOX_CI_BUILD_CACHE_IMAGE
from sandbox.projects.sandbox_ci.managers.artifacts import ArtifactCacheStatus
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.managers.actions_stat import ActionsStatSender
from sandbox.projects.sandbox_ci.utils.squashfs import mksquashfs
from sandbox.projects.sandbox_ci.utils.release import remove_release_branch_prefix

GIT_REFS_CLEAN_REGEX = r'refs/(tags|heads)/'
MERGE_CONFLICT_REGEX = r'(CONFLICT \(.+\):.*)$'
NONEXISTING_OBJECT_REGEX = r'cannot update ref .+ with nonexistent object ([a-z0-9]{40})'
WAIT_TIMEOUT = 5 * 3600  # маскимальное время ожидания сабтасок (5 часов)
DEFAULT_ARC_CLIENT_BIN = 'arc'


class BaseBuildTaskException(Exception):
    pass


class BaseBuildTask(PrepareWorkingCopyMixin, BaseTask):
    class Parameters(parameters.CommonParameters):
        with sdk2.parameters.Group('Source control') as source_block:
            project_git_base_ref = parameters.project_git_base_ref()
            project_git_base_commit = parameters.project_git_base_commit()
            project_git_merge_ref = parameters.project_git_merge_ref()
            project_git_merge_commit = parameters.project_git_merge_commit()
            project_current_branch = parameters.project_current_branch()

        base_params = BaseTask.Parameters

        with sdk2.parameters.Group('Misc') as misc_block:
            stop_outdated_tasks = parameters.stop_outdated_tasks()
            skip_lfs_checkout = parameters.skip_lfs_checkout()

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('События сборки') as build_events:
                is_artifacts_ready = sdk2.parameters.Bool(
                    'Artifacts ready',
                    description='Параметр будет выставлен в True, когда задача подготовит ресурсы сборки.'
                                'Если задача не смогла подготовить ресурсы,'
                                'при завершении параметр будет выставлен в False.',
                )
                is_subtasks_ready = sdk2.parameters.Bool(
                    'Subtasks ready',
                    description='Параметр будет выставлен в True, когда задача подготовит подзадачи.'
                                'Если задача не смогла подготовить дочерние задачи,'
                                'при завершении параметр будет выставлен в False.',
                )

            with sdk2.parameters.Group('Project source hashes') as source_hashes:
                project_tree_hash = parameters.CommitHash('Source tree hash')
                project_base_hash = parameters.CommitHash('Source base commit hash')
                project_base_tree_hash = parameters.CommitHash('Source base commit tree hash')
                project_merged_refs = sdk2.parameters.List('Merged refs')
                project_range_base_hash = parameters.CommitHash('Range base commit hash')
                project_range_base_tree_hash = parameters.CommitHash('Range base commit tree hash')
                project_hash = parameters.project_hash()

            with sdk2.parameters.Group('Build') as output_build_block:
                static_url = sdk2.parameters.String(
                    'Static URL',
                    description='Полученный после шаблонизации URL для статики. '
                                'Пробрасывается в проектные скрипты сборки через переменную окружения STATIC_HOST',
                )
                static_origin = sdk2.parameters.String(
                    'Static URL origin',
                    description='Origin (протокол, хост и порт) URL для статики. '
                                'Пробрасывается в проектные скрипты сборки через '
                                'переменную окружения STATIC_ORIGIN, для возможности более гибкой настройки сборки',
                )
                static_path = sdk2.parameters.String(
                    'Static URL path',
                    description='Path URL для статики. '
                                'Пробрасывается в проектные скрипты сборки через '
                                'переменную окружения STATIC_PATH, для возможности более гибкой настройки сборки',
                )

            git_checkout_params = sdk2.parameters.JSON('Параметры, использованные для чекаута git-репозитория')

    class Context(BaseTask.Context):
        skipped_steps = []

    lifecycle_artifacts_map = {}
    artifacts_resource_types = {}
    artifacts_resource_type_default = SANDBOX_CI_ARTIFACT

    skip_ci_scripts_checkout = False

    @property
    def use_git_cache(self):
        return not self.use_arc

    @property
    def static_output_parameters_to_wait(self):
        return {
            self.id: 'is_static_uploaded',
        }

    @property
    def skip_lfs_checkout(self):
        return self.Parameters.skip_lfs_checkout

    @property
    def __normalized_lifecycle_artifacts_map(self):
        normalize = lambda artifact: {'type': artifact, 'relative_path': '{}.tar.gz'.format(artifact)}

        return {
            lifecycle_step: map(lambda artifact: artifact if isinstance(artifact, dict) else normalize(artifact), self.lifecycle_artifacts_map[lifecycle_step])
            for lifecycle_step in self.lifecycle_artifacts_map.keys()
        }

    @property
    def artifact_types(self):
        return [
            artifact['type']
            for artifacts in self.__normalized_lifecycle_artifacts_map.itervalues()
            for artifact in artifacts
        ]

    @property
    def ref(self):
        if self.Parameters.project_git_merge_ref:
            return self.Parameters.project_git_merge_ref[0]  # первый ref в списке указывает на head (@see FEI-5174)

        if self.Parameters.project_git_base_ref:
            return self.Parameters.project_git_base_ref

        if self.Parameters.project_git_merge_commit:
            return self.Parameters.project_git_merge_commit

        if self.Parameters.project_git_base_commit:
            return self.Parameters.project_git_base_commit

        return None

    @property
    def testpalm_project_suffix(self):
        """
        Суффикс проекта для testpalm

        :rtype: str
        """
        suffix = self.review_request_number or self.pr_number or self.release_version or self.ref
        if not suffix:
            raise BaseBuildTaskException('testpalm_project_suffix is asked, but no pr_number or release_version set')

        return testpalm.sanitize_project_name(suffix)

    @property
    def review_request_number(self):
        """
        Номер ревью-реквеста для сборки из arcanum

        :rtype: str or None
        """
        return str(self.Parameters.arcanum_review_request_id) if self.Parameters.arcanum_review_request_id else None

    @property
    def pr_number(self):
        """
        Первый найденный номер pull request для текущей сборки

        :rtype: str or None
        """
        return self.pr_numbers[0] if self.pr_numbers else None

    @property
    def pr_numbers(self):
        """
        Все номера pull request для текущей сборки

        :rtype [str]
        """
        def parse_number(str):
            pr_number = None
            try:
                pr_number = strip_ref_prefix(str)
            except Exception as error:
                logging.debug('Cannot detect pr_number: {}'.format(error))

            return pr_number

        return filter(lambda x: x, map(parse_number, self.Parameters.project_git_merge_ref))

    @property
    def release_version(self):
        """
        Версия релиза текущей сборки

        :rtype: str or None
        """
        if self.is_release:
            return remove_release_branch_prefix(branch=self.Context.current_branch)
        return None

    @property
    def use_arc(self):
        """
        Использовать arc для чекаута проекта

        :rtype: bool
        """
        use_arc = self.project_conf.get('use_arc', None)
        if use_arc is None:
            use_arc = self.Parameters.use_arc

        logging.debug('Use arc to checkout: {}'.format(use_arc))

        return use_arc

    @singleton_property
    def prefetch_files_config(self):
        return self.config.get_deep_value(['prefetch_files'], {})

    def check_ref_status(self):
        pr_branch = self.Parameters.project_git_merge_ref[0]  # первый ref в списке указывает на head (@see FEI-5174)

        pr_info = self.scripts.run_js('script/github/pr-info.js', {
            'owner': self.Parameters.project_github_owner,
            'repo': self.Parameters.project_github_repo,
            'pr-branch': pr_branch,
        })

        pr_state = pr_info['state']

        if pr_state != 'open':
            raise TaskStop('pull request #{} is {}'.format(pr_info['number'], pr_state))

    def prepare_workcopy(self):
        git_url = self.Context.project_git_url

        if self.project_conf.get('verbose_apply_tool', False):
            env.export({'VERBOSE_APPLY_TOOL': '1'})

        try:
            checkout_params = self.get_checkout_params()

            if not self.Parameters.git_checkout_params:
                self.Parameters.git_checkout_params = copy(checkout_params)

            with (GitWithoutLfsProcessing(git_url) if self.skip_lfs_checkout else NullContextmanager()):
                if self.skip_lfs_checkout:
                    # Если мы можем пропускать загрузку lfs-файлов, то не нужно
                    # запускать их обработку при checkout-е
                    # @see FEI-8054
                    logging.debug('Prepare working copy without LFS processing')
                else:
                    # Если не включена параллельная загрузка lfs-файлов,
                    # то выполняем её синхронно в процессе сборки
                    # @see FEI-4684
                    logging.debug('Prepare working copy with LFS processing')
                    checkout_params['lfs'] = True

                if self.use_arc:
                    checkout_params['use-arc'] = True
                    checkout_params['path-in-arcadia'] = self.Parameters.path_in_arcadia
                    checkout_params['arc-commit'] = self.Parameters.arc_ref

                    hashes = self._prepare_working_copy_sources(git_url, self.project_dir, checkout_params)
                    self.push_merge_commit_to_arc(hashes, checkout_params)
                else:
                    hashes = self._prepare_working_copy_sources(git_url, self.project_sources_dir, checkout_params)
                    os.symlink(str(self.project_sources_dir), str(self.project_dir))

        except SandboxSubprocessError as e:
            with open(e.stderr_full_path, 'r') as f:
                stderr = f.read()

            errors = []
            merge_conflict_messages = re.findall(MERGE_CONFLICT_REGEX, stderr, re.MULTILINE)
            if merge_conflict_messages:
                errors.append('<b>Merge conflicts:</b>')
                errors.extend(merge_conflict_messages)

            nonexisting_object = re.search(NONEXISTING_OBJECT_REGEX, stderr)
            if nonexisting_object:
                errors.append('Could not find object <b>{}</b> in your repository'.format(nonexisting_object.group(1)))

            if errors:
                self.set_info('<br>'.join(errors), do_escape=False)
                raise TaskFailure('Could not clone project, see logs for more details')
            else:
                raise e

        prepare_wc_metrics = extract_pwc_metrics(hashes)
        if prepare_wc_metrics:
            for tag, duration in prepare_wc_metrics.iteritems():
                self._register_action(tag, duration)

        profile = hashes.get('profile', {})
        git_cache_exist = profile.get('gitCacheExist', False)
        self.cache_profiler.register_action('git_cache_exist', 'git_repo', reused=git_cache_exist)

        if not self.Parameters.project_tree_hash:
            # store hashes in output parameters
            self.Parameters.project_tree_hash = hashes.get('merged', hashes['base'])['treeHash']
            self.Parameters.project_hash = hashes.get('merged', hashes['base'])['hash']
            self.Parameters.project_base_tree_hash = hashes['base']['treeHash']
            self.Parameters.project_base_hash = hashes['base']['hash']
            self.Parameters.project_merged_refs = map(lambda r: '{ref}:{hash}'.format(**r), hashes.get('mergeRefs', []))

            range_base_commit = self.get_base_commit(hashes)
            self.Parameters.project_range_base_hash = range_base_commit['hash']
            self.Parameters.project_range_base_tree_hash = range_base_commit['treeHash']

    def get_checkout_params(self):
        # Если рабочая копия ранее уже была получена, нужно клонировать такую же рабочую копию, использовать
        # те же `base` и `head` коммиты. Состояние рабочей копии содержится в выходных параметрах:
        # - `project_base_hash`
        # - `project_merged_refs`
        # @see FEI-5742
        git_base_ref = self.Parameters.project_git_base_ref
        git_base_commit = self.Parameters.project_base_hash or self.Parameters.project_git_base_commit

        git_merge_ref = None
        merge_ref = self.Parameters.project_merged_refs or self.Parameters.project_git_merge_ref
        if merge_ref:
            refs_list = list(merge_ref)
            # перенос первого элемента в конец списка, [a, b, c] → [b, c, a], @see FEI-5174
            refs_list = refs_list[1:] + refs_list[:1]
            git_merge_ref = ','.join(filter(lambda ref: ref is not None, refs_list))

        git_merge_commit = self.Parameters.project_git_merge_commit

        return {
            'ref': git_base_ref,
            'commit': git_base_commit,
            'merge-ref': git_merge_ref,
            'merge-commit': git_merge_commit,
        }

    def push_merge_commit_to_arc(self, hashes, checkout_params):
        merge_ref = checkout_params.get('merge-ref', None)
        push_merge_commit = self.project_conf.get('push_to_arc_after_pr_rebase', False)

        if merge_ref and push_merge_commit:
            with self.profile_action(actions_constants['ARC_PUSH'], 'Pushing commit to arc'):
                arc_push(self.arc_mount_path, hashes['merged']['hash'])

        return hashes

    def _register_action(self, tag, duration):
        self.profiler.register_action(tag, tag, duration)
        self.deep_actions_profiler.register_action(tag, tag, duration)

    def stop_outdated_tasks(self, ignore_task_ids=()):
        """
        Останавливает все предыдущие таски с такими же refs (base и head).

        :param ignore_task_ids: список тасок
        :type ignore_task_ids: iter of int
        """
        same_tasks = list(sdk2.Task.find(
            type=self.type.name,
            status=(ctt.Status.Group.QUEUE + ctt.Status.Group.EXECUTE + ctt.Status.Group.WAIT),
            input_parameters=dict(
                project_git_base_ref=self.Parameters.project_git_base_ref,
                project_git_merge_ref=self.Parameters.project_git_merge_ref,
            ),
        ).order(-sdk2.Task.id).limit(0))  # `limit(0)` — находим все задачи

        logging.debug('Found same tasks: {}'.format(same_tasks))

        # не останавливаем указанные задачи
        same_tasks = filter(lambda task: task.id not in ignore_task_ids, same_tasks)
        # останавливаем только те задачи, которые были созданы раньше, чем текущая
        outdated_tasks = filter(lambda task: task.id < self.id, same_tasks)
        # не останавливаем уже завершающиеся задачи
        finishing_statuses = (ctt.Status.STOPPING, ctt.Status.FINISHING)
        running_outdated_tasks = filter(lambda task: task.status not in finishing_statuses, outdated_tasks)

        self.meta.stop_tasks(running_outdated_tasks)

    def merge_stability_indexes(self):
        pass

    def check_subtasks(self, failed_subtasks=None, not_finished_subtasks=None, failed_skipped_steps=None):
        """
        :param failed_subtasks: список упавших тасок
        :type failed_subtasks: list of sdk2.Task
        :param not_finished_subtasks: список не завершенных тасок
        :type not_finished_subtasks: list of sdk2.Task
        :param failed_skipped_steps: список заскипанных тасок
        :type failed_skipped_steps: list of sdk2.Task
        """
        if failed_subtasks:
            self.meta.store_failed_tasks(failed_subtasks)

            raise TaskFailure('Has failed subtasks, see reports for more details')

        # Если по каким-то причинам таски не успели завершиться на текущий момент, то останавливаем
        # эти таски и записываем информацию о них в описание таски.
        if not_finished_subtasks:
            self.meta.store_not_finished_tasks(not_finished_subtasks)
            self.meta.stop_tasks(not_finished_subtasks)

            raise TaskFailure('Has not finished subtasks, see reports for more details')

        if failed_skipped_steps:
            raise TaskFailure('Has failed steps, see info for more details')

    def on_save(self):
        super(BaseBuildTask, self).on_save()

        self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, self.Requirements.disk_space, None)
        setattr(self.Context, '__do_not_dump_ramdrive', True)  # use setattr to avoid mangling

        # В push событие передается полное имя ветки и мы передаём его без изменений в параметры задачи.
        # Например:
        # - refs/heads/dev для ветки dev
        # - refs/heads/release/v1.0.0 для ветки release/v1.0.0
        # - refs/tags/v1.2.3 для тэгов v1.2.3
        self.Parameters.project_git_base_ref = re.sub(GIT_REFS_CLEAN_REGEX, '', self.Parameters.project_git_base_ref)

        self.Context.project_git_url = 'git@github.yandex-team.ru:{owner}/{repo}.git'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
        )

        # первый ref в списке указывает на head - project_git_merge_ref[0] (@see FEI-5174)
        if self.Parameters.project_git_merge_ref:
            self.Context.current_branch = self.Parameters.project_git_merge_ref[0]
        else:
            self.Context.current_branch = self.Parameters.project_git_base_ref

        try:
            use_arc = self.use_arc
            environ = self.Parameters.environ
            if use_arc:
                environ.update({'USE_ARC': '1'})
            else:
                environ.pop('USE_ARC', None)

            self.Parameters.use_arc = use_arc
            self.Parameters.environ = environ
        except Exception as error:
            logging.exception('Cannot update environ parameter value: {}'.format(error))

    def on_failure(self, prev_status):
        super(BaseBuildTask, self).on_failure(prev_status)
        self.artifacts.remove_not_ready_resources()

    def on_break(self, prev_status, status):
        super(BaseBuildTask, self).on_break(prev_status, status)
        self.artifacts.remove_not_ready_resources()

    def execute(self):
        with self.memoize_stage.build(commit_on_entrance=False):
            contexts = self.build_execute_contexts()

            with nested(*contexts):
                with DaemonProcesses(self.start_files_prefetching_processes(self.prefetch_files_config)):
                    if self.project_conf.get('check_ref_status', False):
                        self.check_ref_status()

                    self.prepare_sources()

                    self.add_tags()

                    self.check_lfs()

                    self.artifacts_resources, cache_status = self.register_artifacts()

                    if self.Parameters.stop_outdated_tasks and self.project_conf.get('stop_outdated_tasks', False):
                        self.stop_outdated_tasks(ignore_task_ids=(self.id,))

                flow.parallel(apply, [
                    self.__start_subtasks,
                    lambda: self._build(cache_status)
                ])

                if self.Context.subtasks:
                    # Максимальное время ожидания сабтасок
                    wait_timeout = max(self.Parameters.kill_timeout, WAIT_TIMEOUT)
                    wait_statuses = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK

                    raise sdk2.WaitTask(self.Context.subtasks, wait_statuses, timeout=wait_timeout)

        with self.memoize_stage.merge_stability_indexes():
            self.merge_stability_indexes()

        with self.memoize_stage.process():
            self.Context.report_resources = self.meta.get_subtasks_artifacts_ids()
            self.check_subtasks(
                failed_subtasks=self.meta.failed_subtasks,
                not_finished_subtasks=self.meta.not_finished_subtasks,
                failed_skipped_steps=self.meta.failed_skipped_steps,
            )

    def add_tags(self):
        try:
            project_tags = self.tags_manager.get_project_tags(
                work_dir=str(self.project_dir),
                owner=self.Parameters.project_github_owner,
                repo=self.Parameters.project_github_repo,
                base_hash=self.Parameters.project_base_hash,
                use_arc=self.use_arc,
            )

            tags = map(lambda tag: tag.lower(), project_tags + self.Parameters.tags)

            logging.debug('Adding tags: {}'.format(tags))

            self.Parameters.tags = list(set(tags))
        except Exception as error:
            logging.exception('Cannot add tags {}'.format(error))

    def build_execute_contexts(self):
        contexts = (GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key(),)

        if self.use_arc:
            mount_options = self._get_arc_workcopy_params()

            contexts += (create_arc_context(**mount_options),)

        return contexts

    def __start_subtasks(self):
        with self.profile_action(actions_constants['CREATE_SUBTASKS'], 'Creating subtasks'):
            created_subtasks = self.create_declared_subtasks()

        with self.profile_action(actions_constants['START_SUBTASKS'], 'Starting subtasks'):
            started_subtasks = self.start_declared_subtasks(created_subtasks)

        self.Context.subtasks = self.meta.get_waitable_tasks_ids(started_subtasks)
        self.Context.non_waitable_subtasks = self.meta.get_non_waitable_tasks_ids(started_subtasks)

    def _build(self, cache_status):
        if cache_status is ArtifactCacheStatus.FOUND:
            logging.info('Using cached build artifacts for project_tree_hash: {}'.format(
                self.Parameters.project_tree_hash,
            ))

            for res in self.artifacts_resources.itervalues():
                self.Context.reused_resources.append(res.id)
                self.cache_profiler.hit(res.__attrs__.type, 'resource')
        else:
            if cache_status is ArtifactCacheStatus.NOT_FOUND:
                logging.info('Cached build artifacts for following project_tree_hash are not found: {}'.format(
                    self.Parameters.project_tree_hash,
                ))

            if cache_status is ArtifactCacheStatus.IGNORED:
                logging.info('Skip using build artifacts by project_tree_hash')

            self._run_build()

        self.Parameters.is_artifacts_ready = True

    @in_case_of('use_overlayfs', '_run_build_in_overlayfs_mode')
    def _run_build(self):
        self.deps()
        self.configure()
        self.build()
        self.create_artifacts()

    def _run_build_in_overlayfs_mode(self):
        rmrf(self.project_dir)

        lower_dirs = [self.project_sources_dir]

        # Необходимо монтировать кэш сборки шаблонов параллельно установке зависимостей,
        # но пока это сделать нельзя из-за https://st.yandex-team.ru/SANDBOX-7998
        with Overlayfs(lower_dirs=lower_dirs, mount_point_symlink=self.project_dir):
            lower_dirs += self.deps()

        build_cache_dir = self._mount_build_cache_artifact()
        lower_dirs += [build_cache_dir] if build_cache_dir else []

        with Overlayfs(lower_dirs, mount_point_symlink=self.project_dir) as (upper_dir, mount_point):
            self.lifecycle.update_vars(build_results_dir=upper_dir)
            self.configure()
            self.build()
            self.create_artifacts()

    def _mount_build_cache_artifact(self):
        if not self.config.get_deep_value(['build', 'cache', 'reuse'], False):
            logging.debug('Reuse of build cache artifact is switched off')
            return

        with self.profile_action(actions_constants['REUSE_BUILD_CACHE'], 'Reuse build cache'):
            attrs = dict(project=self.project_name, project_tree_hash=self.Parameters.project_base_tree_hash)
            build_cache_resource = self.artifacts.get_last_artifact_resource(resource_type=SANDBOX_CI_BUILD_CACHE_IMAGE, **attrs)
            if not build_cache_resource:
                logging.debug('Build cache resource "{}" with attrs "{}" was not found'.format(SANDBOX_CI_BUILD_CACHE_IMAGE, attrs))
                return

            return str(self.artifacts.mount_build_artifact(build_cache_resource))

    def _create_build_cache_artifact(self):
        if not self.config.get_deep_value(['build', 'cache', 'create'], False):
            logging.debug('Creation of build cache artifact is switched off')
            return

        build_cache_dirname = os.environ.get('BUILD_CACHE_DIRNAME', '.build_cache')
        build_cache_path = self.project_dir / build_cache_dirname
        if not build_cache_path.is_dir():
            logging.debug('Build cache path "{}" is not a directory'.format(build_cache_path))
            return

        with self.profile_action(actions_constants['CREATE_BUILD_CACHE'], 'Create build cache'):
            build_cache_artifact_filename = '{}.squashfs'.format(build_cache_dirname)

            mksquashfs(input=str(build_cache_path), output=str(self.path(build_cache_artifact_filename)), options=['-keep-as-directory'])

            self.artifacts.create_ready_artifact_resource(
                relative_path=build_cache_artifact_filename,
                resource_type=SANDBOX_CI_BUILD_CACHE_IMAGE,
                **self.get_common_resource_attrs()
            )

    def start_declared_subtasks(self, created_subtasks):
        subtasks = self.meta.start_subtasks(created_subtasks)
        self.Parameters.is_subtasks_ready = True
        return subtasks

    def prepare_sources(self):
        with self.profile_action(actions_constants['CLONE_PROJECT'], 'Cloning project'):
            self.prepare_workcopy()

    def check_lfs(self):
        """
        Проверка LFS файлов сознательно реализована вне линтеров.

        Линтеры легко проигнорировать, а проблему с закомиченными бинарниками очень тяжело чинить,
        поэтому валим весь конвейер в самом начале.
        """
        if self.use_arc:
            logging.debug('In arc we do not have LFS files, skiping')
            return

        if self.skip_lfs_checkout:
            logging.debug('checking LFS files is not needed, skiping')
            return

        with self.profile_action(actions_constants['CHECK_LFS'], 'Check LFS binary'):
            try:
                self.scripts.run_bash_script('bash/check-lfs.sh')
            except Exception:
                raise TaskFailure(
                    'Yours changes contains binary files which must be committed with git-lfs, '
                    'see log for more details & docs https://nda.ya.ru/3TfREz',
                )

    def create_declared_subtasks(self):
        declared_subtasks = filter(bool, self.declare_subtasks())

        sandbox_tasks = filter(self.meta.is_sandbox_task, declared_subtasks)
        declared_tasks = filter(self.meta.is_task_declaration, declared_subtasks)

        if len(declared_tasks) == 0:
            return sandbox_tasks

        created_declared_subtasks = flow.parallel(self.meta.create_declared_subtask, declared_tasks)
        created_subtasks = map(lambda task: task.get('task'), created_declared_subtasks)

        virtual_subtasks_ids = map(lambda task: task.get('virtual_id'), created_declared_subtasks)
        real_subtasks_ids = self.meta.get_tasks_ids(created_subtasks)

        # Отношение виртуального идентификатора таски к реальному
        tasks_relations = dict(zip(virtual_subtasks_ids, real_subtasks_ids))

        logging.debug('Relationship between the virtual task ID and the real task ID: {relations}'.format(
            relations=tasks_relations,
        ))

        subtasks = sandbox_tasks + created_subtasks

        return flow.parallel(lambda task: self.set_real_tasks_ids(task, tasks_relations), subtasks)

    def set_real_tasks_ids(self, task, tasks_relations):
        custom_fields = []

        if task.Parameters.wait_tasks:
            real_wait_tasks_ids = self.get_real_wait_tasks_ids(task.Parameters.wait_tasks, tasks_relations)

            custom_fields = [{'name': 'wait_tasks', 'value': real_wait_tasks_ids}]
            task.Parameters.wait_tasks = real_wait_tasks_ids

        if task.Parameters.wait_output_parameters:
            real_wait_output_parameters = self.get_real_wait_outputs(task.Parameters.wait_output_parameters, tasks_relations)

            custom_fields.append({'name': 'wait_output_parameters', 'value': real_wait_output_parameters})
            task.Parameters.wait_output_parameters = real_wait_output_parameters

        if len(custom_fields) == 0:
            return task

        logging.debug('Set custom fields to {custom_fields}'.format(
            custom_fields=custom_fields,
        ))

        self.server.task[task.id].update({
            'custom_fields': custom_fields,
        })

        return task

    def get_real_wait_tasks_ids(self, wait_tasks, tasks_relations):
        """
        :param wait_tasks: список ожидаемых тасок
        :type wait_tasks: list of int
        :param tasks_relations: отношение виртуального идентификатора таски к реальному
        :type tasks_relations: dict
        :rtype: list of int
        """
        return map(lambda task_id: tasks_relations.get(task_id, task_id), wait_tasks)

    def get_real_wait_outputs(self, wait_output_parameters, tasks_relations):
        """
        :param wait_output_parameters: словарь ожидаемых параметров от тасок
        :type wait_output_parameters: dict
        :param tasks_relations: отношение виртуального идентификатора таски к реальному
        :type tasks_relations: dict
        :rtype: dict
        """
        parameters = {}

        for task_id, params in wait_output_parameters.iteritems():
            virtual_task_id = int(task_id)
            real_task_id = tasks_relations.get(virtual_task_id, virtual_task_id)
            parameters[str(real_task_id)] = params

        return parameters

    def configure(self):
        # Оторвать после закрытия FEI-10574
        default_template = '//yastatic.net/q/crowdtest/serp-static-nanny/static/sandbox-{project_name}-{tree_hash}/'

        if self.Parameters.static_url_template == default_template and self.config.is_enabled('deploy', 'static_s3'):
            static_url_template = '//serp-static-testing.s3.yandex.net/{project_name}/'
        else:
            static_url_template = self.Parameters.static_url_template

        self.Parameters.static_url = static_url_template.format(
            project_name=self.project_name,
            tree_hash=self.Parameters.project_tree_hash,
        )
        try:
            url_parts = urlparse(self.Parameters.static_url)
            logging.debug('Static URL parts: {}'.format(url_parts))
        except:
            raise TaskError('Could not parse static URL {}', self.Parameters.static_url)

        self.Parameters.static_origin = url_parts.netloc
        self.Parameters.static_path = url_parts.path

        if self.project_conf.get('set_static_url', True):
            self._set_default_env_variable('STATIC_HOST', self.Parameters.static_url)
            self._set_default_env_variable('STATIC_ORIGIN', self.Parameters.static_origin)
            self._set_default_env_variable('STATIC_PATH', self.Parameters.static_path)

        with self.profile_action(actions_constants['CONFIGURE'], 'Configuring'):
            self.lifecycle('configure')

    def _set_default_env_variable(self, name, value):
        logging.debug('Setting default env variable {name}={value}'.format(name=name, value=value))
        os.environ.setdefault(name, value)

    def deps(self):
        result = []

        if self.lifecycle.has_step('npm_install'):
            with self.profile_action(actions_constants['NPM_INSTALL'], 'Installing npm dependencies'):
                result.append(self.dependencies.npm_install())
        else:
            logging.info('Skip installing of npm dependencies because of npm_install lifecycle step absence')

        if self.lifecycle.has_step('bower_install'):
            with self.profile_action(actions_constants['BOWER_INSTALL'], 'Installing bower dependencies'):
                result.append(self.dependencies.bower_install())
        else:
            logging.info('Skip installing of bower dependencies because of bower_install lifecycle step absence')

        return result

    def build(self):
        with self.profile_action(actions_constants['BUILD'], 'Building'):
            self.lifecycle('build')

    def create_build_subtask(self, **params):
        """
        Создаёт задачу сборки.

        :return: Инстанс задачи сборки
        :rtype: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
        """
        # Не создавать задачи у подзадач, чтобы избежать зацикливания.
        if not self.parent:
            return self.meta.create_subtask(
                task_type=self.__class__,
                project_tree_hash=None,
                **params
            )

    def create_build_base_commit_subtask(self, **params):
        """
        Создаёт задачу сборки для базового коммита.

        :return: Инстанс задачи сборки
        :rtype: sandbox.projects.sandbox_ci.task.BaseBuildTask.BaseBuildTask
        """
        base_sha = self.Parameters.project_range_base_hash

        merge_refs = self.Parameters.project_git_merge_ref
        # self.Parameters.project_git_merge_ref[0] бывает равен self.Parameters.project_git_merge_ref[1]
        # в случае MQ Waiting project task
        uniq_merge_refs = set(self.Parameters.project_git_merge_ref)
        build_context = self.Parameters.project_build_context
        check_ref_status = "{}.check_ref_status".format(build_context)
        build_conf = self.Parameters.external_config or {}
        build_conf[check_ref_status] = False

        # В сборке для ПР-ов указывается один project_git_merge_ref, в сборку базового коммита его прокидывать не нужно
        # @see https://st.yandex-team.ru/INFRADUTY-17935#6113a1f2faf87d63dc03f7d0
        if len(merge_refs) == 1:
            merge_refs = []

        if len(uniq_merge_refs) > 1:
            # В случае кэшей, которые собираются для нескольких ПР-ов нужно исключить последний
            # сохраняя формат для N пулл-реквестов: [N, 1, 2, ..., N-1, N]
            # pull/38954, pull/39006, review/1516945, pull/38954 -> review/1516945, pull/39006, review/1516945
            merge_refs = merge_refs[-2:-1] + merge_refs[1:-1]

        return self.create_build_subtask(
            project_git_base_ref=self.Parameters.project_git_base_ref,
            project_git_base_commit=self.Parameters.project_git_base_commit or self.Parameters.project_base_hash,
            project_git_merge_ref=merge_refs,
            project_git_merge_commit=base_sha,
            use_arc=self.use_arc,
            arc_ref=self.Parameters.project_base_hash if self.use_arc else None,
            scripts_last_resource=self.Parameters.scripts_last_resource,
            scripts_resource=self.Parameters.scripts_resource,
            external_config=build_conf,
            **params
        )

    # Сраный костыль.
    # Pulse-задачи сравнивают два состояния кода base- и head-коммиты.
    # 1. Для сборки пулл-реквестов base — последний коммит в базовой ветке (обычно `dev` или `master`),
    #    head - последний коммит в ветке пулл-реквеста.
    # 2. Для сборки dev base — предыдущий коммит в базовой ветке (обычно `dev` или `master`),
    #    head - последний коммит в базовой ветке (обычно `dev` или `master`).
    # Вычисление base для dev-режима происходит с помощью `HEAD~1`.
    # Как вычислять base-коммит настраивается в конфиге `genisys`, в секции tests.pulse.base_templates.
    # Поиск предыдущего релиза был добавлен в FEI-11622.
    # TODO: сделать нормально.
    # @see FEI-6735, FEI-6412, FEI-6411
    def get_base_commit(self, hashes):
        """
        Возвращает базовый коммит.

        Важно: базовый коммит можно вычислять только после получения рабочей копии

        :return: базовый коммит
        :rtype: dict
        """
        base_type = self.project_conf.get('tests', {}).get('pulse', {}).get('base_template')

        logging.debug('Base commit type: {}'.format(base_type))

        merge_refs = hashes.get('mergeRefs', [])

        if self.use_arc and merge_refs:
            # Алгоритм определения базового коммита
            # * для одного merge_ref -> trunk, коммит на который накладываем патч.
            # * для нескольких merge_ref -> коммит полученный после наложения/ребейза предпоследнего {пулл,ревью}-реквеста.
            # @see FEI-17696
            return hashes['base'] if len(merge_refs) == 1 else merge_refs[-2]

        if base_type == 'dev':
            return {'hash': self.Parameters.project_base_hash, 'treeHash': self.Parameters.project_base_tree_hash}

        is_release = self.Parameters.is_release
        # Временно, нужно для попроектного включения пульса в релизном конвейере
        has_pulse_support_in_release = self.Parameters.project_github_repo in ('web4', 'turbo', 'fiji',)

        if is_release and has_pulse_support_in_release:
            return self.get_last_release_base_commit()

        get_hashes_params = {
            'commit': 'HEAD~1'
        }

        if self.use_arc:
            get_hashes_params['use-arc'] = True

        return self.scripts.run_js(
            'script/get-commit-hashes.js',
            get_hashes_params,
            log_prefix='determine_base_commit_sha',
            work_dir=self.project_dir,
        )

    @in_case_of('use_arc', 'get_last_arc_release_base_commit')
    def get_last_release_base_commit(self):
        """
        :return: возвращает коммит и трихэш последнего релиза
        :rtype: dict of str:str
        """
        owner = self.Parameters.project_github_owner
        repo = self.Parameters.project_github_repo
        ref = self.Parameters.project_git_base_ref

        with Debug('serp:*,github-scripts:*'):
            base_commit = self.scripts.run_js(
                'script/get-last-release-hashes.js',
                ref,
                {'owner': owner, 'repo': repo},
                log_prefix='determine_last_release_base_commit_sha',
                work_dir=self.project_dir,
            )

            logging.debug('Last release commit before "{ref}" in {owner}/{repo}: {commit}'.format(
                owner=owner,
                repo=repo,
                ref=ref,
                commit=base_commit,
            ))

            return base_commit

    def get_last_arc_release_base_commit(self):
        ref = self.Parameters.project_git_base_ref

        with Debug('*'):
            base_tag = self.scripts.run_js(
                'script/release/arc/release-tag-search.js',
                'previous',
                {'ref': ref},
                log_prefix='determine_last_release_base_commit_sha',
                work_dir=self.project_dir,
            )

            logging.debug('Last release commit before "{ref}": {tag}'.format(
                ref=ref,
                tag=base_tag,
            ))

            # Трихэш для коммита мы вычисляем кустарным способом, прогоном выхлопа arc ls-tree в md5.
            # Через API Арка сложно аналогичным образом получить трихэш. Получаем его из SB таски по
            # хэшу коммита, мы записываем трихэш коммита в выходной параметр `project_tree_hash`.
            release_task = sdk2.Task.find(
                type=self.type.name,
                status=ctt.Status.RELEASED,
                release='stable',
                input_parameters=dict(
                    project_git_base_commit=base_tag["hash"],
                ),
            ).order(-sdk2.Task.id).first()

            logging.debug('Found tasks: {}'.format(release_task))

            if not release_task:
                logging.debug('Could not find released task, returning tag info as-is')
                return base_tag

            return dict(
                hash=release_task.Parameters.project_git_base_commit,
                treeHash=release_task.Parameters.project_tree_hash,
            )

    def get_common_resource_attrs(self):
        """
        Параметры для идентификации ресурса, нужно для поиска кеша

        :rtype: dict of str:str
        """
        params = {
            'repo': self.Parameters.project_github_repo,
            'owner': self.Parameters.project_github_owner,
            'project': self.project_name,
            'project_tree_hash': self.Parameters.project_tree_hash,
            'YENV': os.environ.get('YENV'),
            'environ': self.get_environ_resource_attr()
        }

        # В генезисе YCONFIG указан не везде
        if 'YCONFIG' in os.environ:
            params['YCONFIG'] = os.environ.get('YCONFIG')

        logging.debug('got common resource attrs: {}'.format(params))

        return params

    def get_environ_resource_attr(self):
        """
        Формирует параметр переменных окружения для идентификации ресурса

        :rtype: str
        """
        # Список переменных окружения из genisys, которые могут влиять на сборку.
        # TODO: удалить после FEI-8541
        genisys_environ = ['YENV', 'YCONFIG', 'CONDUCTOR_BRANCH', 'SERP_VERSION_SUFFIX', 'STATIC_HOST']
        user_environ = self.Parameters.environ.keys()
        env_names = list(set(genisys_environ + user_environ))

        envs = []
        for env_name in sorted(env_names):
            env_val = os.environ.get(env_name)
            if env_val is not None:
                envs.append('{}={}'.format(env_name, env_val))

        return ' '.join(envs)

    def get_cached_artifact_resources(self):
        """
        :return: возвращает список закэшированных ресурсов
        :rtype: dict of str:sdk2.Resource
        """
        cached_resources = {}
        common_resource_attrs = self.get_common_resource_attrs()

        for artifact_type in self.artifact_types:
            artifact_res = self.artifacts.get_last_artifact_resource(
                resource_type=self.artifacts_resource_types.get(artifact_type, self.artifacts_resource_type_default),
                type=artifact_type,
                **common_resource_attrs
            )
            if artifact_res is None:
                return {}
            cached_resources[artifact_type] = artifact_res
        return cached_resources

    def create_artifact_resources(self):
        """
        :return: возвращает список зарегистрированных ресурсов
        :rtype: dict of str:sdk2.Resource
        """
        common_resource_attrs = self.get_common_resource_attrs()

        return {
            artifact['type']: self.artifacts.create_artifact_resource(
                relative_path=artifact['relative_path'],
                resource_type=self.artifacts_resource_types.get(artifact['type'], self.artifacts_resource_type_default),
                type=artifact['type'],
                ref=self.ref,
                **common_resource_attrs
            )
            for artifacts in self.__normalized_lifecycle_artifacts_map.itervalues()
            for artifact in artifacts
        }

    def republish_resources(self, resources):
        """
        :param resources:
        :type resources: dict of str:sdk2.Resource
        :return: возвращает список склонированных ресурсов
        :rtype: dict of str:sdk2.Resource
        """
        # синхронизация ресурсов
        for resource in resources.itervalues():
            from_path = sdk2.ResourceData(resource).path
            dest_path = self.path(from_path.name)
            logging.debug('Syncing {resource} from {from_path} to {dest_path}'.format(
                resource=resource,
                from_path=from_path,
                dest_path=dest_path,
            ))
            shutil.copy(str(from_path), str(dest_path))

        # регистрируем скопированные ресурсы
        cloned_resources = self.create_artifact_resources()
        logging.debug('Cloned resources: {resources}'.format(resources=cloned_resources))

        # публикация ресурсов
        for resource in cloned_resources.itervalues():
            sdk2.ResourceData(resource).ready()

        return cloned_resources

    def register_artifacts(self):
        """
        Регистрация ресурсов. Перепубликация ресурсов, если они брались из кэша (@see FEI-8067).

        :return: возвращает список зарегистрированных ресурсов
        :rtype: tuple of (dict of str:sdk2.Resource, sandbox.projects.sandbox_ci.managers.artifacts ArtifactCacheStatus)
        """
        with self.profile_action(actions_constants['REGISTER_ARTIFACTS'], 'Registering Sandbox resources'):
            if not self.Parameters.reuse_artifacts_cache:
                logging.debug('Forced to ignore cached, creating new resources')
                return self.create_artifact_resources(), ArtifactCacheStatus.IGNORED

            cached_resources = self.get_cached_artifact_resources()

            if cached_resources:
                logging.debug('Cached resources: {resources} to reuse'.format(resources=cached_resources))
                self.republish_resources(cached_resources)
                return cached_resources, ArtifactCacheStatus.FOUND

            logging.debug('Cache is not found, creating resources')
            return self.create_artifact_resources(), ArtifactCacheStatus.NOT_FOUND

    def get_registered_artifact_resource(self, artifact_type):
        artifact_res = self.artifacts_resources.get(artifact_type)
        if not artifact_res:
            raise Exception('Artifact with type {} is not registered'.format(artifact_type))
        return artifact_res

    def get_registered_artifact(self, artifact_type):
        return self.get_registered_artifact_resource(artifact_type)

    def get_registered_artifact_id(self, artifact_type):
        return self.get_registered_artifact(artifact_type).id

    def create_artifacts(self, lifecycle_step=None):
        action_name = actions_constants['ARTIFACTS']
        action_description = 'Creating build artifacts'

        if lifecycle_step:
            if lifecycle_step not in self.__normalized_lifecycle_artifacts_map.keys():
                raise Exception('No such lifecycle step "{}"'.format(lifecycle_step))

            artifacts_lifecycle_steps = [lifecycle_step]
            action_name += '_' + lifecycle_step
            action_description += ' for step ' + lifecycle_step
        else:
            artifacts_lifecycle_steps = self.__normalized_lifecycle_artifacts_map.keys()

        with self.profile_action(action_name, action_description):
            processes = dict([
                (step, self.lifecycle(step, wait=False))
                for step in artifacts_lifecycle_steps
            ])
            for step in artifacts_lifecycle_steps:
                if processes[step].wait():
                    raise Exception('Error building artifacts using lifecycle script: {}'.format(step))
                flow.parallel(self.publish_artifact, map(lambda i: i['type'], self.__normalized_lifecycle_artifacts_map[step]))

    def move_artifact(self, artifacts_type):
        artifact_filename = self.get_registered_artifact_resource(artifacts_type).path
        src_path = self.ramdrive_path(artifact_filename)
        dst_path = self.path(artifact_filename)
        shutil.move(str(src_path), str(dst_path))

    def publish_artifact(self, artifact_type):
        # moving artifacts from ramdrive to the task directory
        if self.ramdrive:
            self.move_artifact(artifact_type)
        # publishing artifacts
        sdk2.ResourceData(self.get_registered_artifact(artifact_type)).ready()

        self.cache_profiler.miss(artifact_type, 'resource')

    def declare_subtasks(self):
        return []

    def on_before_end(self, status):
        if self.Parameters.is_artifacts_ready is None:
            self.Parameters.is_artifacts_ready = False

        if self.Parameters.is_subtasks_ready is None:
            self.Parameters.is_subtasks_ready = False

        self.Context.github_statuses = self.meta.get_github_statuses()
        super(BaseBuildTask, self).on_before_end(status)

    def on_finish(self, prev_status, status):
        self.info.report_info(
            'Skipped steps',
            self.Context.skipped_steps,
            lambda details: '{label} - {reason}'.format(**details),
        )

        super(BaseBuildTask, self).on_finish(prev_status, status)

    def send_deep_profiler_report(self):
        need_upload_profile_actions = self.project_conf.get('stat', {}).get('deep_actions_profile', False)
        if not need_upload_profile_actions:
            logging.debug('Skip uploading deep profile actions')
            return

        try:
            actions_stat_sender = ActionsStatSender(self)

            actions_stat_sender.execute()
        except Exception:
            logging.exception('Failed to upload deep profile data')
            error_traceback = traceback.format_exc()
            self.Context.noncritical_errors.append(error_traceback)

    def prepare_build_environ(self):
        """
        Подготовка переменных окружения необходимых для сборки проекта.
        Сейчас в input-параметры "project_github_commit" и "project_git_base_ref"/"project_git_merge_ref"
        в случае сборки на ARC записываются его commit-sha и номер ревью-реквеста, ветки пока не собираются.
        """
        vcs_commit_hash = getattr(self.Parameters, 'project_github_commit', None)
        vcs_ref = getattr(self.Parameters, 'project_git_base_ref', None)
        vcs_merge_ref = getattr(self.Parameters, 'project_git_merge_ref', None)

        if vcs_commit_hash:
            os.environ['BUILD_VCS_COMMIT'] = vcs_commit_hash

        # В случае запуск кешей MQ здесь будет несколько sha коммитов.
        # Для запусков MQ эта переменная не нужна.
        if vcs_merge_ref and len(vcs_merge_ref) == 1:
            os.environ['BUILD_VCS_REF'] = vcs_merge_ref[0]
        elif vcs_ref:
            os.environ['BUILD_VCS_REF'] = vcs_ref
