# -*- coding: utf-8 -*-

import logging
import shutil

from sandbox import common
from sandbox import sdk2
from contextlib import nested
from sandbox.common.types import task as ctt

from sandbox.common.utils import classproperty
from sandbox.common.errors import TaskFailure

from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.managers import LifecycleManager
from sandbox.projects.sandbox_ci.task import BaseBuildTask
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants


class BaseLintersTask(BaseBuildTask):
    """
        Таск для запуска линтеров в пулл-реквесте.
    """

    class Requirements(BaseBuildTask.Requirements):
        disk_space = 25 * 1024
        kill_timeout = 1800

    linters = {}

    reports = {}

    # Cкрипт для клонирования проекта подмердживает ветку пулл-реквеста к деву
    # git rev-parse HEAD~1 возвращает предыдущий **merge** коммит, а не предыдущий коммит из пулл-реквеста.
    #
    #           A   B   C
    #       PR  *---*---*
    #          /         \
    #  dev -->*-----------* merge коммит
    #         ^
    #         └── переключиться сюда и перенести файлы из коммитов A, B, и C в stage
    #
    lifecycle_steps = {
        'npm_install': 'npm ci',
        'stage_git_changes': 'git reset --soft $(git rev-parse HEAD~1) && git status',
        'stage_arc_changes': 'arc reset --soft $(arc merge-base trunk HEAD) && arc status ./',
    }

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Линтеры'

    @classproperty
    def task_label(self):
        return 'linters'

    @property
    @common.utils.singleton
    def lint(self):
        return LifecycleManager(self, self.linters, self.project_dir)

    @property
    def skip_lfs_checkout(self):
        return True

    def execute(self):
        with nested(*self.build_execute_contexts()):
            self.prepare_sources()
            self.deps()
            self.stage_changes()
            self.run_linters()

    def stage_changes(self):
        """
            Переносит изменённые файлы из пулл-реквеста в stage
        """
        with self.profile_action(actions_constants['STAGE_CHANGES'], 'Staging changed files'):
            vcs = 'arc' if self.use_arc else 'git'
            stage_changes_step = 'stage_{vcs}_changes'.format(vcs=vcs)

            self.lifecycle(stage_changes_step)

    def run_linters(self):
        with self.profile_action(actions_constants['LINTING'], 'Running linters'):
            failed_linters = []
            for linter in self.linters.iterkeys():
                logging.debug('Running {}'.format(linter))
                process = self.lint(linter, wait=False, check=False, log_prefix=linter)
                is_failed = bool(process.wait())

                if is_failed:
                    failed_linters.append(linter)

                report_status = ctt.Status.FAILURE if is_failed else ctt.Status.SUCCESS

                logging.debug('Linter {linter} status: {status}'.format(
                    linter=linter,
                    status=report_status,
                ))

                if linter in self.reports:
                    self.publish_linter_report(linter, report_status)
                else:
                    self.publish_linter_log(process.stdout_path_filename, process.stdout_path, report_status)

        if failed_linters:
            raise TaskFailure('{} failed, see reports'.format(', '.join(failed_linters)))

    def publish_linter_log(self, filename, path, status):
        """
            Прикрепляет лог линтеров к футеру задачи
        """
        # Копируем лог в корневую директорию

        logging.debug('Copying {src} to {dest}'.format(
            src=path,
            dest=self.path(filename),
        ))

        shutil.copy(str(path), str(self.path(filename)))

        self.add_report_resource(filename, status)

    def publish_linter_report(self, linter, status):
        """
            Прикрепляет отчёты линтеров к футеру задачи
        """
        reports = self.reports[linter]

        for report_file in reports if isinstance(reports, list) else [reports]:
            logging.debug('Linter {linter} report: "{report}"'.format(
                linter=linter,
                report=report_file,
            ))

            report_file_path = self.project_path(report_file)

            if not report_file_path.exists():
                logging.debug('Linter {linter} report "{report}" does not exist'.format(
                    linter=linter,
                    report=report_file_path,
                ))
                continue

            # Перенесем репорт из ramdrive/web4/{report-file} в runtime-data/.../{report-file}

            logging.debug('Moving {src} to {dest}'.format(
                src=report_file_path,
                dest=self.path(report_file),
            ))

            shutil.move(str(report_file_path), str(self.path(report_file)))

            self.add_report_resource(report_file, status)

    def add_report_resource(self, filename, status):
        logging.debug('Creating resource from {filename} with status {status}'.format(
            filename=filename,
            status=status,
        ))

        resource = self.artifacts.create_project_artifact_resource(
            status=status,
            type=filename,
            root_path='',
            relative_path=filename,
            resource_type=SANDBOX_CI_ARTIFACT,
            public=True,
        )

        sdk2.ResourceData(resource).ready()
        self.Context.report_resources.append(resource.id)
