# -*- coding: utf-8 -*-

import json
import os
import logging
import time

from sandbox import sdk2

from sandbox.projects.common import solomon

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import process
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.task import OverlayfsMixin, PrepareWorkingCopyMixin, BaseTask
from sandbox.projects.sandbox_ci.utils.context import Debug, GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.types import misc as ctm

TESTPALM_CLI_PACKAGE = '@yandex-int/si.ci.testpalm-cli@3.0'

SOLOMON_PUSH_COMMON_PARAMETERS = {
    'project': 'fei',
    'cluster': 'sandbox',
    'service': 'palmsync',
}

RAMDRIVE_SIZE = 30 * 1024


class BaseManualTestRunTask(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):

    class Requirements(BaseTask.Requirements):
        disk_space = 35 * 1024
        cores = 1
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)

        class Caches(BaseTask.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()
        ref = sdk2.parameters.String('Build Branch')
        palmsync_base_url = sdk2.parameters.String(u'URL для синхронизации проекта в TestPalm (baseUrl)')
        custom_opts = sdk2.parameters.String(u'Опции, которые будут добавлены при запуске palmsync')
        project_suffix = sdk2.parameters.String(u'Суффикс, добавляемый к идентификатору проекта в TestPalm')
        skip_sync = sdk2.parameters.Bool(u'Пропустить синхронизацию проекта (debug only)', default=False)

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'sync': 'node --max-old-space-size=4096 $(npm bin)/palmsync synchronize -p {clone_project_id} {run_custom_opts}',
    }

    @property
    def skip_lfs_checkout(self):
        return self.use_arc or not self.use_overlayfs

    @property
    def needed_artifact_types(self):
        raise Exception('needed_artifact_types should be defined in BaseManualTestRunTask descendants')

    @property
    def lifecycle(self):
        return super(BaseManualTestRunTask, self).lifecycle.update_vars(
            run_custom_opts=self.Parameters.custom_opts,
            clone_project_id=self.testpalm_project,
        )

    @property
    def testpalm_project(self):
        return '{}-{}'.format(self.testpalm_project_id, self.Parameters.project_suffix)

    @staticmethod
    def format_github_context():
        return u'[Sandbox CI] Синхронизация ручного прогона тестов'

    @property
    def github_context(self):
        return self.format_github_context()

    @property
    def task_report_url(self):
        return 'https://testpalm.yandex-team.ru/{}/testruns'.format(self.testpalm_project)

    @property
    def task_report_custom_text(self):
        lines = [
            u'\n',
            u'Список ранов:',
            u'- (({testruns_url}?tags=searchapp&tags=splitview Поисковые приложения и сплитвью))',
            u'- (({testruns_url}?all_tags=explorer Исследовательские раны))',
            u'- (({testruns_url}?all_tags=no_autotest Неавтоматизированные проверки))',
            u'\n',
        ]

        return '\n'.join(lines).format(testruns_url=self.task_report_url)

    @sdk2.header()
    def header(self):
        return u'<h4><a href="{}" target="_blank">Список ранов</a></h4>'.format(self.task_report_url)

    def on_save(self):
        super(BaseManualTestRunTask, self).on_save()

        setattr(self.Context, '__do_not_dump_ramdrive', True)  # use setattr to avoid mangling

        self.Parameters.notifications = map(
            lambda rule: rule if ctt.Status.FAILURE in rule.statuses else sdk2.Notification(
                rule.statuses + [ctt.Status.FAILURE],
                rule.recipients,
                rule.transport
            ),
            self.Parameters.notifications
        )

        # @see FEI-12796
        if not self.is_release:
            self.set_semaphore('SANDBOX_CI_MANUAL_TEST_RUNS_LIMITER', 1)

        if self.use_git_in_overlayfs_mode:
            self.set_ramdrive_size(60 * 1024)

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        self.prepare()

        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self._install_dependencies()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version):
            self.clone()
            self.make_test_runs()

    def execute_in_overlayfs_mode(self):
        with self.prepare_working_copy_context():
            self.prepare()
            with Node(self.Parameters.node_js_version), self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                self.clone()
                self.make_test_runs()

    def prepare(self):
        self.set_environments()

    def set_environments(self):
        os.environ['YANDEX_INT_RESOURCES_PATH'] = str(self.working_path('.yandex-int'))
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)
        os.environ['NODE_EXTRA_CA_CERTS'] = '/etc/ssl/certs/YandexInternalRootCA.pem'
        os.environ['NPM_CONFIG_REGISTRY'] = 'http://npm.yandex-team.ru'
        os.environ['NPM_CONFIG_USER_AGENT'] = 'npm/6.2.0 (verdaccio yandex canary)'
        os.environ['TESTPALM_OAUTH_TOKEN'] = self.vault.read('env.TESTPALM_OAUTH_TOKEN')

    def clone(self):
        source_project = self.testpalm_project_id
        target_project = self.testpalm_project

        command = [
            'npx',
            TESTPALM_CLI_PACKAGE,
            'clone',
            source_project,
            target_project
        ]

        with self.profile_action(actions_constants['CLONE_PROJECT'], 'Clone TestPalm project'):
            with Debug('*'), Node(self.Parameters.node_js_version):
                process.run_process(
                    command,
                    work_dir=str(self.project_dir),
                    log_prefix='testpalm_cli_clone',
                    shell=True,
                )

    def make_test_runs(self):
        if not self.Parameters.skip_sync:
            self.sync()

    def sync(self):
        if self.Parameters.palmsync_base_url:
            os.environ['palmsync_baseUrl'] = self.Parameters.palmsync_base_url

        with self.profile_action(actions_constants['SYNC'], 'Running palmsync synchronize'), Debug('testpalm-api', depth=Debug.INFINITY):
            if self.lifecycle('sync', wait=False).wait():
                raise TaskFailure('Synchronize error, check sync.out.txt')

            self.check_and_report_errors(self.testpalm_project)

    def on_before_end(self, status):
        super(BaseManualTestRunTask, self).on_before_end(status)

        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(
                issue_key,
                status,
                task_report_name=u'Все тест-раны',
                task_report_url=self.task_report_url,
                custom_text=self.task_report_custom_text,
            )

    def check_and_report_errors(self, testpalm_project):
        errors_count = 0

        try:
            testcases = self.fetch_test_cases(testpalm_project)
            errors_count = self.calc_testcase_errors(testcases)
        finally:
            logging.debug('found errors: {}'.format(errors_count))

            if errors_count > 0:
                self._send_solomon_sensors([{
                    'labels': {'sensor': 'palmsync_synchronize_{}'.format(testpalm_project)},
                    'kind': 'IGAUGE',
                    'value': errors_count
                }])

    def fetch_test_cases(self, project):
        cmd = [
            'npx',
            TESTPALM_CLI_PACKAGE,
            'fetch-test-cases',
            project,
            '--apply-definitions',
            '--include=attributes',
        ]

        p = process.run_process(
            cmd,
            log_prefix='testpalm_testcases',
            work_dir=str(self.project_dir),
            outputs_to_one_file=False,
        )

        with open(p.stdout_path, 'r') as out_file:
            out = out_file.read()

        return json.loads(out.strip())

    def calc_testcase_errors(self, testcases):
        errors_count = 0
        for testcase in testcases:
            errors_count += len(testcase.get('attributes').get('palmsync_synchronization_errors', []))

        return errors_count

    def _send_solomon_sensors(self, sensors):
        try:
            solomon.push_to_solomon_v2(
                token=self.vault.read('env.SOLOMON_OAUTH_TOKEN'),
                params=SOLOMON_PUSH_COMMON_PARAMETERS,
                sensors=sensors
            )
            logging.debug('Sensors pushed: {}'.format(sensors))
        except Exception as error:
            logging.exception('Cannot send sensors ({sensors}): {error}'.format(
                sensors=sensors,
                error=error
            ))
