# -*- coding: utf-8 -*-

import json
import logging
import re

from sandbox import sdk2

from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property

from sandbox.projects.sandbox_ci import managers
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.utils import prioritizer


class BasePullRequestRunnerTask(sdk2.Task):
    """Запуск проверок из ПР с флагами"""

    class Parameters(sdk2.Parameters):
        tools = sdk2.parameters.CheckGroup(
            'Tools',
            choices=(
                ('Hermione', 'hermione'),
                ('Hermione e2e', 'hermione-e2e'),
            ),
            default=['hermione', 'hermione-e2e'],
            required=True
        )
        platforms = sdk2.parameters.CheckGroup('Platforms', choices=(), required=True)

        pull_request_ref = sdk2.parameters.String(
            'Pull request ref',
            description='Ссылка на ваш пулл-реквест в формате pull/xxx для github и review/xxxxxx для arcanum',
            default='',
            required=True
        )

        with sdk2.parameters.Group('Флаги') as flags_block:
            use_json_flags = sdk2.parameters.Bool(
                'Use JSON to describe flags',
                default=False,
                sub_fields={
                    'true': ['json_flags'],
                    'false': ['flags']
                }
            )
            flags = sdk2.parameters.Dict('Flags')
            json_flags = sdk2.parameters.JSON('JSON with flags', default={})

    @singleton_property
    def artifacts(self):
        return managers.ArtifactsManager(self)

    @singleton_property
    def meta(self):
        return managers.MetaTaskManager(self)

    @singleton_property
    def task_reports(self):
        return managers.Reports(self)

    _build_task = None

    @sdk2.header()
    def header(self):
        if self.Context.experiments_subtask_id:
            report_ids = self._get_report_ids()
            return self.task_reports.reports_artifacts(report_ids)

        return 'Проверки еще не запущены'

    def _get_report_ids(self):
        report_ids = []
        for tool in self.Parameters.tools:
            task_type = self.experiments_subtask_type._get_task_type(tool)
            report_type = '{}-report'.format(tool)

            subtasks = self.experiments_subtask.find(task_type)
            for subtask in subtasks:
                report = self._find_report(subtask, report_type)
                if report:
                    report_ids.append(report.id)

        return report_ids

    def _find_report(self, task, report_type, **custom_attrs):
        return SANDBOX_CI_ARTIFACT.find(
            task=task,
            attrs=dict(type=report_type, **custom_attrs)
        ).first()

    def on_save(self):
        super(BasePullRequestRunnerTask, self).on_save()
        self.Parameters.priority = prioritizer.get_priority(self)

    def on_execute(self):
        with self.memoize_stage.create_subtask:
            self.check_parameters()
            build_task = self.get_build_task()
            tools_params = self.get_tools_params()

            self.experiments_subtask = self.create_experiments_subtask(build_task, **tools_params)

            raise sdk2.WaitTask(
                tasks=self.meta.start_subtasks([self.experiments_subtask]),
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            )

        with self.memoize_stage.parse_results:
            if self.experiments_subtask.status != ctt.Status.SUCCESS:
                raise TaskFailure('Child task did not finish with success.')

    def check_parameters(self):
        if not re.match(r'^(pull|review)\/\d+$', self.Parameters.pull_request_ref):
            raise TaskFailure('Parameter "pull_request_ref" must be in the format "pull/xxx" or "review/xxxxxx"')

    @property
    def flags(self):
        if self.Parameters.use_json_flags:
            return self.Parameters.json_flags

        return self.Parameters.flags

    def get_build_task(self):
        if not self._build_task:
            self._build_task = self.find_build_task()

        return self._build_task

    def find_build_task(self, **input_params):
        task_search_parameters = dict(
            type=self.build_task_type,
            status=ctt.Status.Group.FINISH,
            input_parameters=dict(
                project_git_merge_ref=[self.Parameters.pull_request_ref],
                project_build_context='pull-request',
                **input_params
            ),
        )

        logging.debug('Trying to find task with the following parameters: {}'.format(task_search_parameters))

        task = sdk2.Task.find(**task_search_parameters).first()

        if task:
            logging.debug('The latest build for {pr_ref} was in task #{id}.'.format(
                pr_ref=self.Parameters.pull_request_ref,
                id=task.id,
            ))
            return task

        user_message = "\n".join([
            "Could not find the latest finished build for the {pr_ref} with the '{task_type}' task type.".format(
                pr_ref=self.Parameters.pull_request_ref,
                task_type=self.build_task_type,
            ),
            'Contact support if the service has at least one finished build for the pull-request.'
        ])

        self.set_info(user_message)

        raise TaskFailure('Could not find a task for the specified pull-request build.')

    @property
    def build_task_type(self):
        raise NotImplementedError

    def create_experiments_subtask(self, build_task, **custom_task_params):
        return self.meta.create_subtask(
            task_type=self.experiments_subtask_type,

            resources_yenv='testing',

            project_github_owner=build_task.Parameters.project_github_owner,
            project_github_repo=build_task.Parameters.project_github_repo,
            project_build_context='pull-request',

            tools=self.Parameters.tools,
            platforms=self.Parameters.platforms,

            project_tree_hash_for_templates=build_task.Parameters.project_tree_hash,
            project_git_base_ref=build_task.Parameters.project_git_merge_ref[0],

            beta_domain=self.beta_domain,

            git_checkout_params=build_task.Parameters.git_checkout_params,

            **custom_task_params
        )

    @property
    def beta_domain(self):
        return 'https://renderer-{project}-{suffix}.hamster.yandex.ru/'.format(
            project=self.project_name,
            suffix=self.get_build_task().Parameters.beta_suffix,
        )

    @property
    def experiments_subtask_type(self):
        raise NotImplementedError

    def get_tools_params(self):
        params = {}

        for tool in self.Parameters.tools:
            params.update(self.get_params_for_tool(tool))

        return params

    def get_params_for_tool(self, tool):
        if tool == 'hermione':
            return dict(
                hermione_env=self.get_flags_env(),
            )
        
        if tool == 'hermione-e2e':
            return dict(
                hermionee2e_env=self.get_flags_env(),
                hermionee2e_base_url=self.beta_domain,
            )

        return {}

    def get_flags_env(self):
        return dict(
            ARCHON_KOTIK_TEMPLATE_FLAG=json.dumps(self.flags),
        )

    @property
    def formatted_flags(self):
        return ['{}={}'.format(k, v) for k, v in self.flags.iteritems()]

    @property
    def project_name(self):
        raise NotImplementedError

    @property
    def experiments_subtask(self):
        return self.experiments_subtask_type[self.Context.experiments_subtask_id]

    @experiments_subtask.setter
    def experiments_subtask(self, subtask):
        self.Context.experiments_subtask_id = subtask.id
