# coding=utf-8
import socket
import traceback
import logging
import re
from requests.exceptions import HTTPError, ConnectionError

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_task_link, singleton_property

from sandbox.projects.sandbox_ci.task import BaseTask
from sandbox.projects.sandbox_ci.pulse.config import load_pulse_genisys_config
from sandbox.projects.sandbox_ci.pulse import parameters, notification
from sandbox.projects.sandbox_ci.pulse.velocity_audit import fetch_velocity_audit_issues
from sandbox.projects.sandbox_ci.parameters import send_comment_to_issue


class BasePulseTask(BaseTask):
    class Parameters(BaseTask.Parameters):
        with sdk2.parameters.Group('General params') as general_params:
            project = parameters.project()

            project_tree_hash = sdk2.parameters.String(
                'Source tree hash',
                required=False,
                default='',
            )

            project_vcs_ref = sdk2.parameters.String(
                'Project VCS ref',
                required=False,
                default='',
            )

            pr_number = sdk2.parameters.String(
                'Pull request number',
                required=False,
                default=None,
            )

        with sdk2.parameters.Group('Fail & report params') as fail_params_block:
            fail_on_limits_exceed = parameters.fail_on_limits_exceed()
            send_email_on_limits_exceed = parameters.send_email_on_limits_exceed()
            should_report_to_stat = parameters.should_report_to_stat()

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = send_comment_to_issue()

    class Context(BaseTask.Context):
        ctx_fail_on_limits_exceed = None
        ctx_send_email_on_limits_exceed = None

    @property
    def project_name(self):
        return self.Parameters.project

    @property
    def fail_on_limits_exceed(self):
        value = self.Parameters.fail_on_limits_exceed
        if self.Context.ctx_fail_on_limits_exceed is not None:
            value = self.Context.ctx_fail_on_limits_exceed

        logging.debug('fail_on_limits_exceed value: {}'.format(value))

        return value

    @property
    def send_email_on_limits_exceed(self):
        value = self.Parameters.send_email_on_limits_exceed
        if self.Context.ctx_send_email_on_limits_exceed is not None:
            value = self.Context.ctx_send_email_on_limits_exceed

        logging.debug('send_email_on_limits_exceed value: {}'.format(value))

        return value

    @property
    def email_to_notify(self):
        if self.project_name == 'fiji':
            return notification.DEFAULT_NOTIFY_FIJI_TO_EMAIL
        return notification.DEFAULT_NOTIFY_TO_EMAIL

    @property
    def github_context(self):
        return 'base-pulse-task'

    @property
    def should_report_to_stat(self):
        return self.Parameters.should_report_to_stat

    @singleton_property
    def pulse_config(self, subsection):
        return load_pulse_genisys_config(subsection)

    def check_exceeded_limits(self):
        exceeded_limits = self.get_limits_excesses()

        if len(exceeded_limits):
            logging.debug('limits exceeded')

            self.set_info('\n'.join(exceeded_limits))

            if self.send_email_on_limits_exceed and self.email_to_notify:
                self._send_email_notification()

            if self.fail_on_limits_exceed:
                raise TaskFailure(u'Limits exceeded')

    @sdk2.header()
    def header(self):
        if self.Context.reused_same_task:
            return u'Задача была реиспользованна. Результаты можно посмотреть в реиспользованной задаче, ссылка ниже.'

        return self._report

    def get_limits_excesses(self):
        """
        :return: Массив строк которые описывают какие лимиты превышены и на сколько
        :rtype: list of str
        """
        raise NotImplementedError()

    def _send_statface_report_safe(self):
        # noinspection PyUnresolvedReferences
        from statface_client import StatfaceClientError

        try:
            self._send_statface_report()
        except (StatfaceClientError, HTTPError, ConnectionError, socket.error) as e:
            error_traceback = traceback.format_exc()
            logging.error('Statface Upload Error: %s', str(e))
            logging.error(error_traceback)

    def _send_statface_report(self):
        raise NotImplementedError()

    def _send_email_notification(self):
        logging.debug('sending email notification to {}'.format(self.email_to_notify))

        pr_description = self.Parameters.description
        pr_match = re.match(u'<a target=\'_blank\' href=\'(.+pull.+)\'>(.+#\d+)</a>', self.Parameters.description)
        if pr_match is not None:
            pr_description = u'Пулл-реквест: {}'.format(pr_match.group(1))

        exception_text = u"""Ссылка на упавший таск: {}
{}""".format(
            get_task_link(self.id),
            pr_description
        )

        notification.send_pulse_email(
            self,
            email=self.email_to_notify,
            title=u'[{}] Превышение лимитов в задаче #{} {}'.format(self.project_name, self.id, self.type),
            text=exception_text
        )

    def on_before_end(self, status):
        super(BasePulseTask, self).on_before_end(status)

        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)

    def on_prepare(self):
        super(BasePulseTask, self).on_prepare()

        self.set_velocity_audit_approve_params()

    def set_velocity_audit_approve_params(self):
        with self.memoize_stage.velocity_audit():
            issue_keys = self.fetch_velocity_audit_issues(
                owner=self.Parameters.project_github_owner,
                repo=self.Parameters.project_github_repo,
                ref=self.Parameters.project_vcs_ref,
            )

            logging.debug('Velocity audit issues: {}'.format(issue_keys))

            if issue_keys:
                logging.debug('Updating context values')
                self.Context.ctx_fail_on_limits_exceed = False
                self.Context.save()

    def fetch_velocity_audit_issues(self, owner, repo, ref):
        with self.profiler.actions.velocity_audit('Fetching audit tickets'):
            try:
                startrek_token = self.vault.read('robot-serp-bot_startrek_token')

                return fetch_velocity_audit_issues(startrek_token, owner, repo, ref)
            except Exception as e:
                logging.error('Error: %s', str(e))
