# -*- coding: utf-8 -*-
import shutil
import logging
import traceback
import datetime
import time
import os

from pprint import pformat
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.common.utils import singleton_property, get_task_link
from sandbox.common.types import misc as ctm, task as ctt, notification as ctn

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common.yasm import push_api

from sandbox.projects.sandbox_ci.utils import env, prioritizer, dict_utils
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils.github import GitHubStatus
from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.constants import MailingLists
from sandbox.projects.sandbox_ci.managers.arc.arc_cli import prefetch_files


class BaseTask(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL
        disk_space = 20 * 1024  # 20 Gb should be enough for everyone
        environments = (
            PipEnvironment('python-statface-client', custom_parameters=["requests==2.18.4"]),
        )

    class Parameters(parameters.CommonParameters):
        dump_disk_usage = False

        project_github_repository = parameters.ProjectGitHubRepositoryParameters

        with sdk2.parameters.Group('Project build') as project_build_block:
            build_id = sdk2.parameters.Integer(
                'Build ID',
                hint=True,
                default=0,
            )
            project_build_context = parameters.project_build_context()
            build_platform = parameters.build_platform()
            static_url_template = sdk2.parameters.String(
                'Static URL template',
                description='URL статики, который будет использоваться в шаблонах при сборке в тестинг. '
                            'Значение динамическое — можно использовать {project_name} и {tree_hash}.',
                default='//yastatic.net/q/crowdtest/serp-static-nanny/static/sandbox-{project_name}-{tree_hash}/',
            )

        with sdk2.parameters.Group('Caching') as caching_block:
            reuse_dependencies_cache = parameters.reuse_dependencies_cache()
            reuse_artifacts_cache = parameters.reuse_artifacts_cache()
            reuse_subtasks_cache = parameters.reuse_subtasks_cache()
            reuse_task_cache = parameters.reuse_task_cache()

        with sdk2.parameters.Group('Webhook') as webhook_block:
            webhook_urls = sdk2.parameters.List('Urls')

        with sdk2.parameters.Group('Relations') as relations_block:
            related_task = sdk2.parameters.Integer('Related task id')
            pull_request_number = sdk2.parameters.Integer('Related pr number')

        tracker_parameters = parameters.TrackerParameters

        with sdk2.parameters.Group('Config') as config_opts:
            external_config = sdk2.parameters.Dict(
                'External config which would be merged with genisys config',
                description='Key starts from context level without project level. '
                            'Key example: pull-request.deploy.static_s3.enabled'
            )
            is_release = sdk2.parameters.Bool('Release build context', default=False)

        debugging_parameters = parameters.DebuggingParameters
        github_statuses = parameters.GithubStatusParameters
        arcanum_checks = parameters.ArcanumChecksParameters
        dependencies = parameters.DependenceParameters
        environment = parameters.EnvironmentParameters
        scripts_sources = parameters.ScriptsSourcesParameters
        statface = parameters.StatfaceParameters
        arcadia_parameters = parameters.ArcadiaParameters

        with sdk2.parameters.Group('System') as system_opts:
            privileged = sdk2.parameters.Bool(
                'Privileged',
                description='Запустить таску в привилегированном режиме (доступно sudo)',
                default=False
            )

        with sdk2.parameters.Group('Overlayfs') as overlayfs:
            use_overlayfs = sdk2.parameters.Bool(
                'Use overlayfs',
                description='Использовать overlayfs (тестовый режим)',
                default=False
            )

        with sdk2.parameters.Group('Arc') as arc_block:
            use_arc = sdk2.parameters.Bool(
                'Use arc',
                description='Использовать arc для чекаута (тестовый режим)',
                default=False
            )

            arc_ref = sdk2.parameters.String(
                'Arc ref',
                description='Использовать arc ref для чекаута репозитория'
            )

        node_js = parameters.NodeJsParameters

    class Context(sdk2.Context):
        noncritical_errors = []
        githubEvent = {}
        report_resources = []
        reused_resources = []
        failed_tasks = []
        not_finished_tasks = []
        current_status = None
        github_context = None
        start_task_version = None

    lifecycle_steps = {}

    fail_on_deps_fail = True

    @property
    def skip_ci_scripts_checkout(self):
        return self.use_arc or not self.use_overlayfs

    @property
    def use_overlayfs(self):
        return self.Parameters.use_overlayfs

    @property
    def use_arc(self):
        return self.Parameters.use_arc or self.project_conf.get('use_arc', False)

    @property
    def sqsh_artifacts(self):
        return self.config.get_deep_value(['sqsh_artifacts'], False)

    def set_ramdrive_size(self, size):
        self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, size, None)

    # conf
    @property
    def conf(self):
        return self.config.get_properties()

    @singleton_property
    def project_conf(self):
        return self.config.get_project_conf(self.conf, {
            'project_name': self.project_name,
            'build_context': self.Parameters.project_build_context or None,
            'external_config': self.Parameters.external_config,
        })

    @singleton_property
    def is_release(self):
        # TODO: нужно поправить все sandbox-ci конфиги, а после удалить метод совсем и везде заменить на Parameters
        return (self.Parameters.is_release or self.Parameters.project_build_context == 'release') and self.Parameters.send_comment_to_searel

    @singleton_property
    def is_dev(self):
        return self.Parameters.project_build_context == 'dev'

    @sdk2.footer()
    def footer(self):
        report = {}

        profiler_actions = self.Context.profiler_actions
        if profiler_actions:
            report.update(**self.task_reports.actions_profile(profiler_actions))

        report.update(**self.task_reports.build_audit_link())

        return report

    @sdk2.header()
    def header(self):
        report = {}

        report_ids = self.Context.report_resources
        if report_ids:
            report.update(**self.task_reports.reports_artifacts(report_ids))

        return report

    @singleton_property
    def registry(self):
        from sandbox.common.config import Registry
        return Registry()

    @property
    def github_context(self):
        raise NotImplementedError

    @property
    def project_name(self):
        raise NotImplementedError

    @property
    def project_dir(self):
        return self.working_path(self.project_name)

    @property
    def project_sources_dir(self):
        if self.use_arc:
            return self.arc_project_dir

        return self.working_path('sources')

    @property
    def arc_mount_path(self):
        return self.working_path('arc_mount')

    @property
    def arc_store_path(self):
        return self.working_path('arc_store')

    @property
    def arc_object_store_path(self):
        return self.working_path('arc_object_store')

    @property
    def arc_project_dir(self):
        return self.arc_mount_path / 'frontend' / 'projects' / self.project_name

    def working_path(self, *args):
        if self.ramdrive and self.ramdrive.path:
            return self.ramdrive_path(*args)
        return self.path(*args)

    def project_path(self, *args):
        return self.project_dir.joinpath(*args)

    def ramdrive_path(self, *args):
        if self.ramdrive and self.ramdrive.path:
            return self.ramdrive.path.joinpath(*args) if args else self.ramdrive.path
        raise Exception('RamDrive is not configured for task')

    def check_task_version_change(self):
        task_version = self.Context.task_version

        # task_version в локальном Sandbox всегда будет равен None
        if not task_version:
            return

        logging.debug('Task code version for {task_type} is {task_version}'.format(
            task_type=self.type,
            task_version=task_version,
        ))

        if not self.Context.start_task_version:
            self.Context.start_task_version = task_version
            return

        if self.Context.start_task_version != task_version:
            self.set_info('Task code version changed in runtime from {prev_version} to {curr_version}.'.format(
                prev_version=self.Context.start_task_version,
                curr_version=task_version,
            ))

    def dump_conf(self):
        logging.debug('Genisys configuration: {}'.format(pformat(self.conf)))
        logging.debug('Project configuration: {}'.format(pformat(self.project_conf)))

    def get_conf_environ(self):
        return self.project_conf.get('environ', {})

    def init_environ(self):
        env.export(self.environ)

        # Оставляем для обратной совместимости —
        # сначала обновим Sandbox задачу, затем скрипты CI, затем удалил эту строку
        env.export({'LOG_PATH': self.log_path()})
        # Добавим переменной более адекватное название
        env.export({'SANDBOX_TASK_LOG_PATH': self.log_path()})

        if self.ramdrive and self.ramdrive.path:
            self.working_path('tmp').mkdir(parents=True, exist_ok=True)
            os.environ['TMP'] = os.environ['TEMP'] = os.environ['TMPDIR'] = str(self.working_path('tmp'))
        env.log(self)

    @singleton_property
    def environ(self):
        """
        Возвращает переменные окружения

        1. из Vault;
        2. из конфигурации genisys;
        3. переданные пользователем через параметр задачи
        :rtype: dict
        """
        task_env = {
            'SANDBOX_TASK_ID': self.id,
            'SANDBOX_TASK_URL': get_task_link(self.id),
        }

        envs = (
            task_env,
            env.from_vault(self),
            self.get_conf_environ(),
            self.Parameters.environ,
        )

        return env.merge(envs)

    def on_enqueue(self):
        self.Context.github_context = self.github_context

        with self.memoize_stage.report_enqueue_status:
            self.scp_feedback.safe_report_self_status(description=u"Ожидает в очереди")
            # self.github_statuses.safe_report_self_status(description=u"Ожидает в очереди")

        if self.Parameters.reuse_task_cache:
            self.wait_same_task()

        if self.Context.reused_same_task:
            logging.debug('Skip waiting dependency tasks')
        else:
            self.wait_dependency_tasks()
            self.wait_dependency_output_parameters()

        if self.Parameters.overwrite_client_tags_flag:
            self.Requirements.client_tags = self.Parameters.overwritten_client_tags

        super(BaseTask, self).on_enqueue()

    def wait_same_task(self):
        finished_statuses = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK

        with self.memoize_stage.wait_same_task:
            same_task = self.find_same_task()

            if same_task:
                self.Context.same_task = same_task.id

                if same_task.status not in ctt.Status.Group.SUCCEED:
                    logging.info('Waiting same task: {}'.format(same_task))

                    raise sdk2.WaitTask(same_task, finished_statuses, timeout=self.wait_timeout)

        if self.Context.same_task:
            same_task = next(iter(sdk2.Task.find(id=self.Context.same_task, children=True).limit(1)))

            if same_task.status in finished_statuses:
                logging.info('The same task {} finished in status {}'.format(same_task, same_task.status))

                if same_task.status in ctt.Status.Group.SUCCEED:
                    self.Context.reused_same_task = True
            else:
                logging.info('Reusing timeout ({timeout}): the same task {task} in {status} status)'.format(
                    task=same_task,
                    timeout=str(datetime.timedelta(seconds=self.wait_timeout)),
                    status=same_task.status
                ))

    @singleton_property
    def wait_timeout(self):
        """
        Маскимальное время ожидания задач.

        Ждём максимально возможное время работы задачи + столько же на накладные расходы Sandbox.

        :rtype: int
        """
        return self.Parameters.kill_timeout * 2

    def find_same_task(self):
        logging.info('Searching same task by parameters: {}'.format(self.cache_parameters))

        succeed_task = self.find_succeed_same_task()

        if succeed_task:
            return succeed_task

        return self.find_started_same_task()

    def find_succeed_same_task(self):
        query = self._find_same_tasks(status=ctt.Status.Group.SUCCEED)
        same_task = next(iter(query.order(-sdk2.Task.id).limit(1)), None)

        logging.info('Found succeed same task: {}'.format(same_task))
        return same_task

    def find_started_same_task(self):
        # статусы из группы EXECUTE, которые могут завершиться успешно
        execute_statuses = (ctt.Status.TEMPORARY, ctt.Status.PREPARING, ctt.Status.EXECUTING, ctt.Status.FINISHING)
        # статусы, которые могут завершиться успешно, отсортированы по степени выполнения задачи:
        # от ENQUEUING — задача ещё не начала выполнение, до FINISHING — задача завершает выполнение
        started_statuses = tuple(ctt.Status.Group.QUEUE) + tuple(ctt.Status.Group.WAIT) + tuple(execute_statuses)

        same_tasks = list(self._find_same_tasks(status=started_statuses).order(-sdk2.Task.id).limit(100))
        # игнорируем текущую задачу
        same_tasks = filter(lambda task: task.id != self.id, same_tasks)
        # пытаемся найти задачу, которая уже почти выполнилась (сортируем по статусу)
        same_tasks = sorted(
            same_tasks,
            key=lambda task: task.status in started_statuses and started_statuses.index(task.status)
        )

        logging.info('Found started same tasks: {}'.format(same_tasks))

        return same_tasks.pop() if same_tasks else None

    def _find_same_tasks(self, status):
        return sdk2.Task.find(
            type=self.type,
            status=status,
            input_parameters=self.cache_parameters,
            children=True,
        )

    def wait_dependency_tasks(self):
        return self.task_dependencies.wait_tasks(fail_on_deps_fail=self.fail_on_deps_fail)

    def wait_dependency_output_parameters(self):
        return self.task_dependencies.wait_output_parameters(timeout=self.wait_timeout)

    @property
    def dependency_output_targets(self):
        return self.task_dependencies.output_parameters

    @singleton_property
    def cache_parameters(self):
        cache_params = dict(_container=self.Parameters._container.id)

        if hasattr(self.Parameters, 'project_tree_hash'):
            cache_params.update(project_tree_hash=self.Parameters.project_tree_hash)

        return cache_params

    def on_create(self):
        super(BaseTask, self).on_create()

    def on_save(self):
        super(BaseTask, self).on_save()
        self.Parameters.priority = prioritizer.get_priority(self)

        self.Requirements.privileged = self.Parameters.privileged

        if self.Context.copy_of:
            self.Parameters.send_statistic = False

        if not self.Parameters.build_id or self.Context.copy_of:
            self.Parameters.build_id = self.id

    def on_prepare(self):
        logging.debug('on_prepare')

        self.dump_conf()
        self.init_environ()

        self.clone_ci_scripts()

        # self.github_statuses.report_self_status(description=u"Выполняется")
        self.scp_feedback.report_self_status(description=u"Выполняется")

    def clone_ci_scripts(self):
        if self.skip_ci_scripts_checkout:
            logging.debug('cloning ci scripts is skipped')
            return

        with self.profile_action(actions_constants['CLONE_SCRIPTS'], 'Cloning scripts'):
            if self.Context.reused_same_task:
                logging.debug('Skip cloning scripts')
            else:
                self.scripts.sync_resource()

    def send_webhooks(self, status):
        if self.Parameters.webhook_urls:
            self.webhook.send(status, self.Parameters.webhook_urls)

    def on_execute(self):
        logging.debug('on_execute')

        self.track_execution_time()
        self.check_task_version_change()

        self.dump_conf()
        self.init_environ()

        if self.Context.reused_same_task:
            logging.debug('Skip executing')
        else:
            self.execute()

    def track_execution_time(self):
        now = time.time()

        if self.Context.first_execution_started_at is ctm.NotExists:
            self.Context.first_execution_started_at = now

        self.Context.execution_started_at = now

    @property
    def execution_time(self):
        if self.Context.execution_started_at is ctm.NotExists:
            return 0

        return int(time.time() - self.Context.execution_started_at)

    @property
    def total_execution_time(self):
        if self.Context.first_execution_started_at is ctm.NotExists:
            return 0

        return int(time.time() - self.Context.first_execution_started_at)

    def execute(self):
        pass

    def on_before_end(self, status):
        """
        Методы on_finish и on_break никак не зависят друг от друга.
        Их общее поведение описывает этот метод.

        :param status: статус задачи
        :type status: str
        """
        self.dump_conf()
        self.init_environ()
        self.send_webhooks(status)
        self.send_status_signal(status)
        self.info.report_all()
        self.save_logging_paths()

    def on_finish(self, prev_status, status):
        logging.debug('on_finish')

        self.on_before_end(status)

        self.Context.current_status = status

        self.profile_actions()
        self.send_profiler_report()
        self.send_deep_profiler_report()

        super(BaseTask, self).on_finish(prev_status, status)

    def send_profiler_report(self):
        need_upload_profile_actions = self.project_conf.get('stat', {}).get('actions_profile', False)
        if not need_upload_profile_actions:
            logging.debug('Skip uploading profile actions to Stat')
            return

        # catch statface errors, see https://st.yandex-team.ru/FEI-5011
        import socket
        from statface_client import StatfaceClientError
        from requests.exceptions import HTTPError, ConnectionError
        try:
            self.profiler.send_report()
        except (StatfaceClientError, HTTPError, ConnectionError, socket.error):
            logging.exception('Failed to upload profile data to Stat')
            error_traceback = traceback.format_exc()
            self.Context.noncritical_errors.append(error_traceback)

    def send_deep_profiler_report(self):
        pass

    def send_status_signal(self, status):
        """
        :param status: статус задачи
        :type status: str
        """
        if getattr(self.Context, 'copy_of', False):
            return

        try:
            push_api.push_signals(
                signals={
                    'status_{}_mmmm'.format(status.lower()): 1,
                    'execution_time_min_hgram': [self.total_execution_time / 60, 1]
                },
                tags={
                    'itype': 'sandboxci',
                    'prj': 'sandboxci',
                    'build_context': (self.Parameters.project_build_context or 'default'),
                    'service_name': self.project_name,
                    'task_type': str(self.type),
                }
            )

            self.zeroline_reporter.report_task({
                'build_id': self.Parameters.build_id,
                'task_type': str(self.type),
                'task_id': self.id,
                'parent_id': (self.parent.id if self.parent else None),
                'project': self.project_name,
                'build_context': (self.Parameters.project_build_context or 'unknown'),
                'status': status,
                'duration': self.total_execution_time,
            })
        except Exception:
            logging.exception('Exception while sending status signal to yasm')

    @contextmanager
    def profile_action(self, act_tag, description):
        # В stat запрещено использовать заглавные буквы и дефисы в названиях колонок.
        # self.profiler записывает данные в колонки, профайлер перед отправкой данных создаёт колонки для
        # несуществующих метрик (act_tag).
        # Поведение профайлера было сломано где то в районе FEI-14103+FEI-15308, а костыль был вставлен в
        # рамках FEI-19369 пока не будет нормальное решение.
        normalized_tag = act_tag.lower().replace('-', '_')

        with self.deep_actions_profiler.actions[act_tag](description), self.profiler.actions[normalized_tag](description):
            yield()

    def profile_actions(self):
        self.profiler.register_wait_actions()
        self.deep_actions_profiler.register_all_actions()

    def on_success(self, prev_status):
        logging.debug('on_success')

        # Should specify state explicitly because of task is in FINISHING status
        self._report_scp_feedback(
            state=GitHubStatus.SUCCESS,
            description=u'Результат реиспользован' if self.Context.reused_same_task else ''
        )

        super(BaseTask, self).on_success(prev_status)

    def on_failure(self, prev_status):
        logging.debug('on_failure')

        # Should specify state explicitly because of task is in FINISHING status
        self._report_scp_feedback(state=GitHubStatus.FAILURE)
        super(BaseTask, self).on_failure(prev_status)

    def on_timeout(self, prev_status):
        logging.debug('on_timeout')

        self._report_scp_feedback(state=GitHubStatus.FAILURE)
        super(BaseTask, self).on_timeout(prev_status)

    def on_break(self, prev_status, status):
        logging.debug('on_break')

        self.on_before_end(status)

        # Should specify state explicitly because of task is in STOPPING status
        self._report_scp_feedback(
            state=GitHubStatus.ERROR,
            description=u'Задача была остановлена' if status == 'STOPPED' else ''
        )

        super(BaseTask, self).on_break(prev_status, status)

    def notify_scp(self, context, description, state, url=None):
        self.scp_feedback.report_status_to_current_sha(
            context=context,
            description=description,
            state=state,
            url=url,
        )

    def notify_github(self, context, description, state, url=None):
        logging.warn('Warning! Method notify_github in BaseTask is just for backward compatibility. Please, '
                     'use notify_scp')
        self.github_statuses.report_status_to_current_sha(
            context=context,
            description=description,
            state=state,
            url=url,
        )

    @staticmethod
    def update_resource_attributes(resource_id, **attrs):
        resource = next(iter(sdk2.Resource.find(id=resource_id).limit(1)), None)
        if resource is None:
            raise Exception('Resource with id={} not found'.format(resource_id))
        for attr, value in attrs.iteritems():
            setattr(resource, attr, value)
        return resource_id

    def save_logging_paths(self):
        try:
            debug_parameters = getattr(self.Parameters, 'debugging_parameters', {})
            is_logging_path_specified = hasattr(debug_parameters, 'logging_paths')

            if is_logging_path_specified:
                for source, dest in self.Parameters.debugging_parameters.logging_paths.items():
                    try:
                        source = self.path(source.strip())
                        dest = self.log_path(dest.strip())

                        if not source.exists():
                            logging.error("Saving path %s does not exist", source)
                            continue

                        if source.is_dir():
                            shutil.copytree(str(source), str(dest))
                        else:
                            dest.mkdir(exist_ok=True)
                            shutil.copy(str(source), str(dest))
                    except Exception:
                        logging.exception("Exception while copying %s->%s", source, dest)
        except Exception:
            logging.exception("Exception while save_logging_paths")

    def set_semaphore(self, name, capacity=None, release=ctt.Status.Group.SEMAPHORES_RELEASE):
        """
        Устанавливает семафор для текущей задачи. `Документация <https://wiki.yandex-team.ru/sandbox/semaphores>`_

        :param name: имя семафора
        :type name: str
        :param capacity: максимальное количество одновременно запущенных экземпляров
        :type capacity: int
        :param release: список статусов, при переходе в которое задача освободит семафор
        :type release: ctt.Status.Group
        """
        if capacity:
            semaphore = ctt.Semaphores.Acquire(name=name, weight=1, capacity=capacity)
        else:
            semaphore = ctt.Semaphores.Acquire(name=name, weight=1)

        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                semaphore,
            ],
            release=release,
        )

    @singleton_property
    def infra_infinity_semaphore(self):
        """
        @TODO: Необходимо удалить этот хак и всё, что с ним связано после https://st.yandex-team.ru/SANDBOX-5938
        """
        return 'infra_infinity_hack'

    def get_test_subtask_description(self, platform, service=None):
        """
        Возвращает описание сабтаски с тестами для разных платформ

        :param platform: платформа
        :type platform: str
        :rtype: str
        """
        desc = self.Parameters.description

        # TODO: удалить после https://st.yandex-team.ru/SANDBOX-4580
        if not isinstance(desc, unicode):
            desc = desc.decode('utf-8')

        if service:
            return u'{} {}/{}'.format(desc, service, platform)

        return u'{} {}'.format(desc, platform)

    def ensure_static_is_uploaded(self, safe=False):
        return self.task_dependencies.ensure_output_parameter('is_static_uploaded', True, 'Static is not uploaded', safe)

    # TODO: удалить после полного перехода на scp feedback
    def _report_github_status(self, state, description=''):
        self.github_statuses.report_self_status(state=state, description=description)

    def _report_scp_feedback(self, state, description=''):
        self.scp_feedback.report_self_status(state=state, description=description)

    @singleton_property
    def is_automated(self):
        return self._is_automated()

    def _is_automated(self, task=None):
        if task is None:
            task = self

        if getattr(task.Context, 'copy_of', False):
            return False

        if task.parent is None:
            return True

        return self._is_automated(task.parent)

    def should_notify_subtask_statuses(self, subtask_parameters):
        """
        Нужно ли уведомлять о статусе подзадачи.
        Уведомления могут быть отключены через параметр `notify_subtask_statuses`, за исключением релизов.
        Уведомления автоматически отключаются для подзадач с тегом `DONT_WAIT`.

        :param subtask_parameters: параметры, которые будут использоваться при создании подзадачи
        :type subtask_parameters: dict
        :rtype: bool
        """
        if self.is_release:
            return True

        should_notify = getattr(self.Parameters, 'notify_subtask_statuses', True)
        dont_wait = self.meta.DONT_WAIT_TAG in subtask_parameters.get('tags', [])

        return should_notify and not dont_wait

    def build_release_critical_notifications(self, notifications=None, mailing_list=MailingLists.INFRADUTY_URGENT):
        """
        :param notifications: правила уведомлений
        :type notifications: list of sdk2.Notification
        :param mailing_list: рассылка, на которую будут отправляться письма
        :type mailing_list: str
        :rtype: list of sdk2.Notification
        """
        default_notifications = self.Parameters.notifications if notifications is None else notifications
        if not self.is_release:
            return default_notifications

        # Связи между тасками построены на ожидании. Если таска переходит в FAILURE, либо её остановил пользователь, то все ожидающие её таски перейдут в STOPPED.
        # Такое поведение приводит к созданию нескольких тикетов на одно событие, например, когда падает обязательная таска (unit-тесты, …), а за ней все остальные.
        # Собственно, поэтому мы исключаем статус STOPPED из нотификаций, чтобы не создавать дубликаты.
        notification_statuses = list(frozenset(ctt.Status.Group.BREAK) - frozenset([ctt.Status.STOPPED])) + [ctt.Status.FAILURE]

        return list(default_notifications) + [sdk2.Notification(
            statuses=notification_statuses,
            recipients=[mailing_list],
            transport=ctn.Transport.EMAIL,
        )]

    def start_files_prefetching_processes(self, prefetch_files_config):
        prefetch_files_config = dict_utils.defaults(prefetch_files_config, {'enabled': False, 'rules': []})

        if not prefetch_files_config['enabled'] or not getattr(self, 'use_arc', False):
            return []

        rules = prefetch_files_config['rules']

        processes = []
        for config in rules:
            config = dict_utils.defaults(config, dict(enabled=False, name=['*.*'], include=['.'], exclude=[]))

            processes += map(lambda include: prefetch_files(self.arc_project_dir, config['name'], include, config['exclude'], check=False, wait=False), config['include'])

        return processes
