# -*- coding: utf-8 -*-

import shutil

from sandbox import sdk2

from sandbox.common.utils import singleton_property, classproperty
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.process import check_process_return_code

from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.task import BaseTask
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants


class BaseUnitTask(BaseTask):
    # TODO: Добавить поддержку на поддержать кэши read-only https://st.yandex-team.ru/FEI-9434

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        project_tree_hash = parameters.project_tree_hash()
        ref = sdk2.parameters.String('Build Branch')

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

    lifecycle_steps = {
        'npm_install': 'make node_modules',
    }
    reports = None

    def on_enqueue(self):
        super(BaseUnitTask, self).on_enqueue()

        self.ensure_static_is_uploaded()

    @singleton_property
    def cache_parameters(self):
        parameters = super(BaseUnitTask, self).cache_parameters
        parameters.update(
            # получаем список из id артефактов
            build_artifacts_resources=map(int, self.Parameters.build_artifacts_resources)
        )
        return parameters

    @classmethod
    def format_github_context(cls, unit_type):
        return u'[Sandbox CI] Unit: {}'.format(unit_type)

    @classproperty
    def github_context(self):
        return self.format_github_context(self.unit_type)

    def execute(self):
        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key():
            self.prepare_sources()
            self.test()

    def prepare_sources(self):
        with self.profile_action(actions_constants['UNPACK_ARTIFACTS'], 'Unpacking build artifacts'):
            self.artifacts.unpack_build_artifacts(self.Parameters.build_artifacts_resources, self.project_dir)
        with self.profile_action(actions_constants['NPM_INSTALL'], 'Installing npm dependencies'):
            self.dependencies.npm_install()

    def generate_report(self):
        pass

    def get_conf_environ(self):
        return env.merge((
            super(BaseUnitTask, self).get_conf_environ(),
            self.project_conf.get('tests', {}).get('environ', {})
        ))

    def test(self):
        with self.profile_action(actions_constants['RUN_TESTS'], 'Running tests'):
            process = self.lifecycle('test', check=False)
            is_failed = bool(process.wait())

            self.generate_report()

            report_status = ctt.Status.FAILURE if is_failed else ctt.Status.SUCCESS
            if self.reports:
                for report_dir, report_path in self.reports.iteritems():
                    self.generate_html_report(
                        {
                            'dir': report_dir,
                            'path': report_path,
                            'status': report_status
                        },
                        process
                    )

            self.artifacts.create_report(
                resource_path=self.artifacts.duplicate_artifact_from_task_log('test.out.txt'),
                type='test.out.txt',
                add_to_context=True,
                status=report_status,
                public=True,
            )

        check_process_return_code(process)

    def generate_html_report(self, report, process):
        report_dir = report.get('dir')
        report_path = report.get('path')
        report_status = report.get('status')

        # Если путь до отчёта не определён, то используем путь запущенного процесса
        if report_path is None:
            report_filename = process.stdout_path_filename
            self.publish_process_log(report_filename, process.stdout_path)

            # Меняем местами, чтобы Sandbox добавлял в ресурс только файл отчёта
            report_path, report_dir = report_dir, report_filename

        self.add_report_resource(self.unit_type, report_dir, report_path, report_status)

    def publish_process_log(self, filename, path):
        """
            Публикует указанный лог в корневую директорию
        """
        shutil.copy(str(path), str(self.path(filename)))

    def add_report_resource(self, unit_type, relative_path, root_path, status):
        """
            Публикует ресурс отчёта
        """
        resource = self.artifacts.create_project_artifact_resource(
            resource_type=SANDBOX_CI_ARTIFACT,
            status=status,
            type='unit-{}-report'.format(unit_type),
            project_tree_hash=self.Parameters.project_tree_hash,
            relative_path=str(relative_path),
            root_path=str(root_path),
            public=True,
        )

        sdk2.ResourceData(resource).ready()
        self.Context.report_resources.append(resource.id)

    def on_before_end(self, status):
        super(BaseUnitTask, self).on_before_end(status)

        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)
