# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common.utils import classproperty

from sandbox.projects.sandbox_ci.task import PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask
from sandbox.projects.sandbox_ci.utils.context import Node
from sandbox.projects.sandbox_ci.utils.process import run_process


class BaseYappyCleanupTask(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """Таск очистки бет в Yappy"""
    class Requirements(BaseTask.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        with sdk2.parameters.Group('Параметры деплоя') as deploy_block:
            suffix = sdk2.parameters.String('Суффикс для беты (pull-1234)', required=True)
            config_path = sdk2.parameters.String('Путь до файла с конфигом', default='./.config/renderer-betas/config.json')
            overwrite_quota = sdk2.parameters.String('Заменить квоту по умолчанию на указанную', default='')

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    lifecycle_steps = {
        'npm_install': 'npm ci'
    }

    @classproperty
    def github_context(self):
        return u'[Sandbox CI] Очищаем бету в Yappy'

    @classproperty
    def task_label(self):
        return 'yappy-cleanup'

    def execute(self):
        deploy_task = self.find_deploy_task()
        if not deploy_task:
            return

        project_hash = deploy_task.Parameters.project_hash
        resources = deploy_task.Parameters.build_artifacts_resources

        if self.Parameters.use_overlayfs:
            with self.prepare_working_copy_context(dict(commit=project_hash)):
                with self._overlayfs(lower_dirs=[self.project_sources_dir], resources=resources, target_dir=self.project_dir):
                    self._install_dependencies()
                    self.cleanup()
        else:
            self._download_sources(resources, self.project_dir)
            self._install_dependencies()
            self.cleanup()

    def find_deploy_task(self):
        return sdk2.Task.find(
            type=self.task_type,
            children=True,
            input_parameters=dict(
                suffix=self.Parameters.suffix,
                project_github_owner=self.Parameters.project_github_owner,
                project_github_repo=self.Parameters.project_github_repo,
            )
        ).order(-sdk2.Task.id).limit(1).first()

    def cleanup(self):
        with self.profiler.actions.cleanup('Closing beta'):
            cmd = 'npx @yandex-int/renderer-betas-cli delete -c "{config_path}" --suffix "{suffix}"'.format(
                config_path=self.Parameters.config_path,
                suffix=self.Parameters.suffix,
            )

            if self.Parameters.overwrite_quota:
                cmd += ' --overwrite-quota {}'.format(self.Parameters.overwrite_quota)

            with Node(self.Parameters.node_js_version), self.vault.ssh_key():
                run_process(
                    [cmd],
                    shell=True,
                    work_dir=self.project_dir,
                    log_prefix='renderer-betas-cli',
                    outputs_to_one_file=False,
                )
