# -*- coding: utf-8 -*-

import logging


MANUAL_TESTING_GITHUB_CONTEXT = u"Ручное тестирование"


class ManualRunMetaTaskMixin(object):
    """
    Добавляет возможности по созданию ManualRun-подзадач и логику выставления статуса проверки "Ручное тестирование" на
    github.
    Не наследует BaseMetaTask, поскольку это сейчас ломает sandbox. Для использования:
    * добавить ManualRunMetaTaskMixin в список базовых классов до BaseMetaTask
    * добавить manual_test_runs = parameters.ManualTestRunsParameters в список параметров
    """
    @property
    def manual_run_task_type(self):
        raise Exception('manual_runs_task_type should be specified in descendants of ManualRunMetaTaskMixin')

    def _manual_testing_skip_reason(self, **kwargs):
        """
        :param issues_info: данные в формате StartrekManager.get_issues_info
        :type issues_info: dict
        :param wait_tasks: список задач, которых нужно дождаться перед выполнением задачи manual_run
        :type wait_tasks: list
        :return: Причина пропуска ручного тестирования
        :rtype: (str, str) or None
        """
        logging.info('manual test runs creation skip detection {}'.format(kwargs))
        issues_info = kwargs['issues_info']

        if issues_info['has_trivial']:
            logging.info('manual test runs creation is skipped because PR is trivial')
            return 'is trivial', u'TRIVIAL не тестируются вручную'

        if issues_info['keys'] and all(map(lambda key: key.startswith('FEI'), issues_info['keys'])):
            logging.info('manual test runs creation is skipped because PR is from fei')
            return 'from fei', u'Задачи из очереди FEI не тестируются вручную'

        if not self.config.is_enabled('deploy', 'testpalm_manual_run'):
            logging.info('manual test runs creation is skipped because PR is switched off in genisys')
            return 'disabled', u'Создание ранов для ручного тестирования выключено в секции genisys deploy.testpalm_manual_run'

        logging.info('manual test runs creation is enabled')
        return None

    def create_manual_run_subtask(self, **kwargs):
        """
        Создает подзадачу для обновления списка ранов в соответствующем проекте testpalm
        :param wait_tasks: список задач, которых нужно дождаться перед выполнением задачи manual_run
        :type wait_tasks: list
        :param issues_info: данные в формате StartrekManager.get_issues_info
        :type issues_info: dict
        :param palmsync_base_url: URL для синхронизации проекта в TestPalm (baseUrl)
        :type palmsync_base_url: str
        :return: созданная задача, если есть
        :rtype: BaseManualTestRunTask or None
        """
        skip_info = self._manual_testing_skip_reason(**kwargs)
        if skip_info:
            skip_reason, skip_description = skip_info

            # Используется механизм meta.skip_step, чтобы информация о скипе попала в context и статусы были скопированы
            # MQ-задачей
            self.meta.skip_step(
                github_context=MANUAL_TESTING_GITHUB_CONTEXT,
                description=skip_description,
                label='manual_testing',
                reason=skip_reason,
            )

            if not self.Parameters.manual_test_runs.force_manual_test_runs:
                return self.meta.skip_step(
                    github_context=self.manual_run_task_type.format_github_context(),
                    description=skip_description,
                    label='manual_runs',
                    reason=skip_reason
                )

        waitable = bool(self.config.get_deep_value(['deploy', 'testpalm_manual_run', 'waitable'], True))
        needed_artifact_types = self.manual_run_task_type.needed_artifact_types

        return self.meta.create_subtask(
            task_type=self.manual_run_task_type,
            waitable=waitable,
            description=self.Parameters.description,
            project_suffix=self.testpalm_project_suffix,
            build_artifacts_resources=self.get_build_artifacts_for_manual_run(needed_artifact_types),
            ref=self.ref,
            **kwargs
        )

    def get_build_artifacts_for_manual_run(self, artifact_types):
        return map(self.get_registered_artifact_id, artifact_types)
