# -*- coding: utf-8 -*-

import logging
import os
import tarfile

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.utils import singleton_property, NullContextmanager

from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.managers.arc.context import create_arc_context
from sandbox.projects.sandbox_ci.managers.config import ConfigManager
from sandbox.projects.sandbox_ci.resources import SandboxCiGitCache
from sandbox.projects.sandbox_ci.utils import dict_utils
from sandbox.projects.sandbox_ci.utils.context import Node, DirContext, GitWithoutLfsProcessing, GitRetryWrapper
from sandbox.projects.sandbox_ci.utils.git import git_cache


class PrepareWorkingCopyMixin(object):
    @singleton_property
    def project_conf(self):
        opts = {
            'project_name': self.project_name,
            'build_context': self.Parameters.project_build_context or None,
            'external_config': self.Parameters.external_config,
        }

        if hasattr(self, 'config') and isinstance(self.config, ConfigManager):
            return self.config.get_project_conf(self.conf, opts)
        else:
            return ConfigManager.get_project_conf(ConfigManager.get_properties(), opts)

    def _download_sources(self, resources, target_dir, **kwargs):
        with self.profile_action(actions_constants['UNPACK_ARTIFACTS'], 'Unpacking build artifacts'):
            return self.artifacts.unpack_build_artifacts(resources, target_dir, **kwargs)

    def _install_dependencies(self):
        with Node(self.Parameters.node_js_version), self.profile_action(actions_constants['NPM_INSTALL'], 'Installing npm dependencies'):
            return self.dependencies.npm_install()

    @property
    def use_git_in_overlayfs_mode(self):
        return not self.use_arc and self.use_overlayfs

    skip_action_profile = False

    def prepare_working_copy_context(self, custom_arc_params={}, custom_git_params={}):
        with NullContextmanager() if self.skip_action_profile else self.profile_action(actions_constants['CLONE_PROJECT'], 'Cloning project'):
            if self.use_arc:
                return self._prepare_context_with_arc_sources(self._get_arc_workcopy_params(custom_arc_params))

            return self._prepare_context_with_git_sources(self._get_git_workcopy_params(custom_git_params))

    def _prepare_context_with_arc_sources(self, workcopy_params):
        return create_arc_context(**workcopy_params)

    def _get_arc_workcopy_params(self, custom_params={}):
        arc_object_store_path = self.arc_object_store_path
        arc_object_store_path.mkdir(parents=True, exist_ok=True)

        arc_config = self.project_conf.get('arc', {})

        extra_params = []
        if arc_config.get('no_threads', False):
            extra_params.append('--no-threads')
        if arc_config.get('use_vfs2', False):
            extra_params.extend(['--vfs-version', '2'])

        arc_apply_attrs = arc_config.get('apply_attrs', {})

        return dict_utils.defaults(custom_params, {
            'mount_point': self.arc_mount_path,
            'store_path': self.arc_store_path,
            'object_store_path': str(arc_object_store_path),
            'commit': getattr(self.Parameters, 'project_hash', ''),
            'extra_params': extra_params,
            'arc_apply_attrs': arc_apply_attrs,
        })

    def _prepare_context_with_git_sources(self, workcopy_params):
        git_url = workcopy_params.get('git_url')
        sources_dir = workcopy_params.get('sources_dir')
        checkout_params = workcopy_params.get('checkout_params', {})
        use_lfs = checkout_params.get('lfs')

        with NullContextmanager() if use_lfs else GitWithoutLfsProcessing(git_url):
            self._prepare_working_copy_sources(git_url, sources_dir, checkout_params)

        return DirContext(sources_dir)

    def _get_git_workcopy_params(self, custom_params):
        git_url = 'https://github.yandex-team.ru/{owner}/{repo}.git'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo
        )
        checkout_params = getattr(self.Parameters, 'git_checkout_params')

        return dict_utils.defaults(custom_params, {
            'git_url': git_url,
            'sources_dir': self.project_sources_dir,
            'checkout_params': dict_utils.defaults(checkout_params, {
                'lfs': not self.skip_lfs_checkout
            })
        })

    skip_lfs_checkout = True

    def _prepare_working_copy_sources(self, git_url, sources_dir, custom_params):
        git_cache_dir = None

        if self.use_git_cache:
            git_cache_dir = self.fetch_git_cache_resource()
        else:
            logging.debug('Skip fetching git cache resource')

        workcopy_params = dict_utils.defaults(custom_params, {
            'dump-hashes': True,
            'cache-dir': git_cache_dir or git_cache(git_url),
            'author-name': 'robot-serp-bot',
            'author-email': 'robot-serp-bot@yandex-team.ru'
        })

        logging.debug('Cloning {url} to {dir} with {params}'.format(
            url=git_url,
            dir=sources_dir,
            params=workcopy_params
        ))

        with GitRetryWrapper():
            return self.scripts.run_js('script/prepare-working-copy.js', git_url, sources_dir, workcopy_params)

    @property
    def use_git_cache(self):
        return not self.use_arc and self.use_overlayfs

    def fetch_git_cache_resource(self):
        owner = self.Parameters.project_github_owner
        repo = self.Parameters.project_github_repo

        git_cache_dir = str(self.working_path('{}_cache'.format(repo)))

        if os.path.exists(git_cache_dir):
            logging.debug('Skip fetching resource: git cache already exists in the folder {cache_dir}'.format(
                cache_dir=git_cache_dir
            ))

            return git_cache_dir

        cache_resource_type = SandboxCiGitCache
        cache_resource_attrs = {
            'lfs': not self.skip_lfs_checkout,
            'git_ref': 'dev',
            'git_url': 'git@github.yandex-team.ru:{}/{}.git'.format(owner, repo)
        }

        logging.debug('Looking for resource "{resource_type}" with attrs: {attrs}'.format(
            resource_type=cache_resource_type,
            attrs=cache_resource_attrs,
        ))

        cache_resource = sdk2.Resource.find(
            resource_type=cache_resource_type,
            attrs=cache_resource_attrs,
            state=ctr.State.READY,
        ).first()

        if cache_resource:
            logging.debug('Cache resource: {}'.format(cache_resource))

            resource_path = sdk2.ResourceData(cache_resource).path
            archive_path = str(self.working_path(resource_path.name))

            logging.debug('Syncing {resource} from {from_path} to {dest_path}'.format(
                resource=cache_resource,
                from_path=resource_path,
                dest_path=archive_path,
            ))

            with tarfile.open(str(resource_path), mode='r:*') as tar:
                logging.debug('Extracting {archive} to {destination}'.format(
                    archive=archive_path,
                    destination=git_cache_dir,
                ))
                tar.extractall(git_cache_dir)

        return git_cache_dir
