# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox import common
from sandbox import projects

from sandbox.common.types import resource as ctr
from sandbox.common.types import task as ctt
from sandbox.common.utils import singleton_property

from sandbox.projects.sandbox.resources import SandboxTasksImage

from sandbox.projects.sandbox_ci.managers.config import ConfigManager


class TasksResourceRequirement(object):
    """
    Mixin для автоматического определения SANDBOX_TASKS_BINARY или SANDBOX_TASKS_IMAGE, на которых должны запускаться задачи Sandbox CI
    """
    @property
    def conf(self):
        return self.config.get_properties()

    @singleton_property
    def project_conf(self):
        if hasattr(self, 'config') and isinstance(self.config, ConfigManager):
            return self.config.get_project_conf(self.conf, {})
        else:
            return ConfigManager.get_project_conf(ConfigManager.get_properties(), {})

    def on_save(self):
        self.__setup_tasks_resource_requirement()

    def __setup_tasks_resource_requirement(self):
        # на локальном Sandbox ревизия равна нулю, а в Taskbox ревизия отсутствует и 'task_resource' выставляется явно автоматически
        revision = projects.__revision__ if hasattr(projects, '__revision__') else None
        if self.Requirements.tasks_resource or revision == 0:
            return

        binary_tasks_config = self.project_conf.get('binary_tasks', {})
        tasks_binary_resource_id = binary_tasks_config.get('tasks_binary_resource_id')
        if tasks_binary_resource_id and not str(self.type) in binary_tasks_config.get('exclude_tasks', []):
            self.__setup_tasks_binary_resource(tasks_binary_resource_id)
        else:
            self.__setup_tasks_archive_resource(revision)

    def __setup_tasks_binary_resource(self, resource_id):
        if resource_id == 'latest':
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                state=ctr.State.READY,
                attrs={'release': ctt.ReleaseStatus.STABLE, 'project': 'sandbox_ci'}
            ).first()
        else:
            self.Requirements.tasks_resource = resource_id

    def __setup_tasks_archive_resource(self, revision):
        """
        Фиксирует указанную версию кода задач

        :param revision: ревизия кода
        :type revision: str
        """

        # obtain latest "official" tasks source code
        resource = SandboxTasksImage.find(
            owner=common.config.Registry().common.service_group,
            state=ctr.State.READY,
            attrs={
                'auto_deploy': True,
                'commit_revision': revision,
            }
        ).first()

        if resource is None:
            return
        self.Requirements.tasks_resource = resource.id
