# -*- coding: utf-8 -*-

import logging
import json
import tarfile
import StringIO
import urllib
import copy

from sandbox import sdk2
from sandbox.common.types import task as ctt, resource as ctr
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import singleton_property, get_task_link

from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine import input_params2 as rm_params2
from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.helpers import startrek_helper

from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.core import task_env

from sandbox.projects.sandbox_ci import managers
from sandbox.projects.sandbox_ci.utils import prioritizer
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.constants import TAGS
from sandbox.projects.sandbox_ci.utils.github import GitHubApi

from sandbox.projects.sandbox_ci.task.ManagersTaskMixin import ManagersTaskMixin
from sandbox.projects.sandbox_ci.sandbox_ci_ab_experiments.ab_experiments_api import ABExperimentsApi

from sandbox.projects.sandbox_ci.task.binary_task import TasksResourceRequirement


class FlagCheckError(Exception):
    pass


@rm_notify.notify2()
class BaseExperimentsReleaseRunnerTask(TasksResourceRequirement, ManagersTaskMixin, sdk2.Task):
    """
    Запуск релизных тестов с экспериментальными флагами
    """

    class Parameters(sdk2.Parameters):
        fail_on_any_error = True

        with sdk2.parameters.Group('Трекер') as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        with sdk2.parameters.Group('Параметры test id') as test_id_block:
            test_id = sdk2.parameters.String('Test id', required=True)
            allow_data_flags = sdk2.parameters.Bool(
                'Разрешить запуск с неверсточными флагами',
                description=u'''Влияет только на тесты с использованием дампов. \
                                Полагаться на результаты прогона теста на дампах, если были \
                                установлены неверсточные флаги, нельзя. \
                                https://wiki.yandex-team.ru/test/serp/prodtesting/#neverstochnyeflagi.''',
                default=False,
            )

        with sdk2.parameters.Group('Шаблоны') as templates_source_block:
            release = sdk2.parameters.RadioGroup(
                'Ветка, из которой нужно брать шаблоны',
                choices=(
                    (u'последний протестированный релиз', 'latest'),
                    (u'указать ветку', 'specify'),
                ),
                sub_fields={'specify': ['project_git_base_ref']},
                default='latest',
                required=True,
            )
            project_git_base_ref = sdk2.parameters.String(
                'Ветка',
                description='В формате release/**/v1.0.0 (например, release/v1.218.0, release/exp3/v1.30.0) или dev',
                required=True,
            )

        with sdk2.parameters.Group('Тесты') as tests_source_block:
            tests_source = sdk2.parameters.RadioGroup(
                'Ветка, из которой нужно брать тесты',
                description='https://wiki.yandex-team.ru/test/serp/prodtesting/#zapusktestovizdevnareliznyxizmenenijax',
                choices=(
                    (u'из той же ветки, из которой берутся шаблоны', 'nothing'),
                    (u'из последнего dev', 'dev'),
                    (u'из указанного dev', 'specify'),
                ),
                sub_fields={'specify': ['tests_hash']},
                default='nothing',
                required=True,
            )
            tests_hash = parameters.CommitHash('Хэш мердж-коммита', required=True)

        with sdk2.parameters.Group('Параметры тестов') as tests_parameters_block:
            with sdk2.parameters.CheckGroup('Инструменты') as tools:
                tools.values['hermione'] = tools.Value('hermione', checked=True)
                tools.values['hermione-e2e'] = tools.Value('hermione-e2e', checked=True)
                tools.values['pulse-shooter'] = tools.Value('pulse-shooter', checked=True)

            with sdk2.parameters.CheckGroup(u'Платформы') as platforms:
                platforms.values['desktop'] = platforms.Value('desktop', checked=True)
                platforms.values['touch-pad'] = platforms.Value('touch-pad', checked=True)
                platforms.values['touch-phone'] = platforms.Value('touch-phone', checked=True)

            hermionee2e_base_url = sdk2.parameters.String('Hermione e2e base url', default='https://hamster.yandex.ru')

            task_retries_delay = sdk2.parameters.List(
                'Паузы в минутах между перезапусками задач',
                description='Массив длительностей пауз в минутах между перезапусками упавших задач',
                default=[0]
            )
            task_retries = sdk2.parameters.Integer(
                'Максимальное количество перезапусков задачи',
                description='Задача будет перезапущена n раз только для упавших тестов',
                default=0
            )
            url_params = sdk2.parameters.Dict(
                'Url parameters',
                description='Параметры, которые будут добавлены в url тестов'
            )

        with sdk2.parameters.Group('Релизная машина') as release_machine_block:
            release_machine_mode = sdk2.parameters.Bool(
                'Запуск из РМ',
                default=False,
                description=(
                    'Используется для отправления уведомлений в приемочный тикет релизной машины. '
                    'Для поиска соответствующего тикета нужно передать название компоненты и номер релиза'
                ),
                sub_fields={'true': ['component_name', 'release_number']},
            )
            component_name_params = rm_params2.ComponentName2()
            release_number = rm_params2.release_number()
            run_flags_mode = sdk2.parameters.Bool(
                'Для компоненты ab_flags запустить выполнение задачи',
                description="Используется только для запусков из компоненты ab_flags RM",
                default=False,
            )

        node_js = parameters.NodeJsParameters

        with sdk2.parameters.Output:
            used_project_git_base_ref = sdk2.parameters.String(u'Использованная ветка для поиска шаблонов')
            used_flags = sdk2.parameters.JSON('Flags')

    class Requirements(sdk2.task.Requirements):
        client_tags = task_env.TaskTags.startrek_client
        environments = [task_env.TaskRequirements.startrek_client]

    @singleton_property
    def vault(self):
        return managers.VaultManager(self)

    @singleton_property
    def experiments_api(self):
        token = self.vault.read('robot-serp-bot_ab_experiments_oauth_token', 'SANDBOX_CI_SEARCH_INTERFACES')
        return ABExperimentsApi(token)

    @singleton_property
    def github_api(self):
        return GitHubApi()

    @singleton_property
    def task_reports(self):
        return managers.Reports(self)

    @singleton_property
    def artifacts(self):
        return managers.ArtifactsManager(self)

    @sdk2.header()
    def header(self):
        if self.Context.subtask_id:
            report_ids = self._get_report_ids()
            return self.task_reports.reports_artifacts(report_ids)

        return u'Проверки еще не запущены'

    @property
    def is_testid_release(self):
        return (
            self.Parameters.release_machine_mode
            and self.Parameters.component_name_params.component_name == 'ab_flags'
        )

    def _get_report_ids(self):
        subtask = self.get_project_experiments_task()
        report_ids = []
        for tool in self.Parameters.tools:
            task_type = self._get_task_type(tool)
            report_type = self._get_report_type(tool)

            tests = subtask.find(task_type)
            for test in tests:
                report = self._find_report(test, report_type)
                if report:
                    report_ids.append(report.id)

        return report_ids

    def _get_task_type(self, tool):
        return self.experiments_subtask_type._get_task_type(tool)

    def _get_report_type(self, tool):
        return '{tool}-report'.format(tool=tool)

    def _find_report(self, task, report_type, **custom_attrs):
        return SANDBOX_CI_ARTIFACT.find(
            task=task,
            attrs=dict(type=report_type, **custom_attrs)
        ).first()

    @property
    def experiments_subtask_type(self):
        raise NotImplementedError

    def get_project_experiments_task(self):
        task_id = self.Context.subtask_id

        task = self._find_task_by_id(task_id)

        if task:
            return task

        raise TaskFailure("Cannot find task #{task_id}.".format(
            task_id=task.id,
        ))

    def set_project_experiments_task(self, task_id):
        self.Context.subtask_id = task_id

    def on_save(self):
        super(BaseExperimentsReleaseRunnerTask, self).on_save()
        self.Parameters.priority = prioritizer.get_priority(self)

    def on_execute(self):
        if (
            self.Parameters.component_name_params.component_name in ["ab_flags", "ab_flags_testids"]
            and not self.Parameters.run_flags_mode
        ):
            self.set_info("Got component 'ab_flags' but run_flags_mode is disabled, should skip this task.")
            return

        self.remind_messages()

        with self.memoize_stage.create_task:
            if self.Parameters.release_machine_mode:
                self._send_st_start_comment()

            test_id = self.Parameters.test_id
            tools = set(self.Parameters.tools)

            params = {}

            if 'hermione' in tools:
                try:
                    flags = self.get_flags(test_id)
                    self.Parameters.used_flags = flags

                    print_flags = ['{}={}'.format(k, v) for k, v in flags.iteritems()]
                    logging.debug('Flags used for {}: {}.'.format(self.test_id_link(test_id), ', '.join(print_flags)))

                    params.update(
                        flags_resource=self.create_flags_resource(flags),
                    )

                except FlagCheckError as e:
                    self.log_warning('Excluding hermione because of check error:\n{error}'.format(error=e.message))
                    tools.remove('hermione')

            if 'pulse-shooter' in tools:
                if self.Parameters.release != 'latest' or not self.Parameters.used_flags:
                    if self.Parameters.release != 'latest':
                        self.log_info('Excluding pulse-shooter because it is not latest release (production) run')

                    if not self.Parameters.used_flags:
                        self.log_info('Excluding pulse-shooter because testid have no suitable flags')

                    tools.remove('pulse-shooter')

            if not tools:
                raise TaskFailure('No tools left for testing')

            params['tools'] = list(tools)

            for tool in tools:
                params[self.get_tool_env_name(tool)] = self.get_tool_params(tool)

                if tool == 'hermione-e2e':
                    params['hermionee2e_base_url'] = self.Parameters.hermionee2e_base_url
                if tool == 'pulse-shooter':
                    params['pulse_shooter_query_params'] = ['test-id={}'.format(self.Parameters.test_id)]
                    params['pulse_shooter_priority'] = 'high' if self.is_testid_release else 'low'

            templates_task = self.get_templates_task()
            tests_task = self.get_tests_task() or templates_task

            self._check_found_task_resources(templates_task)
            self._check_found_task_resources(tests_task)

            task = self.run_project_experiments_task(templates_task, tests_task, **params)

            self.set_project_experiments_task(task_id=task.id)

            raise sdk2.WaitTask(task, statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.parse_results:
            subtask = self.get_project_experiments_task()

            if self.Parameters.release_machine_mode:
                self._send_st_report(subtask.status)

            if subtask.status != ctt.Status.SUCCESS:
                raise TaskFailure('Child task did not finish with success.')

    def remind_messages(self):
        messages = self.Context.messages or []
        for message in messages:
            self.set_info(message, do_escape=False)

        self.Context.messages = []

    def _send_st_start_comment(self):
        # Report about test start to acceptance issue
        # Works only for RM launches with component and release number
        c_info = rmc.COMPONENTS[self.Parameters.component_name_params.component_name]()

        st_auth_token = rm_sec.get_rm_token(self)
        st_helper = startrek_helper.STHelper(st_auth_token)
        st_helper.write_grouped_comment(
            rm_const.TicketGroups.E2ETest,
            '',
            'Start E2E test {}'.format(lb.task_wiki_link(self.id)),
            self.Parameters.release_number,
            c_info,
        )

    def get_flags(self, test_id):
        test_id_params = self.get_related_test_id_params(test_id)
        all_flags = {}
        flag_check_errors = []

        for test_id_param in test_id_params:
            handler = test_id_param.get('HANDLER', None)

            flags, errors = self.get_flags_from_handler(handler, test_id_param)

            all_flags.update(flags)
            flag_check_errors.extend(errors)

        if not all_flags:
            raise FlagCheckError('Test id {} has no flags at CONTEXT.MAIN.REPORT'.format(self.test_id_link(test_id)))

        if flag_check_errors:
            if self.Parameters.allow_data_flags:
                self.log_warning('Ignoring check results:\n{}'.format(flag_check_errors))
            else:
                raise FlagCheckError('Test id {} params seem to change data: {}'.format(
                    self.test_id_link(test_id),
                    ''.join('\t\n' + e for e in flag_check_errors)))

        for name, value in all_flags.iteritems():
            if isinstance(value, dict) or isinstance(value, list):
                all_flags[name] = json.dumps(value, separators=(',', ':'), ensure_ascii=False)

        return all_flags

    def get_related_test_id_params(self, test_id):
        return self.remove_disallowed_handlers(self.get_test_id_params(test_id), self.allowed_handlers)

    def get_test_id_params(self, test_id):
        try:
            config = self.experiments_api.get_test_id(test_id)
            logging.debug('Loaded test_id %s config: %s', test_id, config)
        except Exception as e:
            logging.exception(e)
            raise TaskFailure('Could not load test_id {} configuration'.format(test_id))
        return json.loads(config['params'])

    def remove_disallowed_handlers(self, test_id_params, allowed_handlers):
        return list(filter(lambda test_id_param: test_id_param.get('HANDLER') in allowed_handlers, test_id_params))

    @staticmethod
    def test_id_link(test_id):
        return '<a target="_blank" href="https://ab.yandex-team.ru/testid/{0}">{0}</a>'.format(test_id)

    def get_flags_from_handler(self, handler, test_id_param):
        flags = test_id_param.get('CONTEXT', {}).get('MAIN', {}).get('REPORT', {})

        errors = self.check_flags(flags.keys(), handler, test_id_param)

        return flags, errors

    def check_flags(self, flags, handler, test_id_param):
        errors = self.check_param(test_id_param) + filter(bool, map(lambda flag: self.check_flag(handler, flag), flags))

        return errors

    def check_param(self, param):
        """
        Возвращает список параметров конфига test id, которые потенциально могут влиять на данные.

        :param param: object
        :rtype: list of str
        """
        errors = []
        key_path = []

        def append_error(text):
            errors.append('Key {}: {}'.format('.'.join(key_path), text))

        def check_dict_keys(dictionary, callback):
            for key, value in dictionary.iteritems():
                key_path.append(key)
                callback(key, value)
                key_path.pop()

        def check_test_param(key, value):
            if key == 'HANDLER':
                if value not in self.allowed_handlers:
                    append_error('unknown HANDLER {}'.format(value))
            elif key == 'CONDITION':
                pass
            elif key == 'CONTEXT':
                check_dict_keys(value, check_context_param)
                pass
            elif key == 'TESTID':
                pass
            elif key == 'RESTRICTIONS':
                pass
            else:
                append_error('unknown key {}'.format(key))

        def check_context_param(key, value):
            if key == 'MAIN':
                check_dict_keys(value, check_context_param)
            elif key == 'REPORT':
                pass
            else:
                append_error('unknown key {}'.format(key))

        check_dict_keys(param, check_test_param)

        return errors

    @property
    def allowed_handlers(self):
        raise NotImplementedError

    def check_flag(self, handler, flag):
        flag_info = self.experiments_api.get_flag(handler, flag)

        if not flag_info.get('is_used_in_frontend'):
            return 'Flag {} is data-dependent according to https://ab.yandex-team.ru'.format(flag)

    def create_flags_resource(self, flags):
        flags_archive = 'exp-flags.tar.gz'

        with tarfile.open(flags_archive, 'w:gz') as tar:
            data = json.dumps(flags)

            tarinfo = tarfile.TarInfo('exp-flags.json')
            tarinfo.size = len(data)

            logging.debug('Adding to {archive} flags: {flags}'.format(
                archive=flags_archive,
                flags=data,
            ))

            tar.addfile(tarinfo, StringIO.StringIO(data))

        resource = self.artifacts.create_artifact_resource(
            relative_path=flags_archive,
            resource_type=SANDBOX_CI_ARTIFACT,
            type='exp-flags',
        )

        logging.debug('Resource with {archive}: {resource}'.format(
            archive=flags_archive,
            resource=resource,
        ))

        sdk2.ResourceData(resource).ready()

        return resource

    def get_tool_env_name(self, tool):
        tool_without_separator = ''.join(tool.split('-'))

        return '{tool}_env'.format(tool=tool_without_separator)

    def get_tool_params(self, tool):
        params = {}

        env_prefix = self._get_env_prefix(tool)

        params.update(self.get_url_params(env_prefix))

        if tool == 'hermione-e2e':
            params.update(
                {'HERMIONE_URL_CUSTOM_QUERIES': 'test-id={}'.format(self.Parameters.test_id)}
            )
        else:
            params.update(
                {'ARCHON_KOTIK_TEMPLATE_FLAG': json.dumps(self.Parameters.used_flags)}
            )

        return params

    def _get_env_prefix(self, tool):
        tool_name = tool.split('-')[0]

        return '{}_URL_QUERY'.format(tool_name.upper())

    def get_url_params(self, env_prefix):
        return dict(map(lambda param: ('{}_{}'.format(env_prefix, param[0]), param[1]), self.Parameters.url_params.iteritems()))

    def encode_flags(self, flags):
        return [urllib.quote_plus('{}={}'.format(k, v)) for k, v in flags.iteritems()]

    def get_templates_task(self):
        """
        Возвращает объект метазадачи, которая удовлетворяет входным параметрам
        :rtype: sdk2.Task
        """
        templates_source = self.Parameters.release

        if templates_source == 'latest':
            return self.get_latest_release_task()

        if templates_source == 'specify':
            if self.Parameters.project_git_base_ref == 'dev':
                return self.get_dev_task()
            else:
                return self.get_specified_release_task(self.Parameters.project_git_base_ref)

        raise ValueError('Invalid value: {}'.format(templates_source))

    def get_latest_release_task(self, **input_params):
        """
        Получаем задачу для последнего релиза
        :rtype: sdk2.Task
        """
        task_type = self.project_task_type
        task_search_parameters = dict(
            type=task_type,
            status=ctt.Status.RELEASED,
            release='stable',
            children=True,
            tags=TAGS.RELEASE.value,
            input_parameters=input_params
        )

        logging.debug('Trying to find task with the following parameters: {}'.format(task_search_parameters))

        task = sdk2.Task.find(**task_search_parameters).first()

        if task:
            logging.debug('The latest release was build in task #{}.'.format(task.id))
            return task

        user_message = "\n".join([
            "Could not find the latest stable release for the service with the '{task_type}' task type.".format(task_type=task_type),
            'Contact support if the service has been released at least once previously.'
        ])

        self.set_info(user_message)

        raise TaskFailure('Could not find a task for the latest release.')

    def get_dev_task(self, **input_params):
        task_type = self.project_task_type
        task_search_parameters = dict(
            type=task_type,
            status=ctt.Status.Group.FINISH,
            input_parameters=dict(
                project_git_base_ref='dev',
                project_build_context='dev',
                **input_params
            ),
        )

        logging.debug('Trying to find task with the following parameters: {}'.format(task_search_parameters))

        task = sdk2.Task.find(**task_search_parameters).first()

        if task:
            logging.debug('The latest build for trunk was in task #{}.'.format(task.id))
            return task

        user_message = "\n".join([
            "Could not find the latest finished build for the trunk ('dev' branch) with the '{task_type}' task type.".format(task_type=task_type),
            'Contact support if the service has at least one finished build for the trunk.'
        ])

        self.set_info(user_message)

        raise TaskFailure('Could not find a task for the specified trunk build.')

    def get_specified_release_task(self, project_git_base_ref):
        """
        :param project_git_base_ref:
        :type project_git_base_ref: str
        :rtype: sdk2.Task
        """
        task_type = self.project_task_type
        task_search_parameters = dict(
            type=task_type,
            status=ctt.Status.Group.FINISH,
            input_parameters={'project_git_base_ref': project_git_base_ref},
        )

        logging.debug('Trying to find task with the following parameters: {}'.format(task_search_parameters))

        task = sdk2.Task.find(**task_search_parameters).first()

        if task:
            logging.debug('The latest build for the specified release was in task #{}.'.format(task.id))
            return task

        user_message = "\n".join([
            "Could not find finished build for the '{ref}' ref with the '{task_type}' task type.".format(
                ref=project_git_base_ref,
                task_type=task_type
            ),
            'Contact support if the specified ref actually exists and there is at least one finished build for it.'
        ])

        self.set_info(user_message)

        raise TaskFailure('Could not find a task for the specified release.')

    def get_tests_task(self):
        tests_source = self.Parameters.tests_source

        if tests_source == 'nothing':
            return None

        if tests_source == 'dev':
            return self.get_dev_task()

        if tests_source == 'specify':
            return self.get_dev_task(
                project_git_base_commit=self.Parameters.tests_hash
            )

        raise ValueError('Invalid value: {}'.format(tests_source))

    def run_project_experiments_task(self, templates_task, tests_task, **params):
        # Подменяем ветку на тег. Это обусловлено тем, что на момент запуска этой таски, как правило,
        # релизной ветки уже не существует и нам нечего чекаутить.
        # В релизах тег всегда сохраняется
        git_checkout_parameters = copy.deepcopy(tests_task.Parameters.git_checkout_params)
        branch = git_checkout_parameters['ref']

        if branch != 'dev':
            git_checkout_parameters['ref'] = self._get_tag_by_release_branch(branch)

        self.set_info('The project task will use tests from task {tests_task_link} on templates from task {templates_task_link}.'.format(
            tests_task_link='<a href="{}">#{}</a>'.format(get_task_link(tests_task.id), tests_task.id),
            templates_task_link='<a href="{}">#{}</a>'.format(get_task_link(templates_task.id), templates_task.id),
        ), do_escape=False)

        """
        :param params: Параметры таски
        :type params: dict of str:str
        :rtype: sdk2.Task
        """
        return self.experiments_subtask_type(
            self,

            description='Testing with test id {}'.format(self.Parameters.test_id),

            project_github_owner=templates_task.Parameters.project_github_owner,
            project_github_repo=templates_task.Parameters.project_github_repo,
            project_build_context=tests_task.Parameters.project_build_context,

            platforms=self.Parameters.platforms,

            project_tree_hash_for_templates=templates_task.Parameters.project_tree_hash,
            project_git_base_ref=templates_task.Parameters.project_git_base_ref,
            merge_commit=tests_task.Parameters.project_git_base_commit,
            resources_yenv='production' if self.is_template_from_release else 'testing',

            send_comment_to_searel=True,
            send_comment_to_issue=self.Parameters.send_comment_to_issue,

            task_retries_delay=self.Parameters.task_retries_delay,
            task_retries=self.Parameters.task_retries,

            git_checkout_params=git_checkout_parameters,

            **params
        ).enqueue()

    def _get_tag_by_release_branch(self, branch=''):
        tag_components = branch.split('/')[1:]

        return '/'.join(tag_components)

    @property
    def is_template_from_release(self):
        return self.Parameters.release == 'latest' or self.Parameters.project_git_base_ref.startswith('release')

    def log_warning(self, warning):
        logging.warn(warning)
        self.show_message('<strong>{}</strong>'.format(warning))

    def log_info(self, info):
        logging.info(info)
        self.show_message(info)

    def show_message(self, message):
        """
        Выводит сообщение через set_info. Запоминает в контекст, чтобы при последующих запусках показывать старые
        сообщения.
        :param message: текст сообщения
        :type message: str
        """
        if not self.Context.messages:
            self.Context.messages = []
        self.Context.messages.append(message)

        self.set_info(message, do_escape=False)

    def _send_st_report(self, test_status):
        # Report about test result to acceptance issue
        # Works only for RM launches with component and release number
        c_info = rmc.COMPONENTS[self.Parameters.component_name_params.component_name]()

        st_auth_token = rm_sec.get_rm_token(self)
        st_helper = startrek_helper.STHelper(st_auth_token)

        report_ids = self._get_report_ids()
        message = 'SB: {}\n{}'.format(
            lb.task_wiki_link(self.id), self.task_reports.wiki_report_artifacts(report_ids)
        )
        title = 'E2E test result for {e2e_base_url} is **!!({status_color}){status}!!**'.format(
            e2e_base_url=self.Parameters.hermionee2e_base_url,
            status_color='green' if test_status == ctt.Status.SUCCESS else 'red',
            status=test_status
        )
        st_helper.write_grouped_comment(
            rm_const.TicketGroups.E2ETest, title, message, self.Parameters.release_number, c_info
        )

    def _check_found_task_resources(self, task):
        """
        Проверяем, что найденный таск обладает необходимыми ресурсами.
        Эта проверка имеет смысл, потому что мы разрешаем использовать 'FAILURE' таски, которые могут упасть до публикации ресурсов.
        Проверка грубая и наивная, потому что опирается на число ресурсов.

        :type task: sdk2.Task
        """
        # Запрашиваем максимум два ресурса, потому что наша проверка нацелена лишь на отсеивание поломанных тасок,
        # в которых ресурсы не были созданы вообще. Как минимум, один ресурс всегда присутствует (логи).
        ready_resources_limit = 2

        logging.debug('Trying to fetch a list of resources for task #{}.'.format(task.id))

        ready_resources = list(sdk2.Resource.find(
            task=task,
            state=ctr.State.READY
        ).limit(ready_resources_limit))

        logging.debug('Task #{task_id} has the following resources: {resources}'.format(task_id=task.id, resources=ready_resources))

        if len(ready_resources) == ready_resources_limit:
            logging.debug('Looks like task #{} can be used for further work.'.format(task.id))
            return

        user_message = "\n".join([
            'The found task #{} does not have the necessary resources.'.format(task.id),
            'If you are running testing on the latest stable release, then contact support.',
            'Otherwise, check the current incidents and try restarting this task later, when the next build of the service will be finished.'
        ])

        self.set_info(user_message)

        raise TaskFailure('Task #{} does not have the necessary resources.'.format(task.id))

    def _find_task_by_id(self, task_id):
        return sdk2.Task.find(id=task_id, children=True).first()
