# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.pulse import parameters as pulse_params
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.projects.sandbox_ci.task import BaseTask

from sandbox.projects.release_machine.core import task_env


class BaseExperimentsTask(BaseTask):
    """
    Мета-задача для запуска тестов с экспериментальными флагами
    """

    github_context = u'irrelevant'

    class Context(BaseTask.Context):
        subtasks = []

    class Parameters(BaseTask.Parameters):
        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        with sdk2.parameters.Group('Параметры версии репозитория') as repo_version_block:
            project_tree_hash_for_templates = parameters.CommitHash('Project tree hash for templates')
            project_git_base_ref = parameters.project_git_base_ref()

            merge_commit = parameters.CommitHash('Merge commit дева, из которого взяты тесты', default='')

            resources_yenv = sdk2.parameters.String(
                'YENV ресурсов',
                choices=[
                    ('testing', 'testing'),
                    ('pre-production', 'pre-production'),
                    ('production', 'production'),
                ],
                default='production',
            )

        with sdk2.parameters.Group('Параметры тестов') as tests_block:
            with sdk2.parameters.CheckGroup('Инструменты') as tools:
                tools.values['hermione'] = tools.Value('hermione', checked=True)
                tools.values['hermione-e2e'] = tools.Value('hermione-e2e', checked=True)
                tools.values['pulse-shooter'] = tools.Value('pulse-shooter', checked=False)

            with sdk2.parameters.CheckGroup(u'Платформы') as platforms:
                platforms.values['desktop'] = platforms.Value('desktop', checked=True)
                platforms.values['touch-pad'] = platforms.Value('touch-pad', checked=True)
                platforms.values['touch-phone'] = platforms.Value('touch-phone', checked=True)

            hermione_env = sdk2.parameters.Dict('Hermione env parameters')
            hermionee2e_env = sdk2.parameters.Dict('Hermione e2e env parameters')
            hermionee2e_base_url = sdk2.parameters.String('Hermione e2e base url', default='https://hamster.yandex.ru')
            pulse_shooter_query_params = sdk2.parameters.List('Pulse Shooter query parameters')
            pulse_shooter_priority = pulse_params.soy_operation_priority()

            beta_domain = sdk2.parameters.String(
                'Beta domain from PR',
                description='Используется, чтобы подставлять в отчет бету из PR вместо автобеты на которой гонялись тесты',
            )

            task_retries_delay = sdk2.parameters.List(
                'Паузы в минутах между перезапусками задач',
                description='Массив длительностей пауз в минутах между перезапусками упавших задач',
                default=[0]
            )
            task_retries = sdk2.parameters.Integer(
                'Максимальное количество перезапусков задачи',
                description='Задача будет перезапущена n раз только для упавших тестов',
                default=0
            )

        with sdk2.parameters.Group('Флаги') as flags_block:
            flags_resource = sdk2.parameters.Resource(
                'Flags',
                description=u'Ресурс с экспериментальными флагами',
            )

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    class Requirements(BaseTask.Requirements):
        client_tags = task_env.TaskTags.startrek_client
        environments = [task_env.TaskRequirements.startrek_client]

    def execute(self):
        with self.memoize_stage.create:
            resources = self.get_build_artifacts()
            if self.Parameters.flags_resource:
                resources.append(self.Parameters.flags_resource)

            subtasks = self.create_subtasks(resources)

            if len(subtasks) == 0:
                raise TaskFailure('No subtasks were created')

            self.Context.subtasks = map(lambda task: task.id, subtasks)

            started_subtasks = self.meta.start_subtasks(subtasks)
            raise sdk2.WaitTask(
                tasks=self.meta.get_waitable_tasks_ids(started_subtasks),
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        with self.memoize_stage.process:
            failed_subtasks = self.meta.failed_subtasks

            if failed_subtasks:
                self.meta.store_failed_tasks(failed_subtasks)

                raise TaskFailure('Has failed subtasks, see reports for more details')

    def get_build_artifacts(self):
        return map(self.get_build_artifact, self.build_artifact_types)

    def get_build_artifact(self, artifact_type, resource_type=SANDBOX_CI_ARTIFACT):
        resource_search_parameters = dict(
            resource_type=resource_type,
            state=ctr.State.READY,
            attrs=dict(
                project_tree_hash=self.Parameters.project_tree_hash_for_templates,
                YENV=self.Parameters.resources_yenv,
                owner=self.Parameters.project_github_owner,
                repo=self.Parameters.project_github_repo,
                project=self.project_name,
                ref=self.Parameters.project_git_base_ref,
                type=artifact_type,
            ),
        )

        logging.debug('Trying to find build artifact with the following parameters: {}'.format(resource_search_parameters))

        resource = sdk2.Resource.find(**resource_search_parameters).first()

        if resource:
            logging.debug('The latest build artifact in resource #{}.'.format(resource.id))
            return resource

        user_message = "\n".join([
            "Cloud not find resource with templates for the following attributes: type={resource_type} project_tree_hash={project_tree_hash} YENV={yenv}".format(
                resource_type=artifact_type,
                project_tree_hash=self.Parameters.project_tree_hash_for_templates,
                yenv=self.Parameters.resources_yenv,
            ),
            'Contact support if this task was created automatically.',
            'Otherwise, make sure that resources exist for the specified parameters.'
        ])

        self.set_info(user_message)

        raise TaskFailure('')

    @property
    def build_artifact_types(self):
        raise NotImplementedError

    @property
    def pulse_platforms(self):
        raise NotImplementedError

    def create_subtasks(self, resources):
        subtasks = []

        for tool in self.Parameters.tools:
            env = getattr(self.Parameters, self._get_env_param_name(tool), None)
            task_type = self._get_task_type(tool)

            for platform in self.Parameters.platforms:
                if tool == 'hermione-e2e':
                    hermione_e2e_subtask = self.create_subtask(
                        task_type=task_type,
                        resources=resources,
                        platform=platform,
                        env=env,
                        hermione_base_url=self.Parameters.hermionee2e_base_url,
                        project_name=self.project_name
                    )
                    subtasks.append(hermione_e2e_subtask)
                elif tool == 'hermione':
                    subtasks.append(self.create_subtask(task_type, resources, platform, env))
                elif tool == 'pulse-shooter':
                    continue

        return subtasks

    def _get_env_param_name(self, tool):
        tool_without_separator = ''.join(tool.split('-'))

        return '{tool}_env'.format(tool=tool_without_separator)

    @staticmethod
    def _get_task_type(self, tool):
        raise NotImplementedError

    def create_subtask(self, task_type, resources, platform, env, **kwargs):
        return self.meta.create_subtask(
            tags=['flags', platform],
            platform=platform,
            description=platform,
            **self._get_common_task_params(task_type, resources, env, **kwargs)
        )

    def _get_common_task_params(self, task_type, resources, env, **kwargs):
        return dict(
            task_type=task_type,
            build_artifacts_resources=resources,
            project_build_context=self.Parameters.project_build_context,
            ref=self.Parameters.project_git_base_ref,

            # Отключаем доставку статусов на Гитхаб
            report_github_statuses=False,

            # Отключаем переиспользование старых результатов
            reuse_subtasks_cache=False,
            reusable=False,

            # Отключаем отправку статистики в statface
            send_statistic=False,
            environ=env,

            # Выключаем селективность
            selective_run=False,

            # Ретраи
            task_retries=self.Parameters.task_retries,
            task_retries_delay=self.Parameters.task_retries_delay,

            beta_domain=self.Parameters.beta_domain,

            **kwargs
        )
