# -*- coding: utf-8 -*-

import json
import socket
import logging
import datetime as dt
import traceback
from sandbox import sdk2

STATFACE_REPORT_PREFIX = 'Yandex.Productivity/infratest'


def send_stat_report(task):
    stat_file_path = sdk2.Path(task.project_name) / 'stat-reporter/{}.json'.format(task.tool)

    if not task.path(stat_file_path).exists():
        logging.warn('Stat file {} does not exist on filesystem'.format(stat_file_path))
        return

    from statface_client import StatfaceClientError
    from requests.exceptions import HTTPError, ConnectionError

    try:
        report = get_report(task)
        report_config = build_report_config(task)
        report_data = load_json(stat_file_path)

        current_date = dt.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        for item in report_data:
            item['fielddate'] = current_date
            item['task_id'] = task.id
            item['parent_task_id'] = task.parent.id if task.parent else None
            item['git_ref'] = get_git_ref(task)

        report.upload_config(
            config=report_config,
            request_kwargs=dict(timeout=60),
            _request_timeout=60,
        )

        report.upload_data(
            scale='s',
            data=report_data,
            request_kwargs=dict(timeout=60),
            _request_timeout=60,
        )

    except (StatfaceClientError, HTTPError, ConnectionError, socket.error):
        error_traceback = traceback.format_exc()
        task.Context.noncritical_errors.append(error_traceback)
        logging.error(error_traceback)
    except Exception:
        error_traceback = traceback.format_exc()
        logging.error(error_traceback)


def load_json(file_path):
    json_file = open(str(file_path), 'r').read()
    logging.info('Load json: {}'.format(json_file))
    return json.loads(json_file)


def get_report(task):
    report_path = '{}/{}'.format(STATFACE_REPORT_PREFIX, create_report_path(task))
    return task.statface.get_report(report_path)


def build_report_config(task):
    from statface_client.report import StatfaceReportConfig

    return StatfaceReportConfig(
        title=create_report_path(task, ' / '),
        dimensions=[
            ('browser', 'string'),
            ('fielddate', 'date')
        ],
        measures=[
            ('tests', 'number'),
            ('passed', 'number'),
            ('failed', 'number'),
            ('skipped', 'number'),
            ('retries', 'number'),
            ('status', 'string'),
            ('start', 'string'),
            ('end', 'string'),
            ('task_id', 'number'),
            ('parent_task_id', 'number'),
            ('git_ref', 'string')
        ],
    )


def create_report_path(task, delimiter='/'):
    service = task.Parameters.service if hasattr(task.Parameters, 'service') else None
    report_prefix = task.project_conf.get('stat', {}).get('report_prefix', None)

    params = [
        task.project_name,
        report_prefix,
        service,
        task.tool,
        task.Parameters.platform
    ]

    return delimiter.join(filter(lambda x: bool(x), params))


def get_git_ref(task):
    if not task.parent:
        return

    parameters = task.parent.Parameters

    git_ref = None

    if hasattr(parameters, 'project_git_base_ref') and parameters.project_git_base_ref:
        git_ref = parameters.project_git_base_ref

    if hasattr(parameters, 'project_git_merge_ref') and parameters.project_git_merge_ref:
        git_ref = parameters.project_git_merge_ref[0]  # первый ref в списке указывает на head (@see FEI-5174)

    return git_ref


__all__ = [send_stat_report]
