# -*- coding: utf-8 -*-

import os
from sandbox import sdk2

from sandbox.common.utils import classproperty
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.task import OverlayfsMixin, PrepareWorkingCopyMixin, BaseTask
from sandbox.projects.sandbox_ci.utils import env
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node


class SandboxCiSkipValidator(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    """
        Таск валидации тестов, заскипанных по задаче из пулл-реквеста
    """
    class Requirements(BaseTask.Requirements):
        cores = 4
        ram = 8192

        class Caches(BaseTask.Requirements.Caches):
            pass

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = parameters.build_artifacts_resources()
        issue_keys = sdk2.parameters.List('Startrek issue keys', sdk2.parameters.String)
        project_name = sdk2.parameters.String('Project name', required=True)
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'validate': './node_modules/@yandex-int/skip-validator/bin/skip-validator {issuesOpt}'
    }

    github_context = u'[Sandbox CI] Проверка всех включённых тестов по задаче'

    @property
    def project_name(self):
        return str(self.Parameters.project_name)

    @classproperty
    def task_label(self):
        return 'skip-validator'

    def on_prepare(self):
        super(SandboxCiSkipValidator, self).on_prepare()

        self.lifecycle.update_vars(
            issuesOpt=getIssuesOpt(self.Parameters.issue_keys)
        )

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        if not self.Parameters.issue_keys:
            return

        self.set_environments()
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)
        self._install_dependencies()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key():
            self.lifecycle('validate')

    def execute_in_overlayfs_mode(self):
        if not self.Parameters.issue_keys:
            return

        self.set_environments()

        with self.prepare_working_copy_context():
            with Node(self.Parameters.node_js_version), self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir):
                self.lifecycle('validate')

    def set_environments(self):
        # Нужно для корректной работы плагина external-skip
        os.environ['ISSUE_KEYS'] = ','.join(self.Parameters.issue_keys or [])

        os.environ['hermione_oauth_token'] = self.vault.read('env.SURFWAX_TOKEN')

    def get_conf_environ(self):
        return env.merge((
            super(SandboxCiSkipValidator, self).get_conf_environ(),
            self.config.get_deep_value(['tests', 'skip-validator', 'environ'], {})
        ))


def getIssuesOpt(issues):
    return ' '.join(['--id {}'.format(issue) for issue in issues])
