# -*- coding: utf-8 -*-

import os
import re
import shutil
import stat
import subprocess32 as subprocess
import logging

from sandbox.sandboxsdk import errors

from sandbox import sdk2
from sandbox.common.utils import singleton_property
from sandbox.common.types import task as ctt
from sandbox.common import config

from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils import flow, env, list_utils, poll
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper, Node, Overlayfs, DaemonProcesses
from sandbox.projects.sandbox_ci.decorators.in_case_of import in_case_of
from sandbox.projects.sandbox_ci.task import OverlayfsMixin, PrepareWorkingCopyMixin, BaseTask, infratest_profiler
from sandbox.projects.sandbox_ci.task.test_task.plugins import HtmlReporter
from sandbox.projects.sandbox_ci.task.test_task.plugins import TemplateErrorCollector
from sandbox.projects.sandbox_ci.task.test_task.LinkBuilder import LinkBuilder
from sandbox.projects.sandbox_ci.task.test_task.RepeaterKiller import RepeaterKiller
from sandbox.projects.sandbox_ci.managers.actions_constants import actions_constants
from sandbox.projects.sandbox_ci.resources import SANDBOX_CI_ARTIFACT
from sandbox.common.errors import TaskStop

COMMON_SEMAPHORE_PREFIX = 'selenium'
AG_SEMAPHORE_PREFIX = 'selenium_autogenerated'
RAMDRIVE_SIZE = 15 * 1024


class BaseTestTask(PrepareWorkingCopyMixin, OverlayfsMixin, BaseTask):
    # Для задачи, создающей ручные раны, важно получать ресурсы для обычного прогона gemini, обычного прогона hermione и
    # прогона заскипанных hermione. Чтобы различать эти ресурсы, добавляем им атрибут mod.
    # default - обычный прогон (по умолчанию)
    # skipped - прогон заскипанных
    mod = 'default'

    class Parameters(BaseTask.Parameters):
        build_artifacts_resources = sdk2.parameters.Resource(
            'Build artifacts',
            resource_type=SANDBOX_CI_ARTIFACT,
            multiple=True,
            register_dependency=False,
        )
        platform = parameters.platform()
        project_tree_hash = parameters.project_tree_hash()
        project_base_hash = parameters.project_base_hash()
        project_hash = parameters.project_hash()
        ref = sdk2.parameters.String('Build Branch')
        issue_keys = sdk2.parameters.List('Startrek issue keys', sdk2.parameters.String)
        custom_command = sdk2.parameters.String('Кастомная команда перед запуском тестов', multiline=True)
        custom_opts = sdk2.parameters.String('Опции, которые будут добавлены при запуске тестов')
        coredump_filter_pattern = parameters.coredump_filter_pattern()
        send_statistic = sdk2.parameters.Bool('Отправлять статистику в statface', default=True)
        beta_domain = sdk2.parameters.String(
            'Домен беты из PR',
            description='Используется, чтобы подставлять в отчет бету из PR вместо автобеты на которой гонялись тесты',
        )
        use_synchrophazotron = sdk2.parameters.Bool('Использовать synchrophazotron для скачивания ресурсов', default=True)

        with BaseTask.Parameters.tracker_block() as tracker_block:
            send_comment_to_issue = parameters.send_comment_to_issue()

        git_checkout_params = sdk2.parameters.JSON('Параметры для чекаута git-репозитория в режиме overlayfs', default={})

    class Requirements(BaseTask.Requirements):
        disk_space = 30 * 1024
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, RAMDRIVE_SIZE, None)
        kill_timeout = 2400

    @property
    def skip_lfs_checkout(self):
        return self.use_arc or not self.use_overlayfs

    lifecycle_steps = {
        'npm_install': 'npm ci',
        'custom_command': '{custom_command}',
        'test': 'node {run_cmd} {run_opts} {run_custom_opts}',
    }

    @property
    def priority(self):
        return tuple(filter(bool, self.config.get_deep_value(['tests', self.tool, 'priority'], '').split(':')))

    @property
    def lifecycle(self):
        return super(BaseTestTask, self).lifecycle.update_vars(
            tool=self.tool,
            run_cmd=self.run_cmd,
            run_opts=self.run_opts,
            run_custom_opts=self.Parameters.custom_opts,
            custom_command=self.Parameters.custom_command,
        )

    @property
    def run_cmd(self):
        raise NotImplementedError

    @property
    def run_opts(self):
        browsers = self.get_browsers_to_run()
        return ' '.join(['-b {} {}'.format(bro['name'], self.browsers_config.format_sessions_per_browser_cli_option(bro['name'], bro['sessions'])) for bro in browsers])

    def get_browsers_to_run(self):
        return self.browsers_config.get_browsers_to_run()

    @singleton_property
    def cache_parameters(self):
        parameters = super(BaseTestTask, self).cache_parameters
        parameters.update(
            platform=self.Parameters.platform,
            # получаем список из id артефактов
            build_artifacts_resources=map(int, self.Parameters.build_artifacts_resources)
        )
        return parameters

    @classmethod
    def format_github_context(cls, description):
        return u'[Sandbox CI] {}: {}'.format(cls.tool.capitalize(), description)

    @property
    def github_context(self):
        return self.format_github_context(self.report_description)

    @property
    def report_description(self):
        return self.Parameters.platform

    @property
    def int_resources_path(self):
        return self.working_path('.yandex-int')

    @property
    def int_logs_path(self):
        return self.int_resources_path / 'logs'

    @property
    def should_poll_dependencies(self):
        return self.config.get_deep_value(['tests', self.tool, 'poll_dependencies', 'enabled'], False)

    @property
    def poll_dependencies_interval(self):
        return int(self.config.get_deep_value(['tests', self.tool, 'poll_dependencies', 'interval'], 10))

    @property
    def poll_dependencies_timeout(self):
        return int(self.config.get_deep_value(['tests', self.tool, 'poll_dependencies', 'timeout'], 60))

    @singleton_property
    def prefetch_files_config(self):
        return self.config.get_deep_value(['tests', self.tool, 'prefetch_files'], {})

    def wait_dependency_output_parameters(self):
        if not self.should_poll_dependencies:
            super(BaseTestTask, self).wait_dependency_output_parameters()

    def wait_dependency_resources(self):
        from sandbox.yasandbox.manager.task import TaskManager
        map(lambda resource: TaskManager.register_dep_resource(self.id, resource), self.Parameters.build_artifacts_resources)

    def on_enqueue(self):
        super(BaseTestTask, self).on_enqueue()

        if not self.should_poll_dependencies:
            self.ensure_static_is_uploaded()
            self.wait_dependency_resources()

        setattr(self.Context, '__do_not_dump_ramdrive', True)  # use setattr to avoid mangling

        self.dump_conf()

        if self.Context.reused_same_task:
            return

        if self.Context.is_base_task_failed:
            # @TODO: заменить на 'self.Requirements.semaphores = None' после https://st.yandex-team.ru/SANDBOX-5938
            self.set_semaphore(name=self.infra_infinity_semaphore)
            return

        semaphores = []
        if self.Requirements.semaphores:
            semaphores = self.Requirements.semaphores.to_dict()['acquires']

        grid_semaphores = self.get_selenium_grid_semaphores()
        if grid_semaphores:
            semaphores = semaphores + grid_semaphores

        if semaphores:
            try:
                RepeaterKiller(self, semaphores).maybe_free_semaphores()
            except:
                logging.exception('RepeaterKiller failed')

            self.Requirements.semaphores = ctt.Semaphores(
                acquires=semaphores,
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT, ctt.Status.FINISHING)
            )

    def get_selenium_grid_semaphores(self):
        return self.browsers_config.get_semaphores()

    @in_case_of('use_overlayfs', 'execute_in_overlayfs_mode')
    def execute(self):
        self.configure_plugins()
        self.set_environments()
        if self.should_poll_dependencies:
            self.poll_dependencies()
        self.prepare()

        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key():
            self.deps()
            self.configure()
            self.execute_custom_command()
            self.test()

    def execute_in_overlayfs_mode(self):
        with GitRetryWrapper(), Node(self.Parameters.node_js_version), self.vault.ssh_key():
            with self.prepare_working_copy_context():
                self.configure_plugins()
                self.set_environments()

                with self.mount_overlayfs():
                    self.configure()
                    self.execute_custom_command()
                    self.test()

    @in_case_of('should_poll_dependencies', 'mount_overlayfs_in_poll_dependencies_mode')
    def mount_overlayfs(self):
        return self._overlayfs(lower_dirs=[self.project_sources_dir], resources=self.Parameters.build_artifacts_resources, target_dir=self.project_dir)

    def mount_overlayfs_in_poll_dependencies_mode(self):
        with DaemonProcesses(self.start_files_prefetching_processes(self.prefetch_files_config)):
            lower_dirs = [self.project_sources_dir]

            with Overlayfs(lower_dirs=lower_dirs, mount_point_symlink=self.project_dir):
                lower_dirs = list_utils.flatten(self._install_dependencies()) + lower_dirs

            with Overlayfs(lower_dirs=lower_dirs, mount_point_symlink=self.project_dir):
                lower_dirs = list_utils.flatten(self._download_resources(self.project_dir)) + lower_dirs

            self.poll_dependencies()

            lower_dirs = list_utils.flatten(super(BaseTestTask, self)._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)) + lower_dirs

            return Overlayfs(lower_dirs=lower_dirs, mount_point_symlink=self.project_dir)

    def poll_dependencies(self):
        with self.profile_action(actions_constants['WAIT_DEPENDENCIES'], 'Waiting dependencies'):
            try:
                poll.wait_for(
                    dict(
                        ensure_build_artifacts_resources_are_ready=lambda: self.ensure_build_artifacts_resources_are_ready(),
                        ensure_static_is_uploaded=lambda: self.ensure_static_is_uploaded(safe=True),
                    ),
                    interval=self.poll_dependencies_interval,
                    timeout=self.poll_dependencies_timeout
                )
            except Exception as e:
                raise TaskStop(e.message)

            flow.parallel(apply, map(lambda resource: lambda: resource.reload(), self.Parameters.build_artifacts_resources))

    def ensure_build_artifacts_resources_are_ready(self):
        build_artifacts_ids = map(lambda r: r.id, self.Parameters.build_artifacts_resources)

        resources = flow.parallel(apply, map(lambda id: lambda: sdk2.Resource.find(id=id, state=ctr.State.READY).first(), build_artifacts_ids))

        result = list_utils.every(resources, bool)

        logging.info('Build artifacts resources readiness: {}'.format(result))

        return result

    def get_conf_environ(self):
        return env.merge((
            super(BaseTestTask, self).get_conf_environ(),
            self.config.get_deep_value(['tests', 'environ'], {}),
            self.config.get_deep_value(['tests', self.tool, 'environ'], {})
        ))

    def configure_plugins(self):
        self._plugins = [
            HtmlReporter(self),
            TemplateErrorCollector(self)
        ]

    def set_environments(self):
        os.environ['YANDEX_INT_RESOURCES_PATH'] = str(self.int_resources_path)
        os.environ['SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron) if self.Parameters.use_synchrophazotron else ''
        os.environ['TUNNELER_HOST_ID'] = config.Registry().this.id
        # IPBUS_BROKER_LOGDIR нужно установить как можно раньше
        # чтобы исключить ситуацию, когда брокер уже запустился, а переменная ещё не установлена, и логи потеряны
        os.environ['IPBUS_BROKER_LOGDIR'] = str(self.int_logs_path / 'ipbus')

        os.environ['retry_limiter_enabled'] = 'true'

        # Нужно для корректной работы плагина skip-validator и external-skip
        os.environ['BUILD_BRANCH'] = self.Parameters.ref

        # Нужно для корректной работы плагина external-skip
        os.environ['ISSUE_KEYS'] = ','.join(self.Parameters.issue_keys or [])

        os.environ['json_reporter_enabled'] = 'true'
        os.environ['json_reporter_path'] = self.json_reporter_file_path

        os.environ['SANDBOX_TASK_ID'] = str(self.id)
        os.environ['html_reporter_extender_s_3_images_path'] = str(self.id)
        os.environ['html_reporter_extender_s_3_report_data_path'] = str(self.id)

        self.stat_report.set_env()

        self.html_report.set_extra_items()

        for plugin in self._plugins:
            plugin.set_env()

    def prepare(self):
        self._download_sources(self.Parameters.build_artifacts_resources, self.project_dir)

    def configure(self):
        pass

    def deps(self):
        self._install_dependencies()

    def execute_custom_command(self):
        if self.Parameters.custom_command:
            with self.profile_action(actions_constants['CUSTOM_COMMAND'], 'Executing {} custom command'.format(self.tool)):
                self.lifecycle('custom_command')

    def test(self):
        status = ctt.Status.SUCCESS

        try:
            with self.profile_action(actions_constants['TEST'], 'Running {} tests'.format(self.tool)):
                # force disable execution through the shell
                # to register pid of process rather than shell
                process = self.lifecycle('test', shell=False)
                logging.debug('process {} was finished with exit code {}'.format(self.tool, process.returncode))
        except errors.SandboxSubprocessError, e:
            status = ctt.Status.FAILURE
            raise e
        finally:
            if self.Requirements.semaphores:
                sdk2.Requirements.semaphores.release()
                logging.info('{} tests released semaphores'.format(self.tool))

            self.make_reports(status)
            self.send_statistic()

        self._validate_plugins_result()

        # TODO: fail task with reasonable error message in case of failing tests

    def _validate_plugins_result(self):
        for plugin in self._plugins:
            plugin.validate_result()

    @sdk2.header()
    def header(self):
        report = super(BaseTestTask, self).header()

        linkBuilder = LinkBuilder(self)

        report.update({
            '<h3>Tests</h3>': '<p>{}</p>'.format(
                linkBuilder.make_testcop_link()
            )
        })

        return report

    @property
    def json_reporter_file_path(self):
        return '{}-{}.json'.format(self.tool, self.Parameters.platform)

    @property
    def report_common_attributes(self):
        return {
            'tool': self.tool,
            'mod': self.mod,
            'platform': self.Parameters.platform
        }

    def _get_reports_attrs(self, status):
        reports_attrs = []

        for plugin in self._plugins:
            reports_attrs += plugin.get_reports_attrs(status, self.report_common_attributes)

        reports_attrs.extend(self.stat_report.get_reports_attrs(self.report_common_attributes))

        reports_attrs.append(dict(
            self.report_common_attributes,
            resource_path='faildump.json',
            type='faildump',
            ttl=90,
        ))

        reports_attrs.append(dict(
            self.report_common_attributes,
            resource_path=self.json_reporter_file_path,
            type='json-reporter',
            task_id=self.id,
            tags=','.join(self.Parameters.tags),
            automated=self.is_automated,
        ))

        reports_attrs.append(dict(
            self.report_common_attributes,
            resource_path=self.int_logs_path,
            type='yandex-int-logs',
            task_id=self.id,
            tool=self.tool,
            ttl=30,
        ))

        return reports_attrs

    def make_reports(self, status):
        register_reports_in_parallel = self.config.get_deep_value(['tests', self.tool, 'register_reports_in_parallel'], False)

        self.artifacts.create_reports(self._get_reports_attrs(status), parallel=register_reports_in_parallel)

    def save_coredumps(self):
        coredump_filter_pattern = self.Parameters.coredump_filter_pattern
        if not coredump_filter_pattern:
            logging.debug('Skip saving coredumps')
            return

        logging.debug('Saving coredumps with pattern %s', coredump_filter_pattern)
        coredump_filter_pattern = re.compile(coredump_filter_pattern)

        coredumps_dir = self.registry.client.tasks.coredumps_dir
        logging.debug('Checking coredumps dir %s', coredumps_dir)
        for core_name in os.listdir(coredumps_dir):
            if core_name == 'core_watcher.timestamp':
                continue

            logging.debug('Found %s', core_name)
            if coredump_filter_pattern.search(core_name) is not None:
                logging.debug('Saving %s', core_name)
                self.__save_coredump(os.path.join(coredumps_dir, core_name))

    def send_statistic(self):
        infratest_profiler.send_stat_report(self)

    def on_before_end(self, status):
        super(BaseTestTask, self).on_before_end(status)

        self.save_coredumps()

        issue_key = self.Parameters.send_comment_to_issue
        if issue_key:
            self.release.add_status_comment(issue_key, status)

    # Mostly copy-paste from sdk2.helpers.process
    def __save_coredump(self, coredump_path):
        coredump_path = os.path.abspath(coredump_path)
        logging.debug('Save coredump {}'.format(coredump_path))
        coredump_filename = os.path.basename(coredump_path)
        saved_coredump_path = str(self.path(coredump_filename))
        gzipped_coredump_path = saved_coredump_path + '.gz'
        try:
            if coredump_path != saved_coredump_path:
                shutil.copy(coredump_path, saved_coredump_path)
            subprocess.call(['gzip', '-f', saved_coredump_path])
        except OSError:
            logging.exception('Error while copying coredump {}'.format(coredump_filename))
            self.set_info('Cannot dump coredump {}'.format(coredump_filename))
            return None
        mode = stat.S_IMODE(os.stat(gzipped_coredump_path).st_mode)
        mode |= stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH
        os.chmod(gzipped_coredump_path, mode)
        coredump_resource = sdk2.Resource['CORE_DUMP'](
            self, '{} coredump'.format(coredump_filename), gzipped_coredump_path
        )
        sdk2.ResourceData(coredump_resource).ready()
        self.set_info('COREDUMP was saved as resource:{}'.format(coredump_resource.id))
        self.set_info('<hr/>', do_escape=False)
        return coredump_resource

    def _report_scp_feedback(self, state, description=''):
        report_url = self.html_report.get_html_report_url(tool=self.tool)

        if report_url is not None:
            self.scp_feedback.report_status_to_current_sha(
                context=self.github_context,
                state=state,
                description=description,
                url=report_url
            )
        else:
            super(BaseTestTask, self)._report_scp_feedback(state, description)
