# coding=utf-8
import json
import logging
import urlparse

from sandbox.common import utils
from sandbox.common.types import task as ctt

from sandbox.projects.sandbox_ci.utils.request import send_request


ARCANUM_HOST_NAME = 'a.yandex-team.ru'
ARCANUM_API_PROTOCOL = 'https'


def convert_sandbox_status_to_arcanum_status(task_status):
    """
    :param task_status: Sandbox task status
    :type task_status: str
    :return:
    """
    state = ArcanumCheckStatus.PENDING
    if task_status in ctt.Status.Group.SUCCEED:
        state = ArcanumCheckStatus.SUCCESS
    elif task_status == ctt.Status.FAILURE:
        state = ArcanumCheckStatus.FAILURE
    elif task_status in ctt.Status.Group.BREAK:
        state = ArcanumCheckStatus.ERROR
    elif task_status in ctt.Status.Group.REALEXECUTE:
        state = ArcanumCheckStatus.PENDING

    logging.debug('Task status "{}", Arcanum status "{}"'.format(task_status, state))

    return state


class ArcanumApi(object):
    def __init__(self, token=None, protocol=ARCANUM_API_PROTOCOL, host=ARCANUM_HOST_NAME, base_path='/api/v1'):
        self._url = urlparse.urlunparse((protocol, host, base_path, None, None, None))
        self._token = token
        self._headers = {
            'Content-Type': 'application/json',
            'accept': 'application/json'
        }
        if token:
            self._headers.update({'Authorization': 'OAuth {}'.format(token)})

    def _get_url(self, path):
        return '{}{}'.format(self._url, path)

    def post_request(self, path, data):
        return send_request('post', self._get_url(path), headers=self._headers, data=json.dumps(data))

    def get_request(self, path, params={}):
        return send_request('get', self._get_url(path), headers=self._headers, params=params)

    def create_check(self, review_request_id, diff_set_id, check_type, status, url, description=''):
        data = {
            'system': 'sandbox_ci',
            'type': check_type,
            'status': status,
            'description': description,
            'url': url
        }

        path = '/review-requests/{}/diff-sets/{}/checks'.format(
            review_request_id,
            diff_set_id
        )
        res = self.post_request(path, data)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)['data']

    def get_checks(self, review_request_id, diff_set_id):

        path = '/review-requests/{}/diff-sets/{}/checks'.format(
            review_request_id,
            diff_set_id
        )

        res = self.get_request(path)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)['data']

    def get_pr_info(self, review_request_id, fields='id,author,summary,description'):
        """
        Возвращает информацию о ревью-реквесте
        :param review_request_id: номер ревью-реквеста
        :type review_request_id: int
        :param fields: запрашиваемые поля ревью-реквеста
        :type fields: string
        :return: информация о ревью-реквесте
        :rtype: dict
        """
        path = '/review-requests/{}'.format(
            review_request_id
        )
        params = {
            'fields': fields
        }
        res = self.get_request(path, params)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)['data']

    def get_pr_title(self, review_request_id):
        pr = self.get_pr_info(review_request_id)

        return pr['summary']

    def send_comment(self, review_request_id, comment):
        path = '/review-requests/{}/comments'.format(review_request_id)

        res = self.post_request(path, {'content': comment})

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)['data']


class ArcanumCheckStatus(utils.Enum):
    SUCCESS = 'success'
    PENDING = 'pending'
    ERROR = 'error'
    FAILURE = 'failure'
    CANCELLED = 'cancelled'


def format_review_request_url(rr_number):
    path = 'review/{rr_number}'.format(rr_number=rr_number)

    return urlparse.urlunparse((ARCANUM_API_PROTOCOL, ARCANUM_HOST_NAME, path, None, None, None))
