# -*- coding: utf-8 -*-

import os
import traceback
import tempfile
import logging
import six
from six import reraise as raise_

from sandbox import sdk2
from sandbox.common.utils import singleton_property

from sandbox.projects.sandbox_ci.utils.list_utils import flatten
from sandbox.projects.sandbox_ci.utils.fs import mkdir, rmrf
from sandbox.sandboxsdk import process, environments
from sandbox.sdk2.vcs.git import Git
from sandbox.common.errors import TemporaryError


class BaseEnvironmentContext(object):
    def __init__(self):
        self.__env_backup = {}

    def __exit__(self, exc_type, exc_val, exc_tb):
        # unset ENV
        for k, v in six.iteritems(self.__env_backup):
            if v is not None:
                os.environ[k] = v
            elif os.environ[k]:
                del os.environ[k]

    def set_env(self, key, value):
        self.__env_backup[key] = os.environ.get(key)
        os.environ[key] = value


class Debug(BaseEnvironmentContext):
    INFINITY = 'null'

    def __init__(self, namespaces, depth=None, max_array_length=None):
        if not isinstance(namespaces, list):
            namespaces = [namespaces]

        self.namespaces = namespaces
        self.depth = depth
        self.max_array_length = max_array_length
        super(Debug, self).__init__()

    @staticmethod
    def parse_opt(value):
        """
        Converts value to string representing value as environment variable

        https://github.com/visionmedia/debug/blob/285dfe10a5c06d4a86176b54bef2d7591eedaf40/src/node.js#L57-L60
        """
        if value is False:
            return 'false'
        elif value is True:
            return 'true'
        elif isinstance(value, int):
            return str(value)
        elif value == Debug.INFINITY:
            return Debug.INFINITY
        else:
            raise ValueError('Invalid debug option value: ' + str(value))

    def __enter__(self):
        debug_env = os.environ.get('DEBUG')
        namespaces = self.namespaces
        if debug_env is not None:
            namespaces = [debug_env] + namespaces
        self.set_env('DEBUG', ','.join(namespaces))
        if self.depth is not None:
            self.set_env('DEBUG_DEPTH', self.parse_opt(self.depth))
        if self.max_array_length is not None:
            self.set_env('DEBUG_MAX_ARRAY_LENGTH', self.parse_opt(self.max_array_length))


class Node(BaseEnvironmentContext):
    """
    Context manager for using nvm.

    Usage:
        with Node('current'): # for current version or pass empty string
            ...
        with Node(6): # for node@6.x
            ...
    """

    def __init__(self, node_version):
        if not node_version:
            node_version = 'current'

        self._node_version = os.environ.get('NODEJS', node_version)
        super(Node, self).__init__()

    def __enter__(self):
        # set ENV
        nvm_dir = '/opt/nvm'
        self.set_env('NVM_DIR', nvm_dir)
        self.set_env('NO_UPDATE_NOTIFIER', 'true')

        # source nvm
        stdout, stderr = process.run_process(
            ['. {}/nvm.sh && nvm which {}'.format(nvm_dir, self._node_version)],
            shell=True,
            outs_to_pipe=True,
            check=True).communicate()
        if stderr:
            logging.debug(stderr)

        logging.debug(stdout)
        node_path = os.path.dirname(stdout)

        # set ENV
        self.set_env('PATH', '{}:{}'.format(node_path, os.environ['PATH']))


class GitRetryWrapper(BaseEnvironmentContext):
    """
    Context manager for using `git retry` transparently for any `git` invocation.

    Usage:
        with GitRetryWrapper():
            ...
    """

    def __enter__(self):
        # set ENV
        self.set_env('PATH', ':'.join(['/opt/git-retry-wrapper', '/opt/depot_tools', os.environ['PATH']]))


class GitWithoutLfsProcessing(object):
    """
    Context manager for disabling Git LFS processing.

    Usage:
        with GitWithoutLfsProcessing():
            ...
    """

    def __init__(self, git_url):
        """
        :param git_url: Path to git repository
        :type git_url: str
        """
        self._git = Git(git_url)
        super(GitWithoutLfsProcessing, self).__init__()

    def git_exec(self, *args, **kwargs):
        git_process = self._git.raw_execute(args, shell=True, **kwargs)

        return git_process.communicate()[0].strip()

    def __enter__(self):
        logging.debug('entering to "GitWithoutLfsProcessing" context')

        self._git_filter_lfs_required = self.git_exec('config', '--global', '--get', 'filter.lfs.required')
        self._git_filter_lfs_process = self.git_exec('config', '--global', '--get', 'filter.lfs.process')

        logging.debug('git config filter.lfs.required value: "{}"'.format(self._git_filter_lfs_required))
        logging.debug('git config filter.lfs.process value: "{}"'.format(self._git_filter_lfs_process))

        self.git_exec('config', '--global', 'filter.lfs.required', 'false')
        self.git_exec('config', '--global', 'filter.lfs.process', "''")

    def __exit__(self, exc_type, exc_val, exc_tb):
        logging.debug('exiting from "GitWithoutLfsProcessing" context')

        if self._git_filter_lfs_required is None or self._git_filter_lfs_required == '':
            self.git_exec('config', '--global', '--unset', 'filter.lfs.required')
        else:
            self.git_exec('config', '--global', 'filter.lfs.required', "'{}'".format(self._git_filter_lfs_required))

        if self._git_filter_lfs_process is None or self._git_filter_lfs_process == '':
            self.git_exec('config', '--global', '--unset', 'filter.lfs.process')
        else:
            self.git_exec('config', '--global', 'filter.lfs.process', "'{}'".format(self._git_filter_lfs_process))


class BowerNpmInjection(BaseEnvironmentContext):
    """
    Context manager for using `npm` wrapper transparently for any `npm` invocation.

    Usage:
        with BowerNpmInjection():
            ...
    """

    def __init__(self, bin_dir):
        self._bin_dir = bin_dir
        super(BowerNpmInjection, self).__init__()

    def __enter__(self):
        build_cache_dir = environments.SandboxEnvironment.build_cache_dir

        # set ENV
        self.set_env('PATH', '{}:{}'.format(self._bin_dir, os.environ['PATH']))
        self.set_env('NPM_CACHE_ROOT', os.path.join(build_cache_dir, 'npm'))
        self.set_env('BOWER_CACHE_ROOT', os.path.join(build_cache_dir, 'bower'))


class TaskRetryContext(object):
    """
    Context that automatically restarts task
    using TemporaryError exception, if an error occurs in its block.
    """

    def __init__(self, task):
        self.__task = task

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is not None:
            error_traceback = traceback.format_exc()
            self.__task.Context.noncritical_errors.append(error_traceback)
            raise_(TemporaryError, 'Task will be restarted, because an error occurred during its execution', exc_tb)


class Overlayfs(object):
    def __init__(self, lower_dirs, mount_point=None, mount_point_symlink=None):
        self.__lower_dirs = lower_dirs
        self.__mount_point = mount_point
        self.__mount_point_symlink = mount_point_symlink

    @singleton_property
    def lower_dirs(self):
        return map(lambda i: os.readlink(i) if os.path.islink(i) else i, map(str, flatten(self.__lower_dirs)))

    @singleton_property
    def mount_point(self):
        return mkdir(str(self.__mount_point)) if self.__mount_point else tempfile.mkdtemp()

    @singleton_property
    def mount_point_symlink(self):
        return str(self.__mount_point_symlink)

    @singleton_property
    def work_dir(self):
        return tempfile.mkdtemp()

    def __enter__(self):
        upper_dir = tempfile.mkdtemp()

        sdk2.os.mount_overlay(self.mount_point, self.lower_dirs, upper_dir=upper_dir, work_dir=self.work_dir)
        self.mount_point_symlink and os.symlink(self.mount_point, self.mount_point_symlink)

        return upper_dir if self.__mount_point else (upper_dir, self.mount_point)

    def __exit__(self, exc_type, exc_val, exc_tb):
        # https://st.yandex-team.ru/FEI-15971 – 'sdk2.os.umount' может упасть с ошибкой о невозможности размонтирования
        try:
            sdk2.os.umount(self.mount_point)
            rmrf(self.work_dir)
            rmrf(self.mount_point)
        except:
            logging.debug('Cannot umount {}, reason:\n{}'.format(self.mount_point, traceback.format_exc()))

        self.mount_point_symlink and os.unlink(self.mount_point_symlink)


class DaemonProcesses(object):
    def __init__(self, processes):
        self._processes = processes

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_val, exc_tb):
        map(lambda p: p.terminate(), self._processes)


class DirContext(object):
    def __init__(self, target_dir):
        self._target_dir = target_dir

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_val, exc_tb):
        rmrf(self._target_dir)
