# -*- coding: utf-8 -*-

import os
import json
import logging

from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.sandbox_ci.utils.process import run_process
from sandbox.projects.sandbox_ci.utils.startrek import parse_issue_keys
from sandbox.projects.sandbox_ci.utils.context import GitRetryWrapper


def git_clone(url, ref, target_dir='.'):
    git = Git(url)
    git.clone(str(target_dir), ref)


def git_cache(url):
    git = Git(url)
    git.touch_cache()
    return sdk2.Path(git.cache_repo_dir)


def save_git_changed_files(work_dir, file_path):
    logging.debug('Saving changed files for {} to {}'.format(work_dir, file_path))
    with GitRetryWrapper():
        run_process(
            ['git diff --name-only --no-renames HEAD~1', '>', str(file_path)],
            work_dir=work_dir,
            log_prefix='save_git_changed_files'
        )


def save_git_changed_files_with_statuses(work_dir, file_path):
    logging.debug('Saving changed files with statuses for last commit in {} to {}'.format(work_dir, file_path))
    with GitRetryWrapper():
        p = run_process(
            ['git diff --name-status --no-renames HEAD~1'],
            work_dir=work_dir,
            log_prefix='save_git_changed_files_with_statuses',
        )

    with open(p.stdout_path, 'r') as f:
        data = f.readlines()

    res = []
    for line in data:
        status = line[0]
        path = line[1:].strip()
        file = {'path': path}

        if status == 'A':
            file['status'] = 'new file'
        elif status == 'D':
            file['status'] = 'deleted'
        else:
            file['status'] = 'modified'

        res.append(file)

    with open(os.path.join(str(work_dir), file_path), 'w') as f:
        json.dump(res, f)


def get_git_last_merge_commit_issues(work_dir):
    logging.debug('Getting last merge commit issues for {}'.format(work_dir))
    with GitRetryWrapper():
        p = run_process(
            ['git log -n 1 --no-color'],
            work_dir=work_dir,
            log_prefix='git_log_merge_commit'
        )

    with open(p.stdout_path, 'r') as f:
        return parse_issue_keys(f.readlines())
