# coding=utf-8
import re
import json
import logging
import urlparse

from sandbox.common import utils
from sandbox.common.types import task as ctt

from sandbox.projects.sandbox_ci.utils.request import send_request, make_request, RETRY_METHODS

# Максимальное количество запросов к api для получения списка коммитов
MAX_GET_COMMITS_REQUESTS = 10
GITHUB_HOST_NAME = 'github.yandex-team.ru'
GITHUB_API_PROTOCOL = 'https'


class GitHubApi(object):
    def __init__(self, token=None, protocol=GITHUB_API_PROTOCOL, host=GITHUB_HOST_NAME, port=443, base_path='/api/v3'):
        netloc = '{host}:{port}'.format(host=host, port=port)
        self._url = urlparse.urlunparse((protocol, netloc, base_path, None, None, None))
        self._token = token

        if token:
            self._headers = {
                'Authorization': 'token {}'.format(token),
                'Content-Type': 'application/json'
            }

    def __get_request_url_by_path(self, path):
        return '{}/{}'.format(self._url, path)

    def request(self, path, **kwargs):
        url = self.__get_request_url_by_path(path)

        res = send_request('get', url, **kwargs)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)

    def __http_send_request(self, method, url, data, **kwargs):
        headers = {}

        if self._token:
            headers.update({
                'Authorization': 'token {}'.format(self._token),
                'Content-Type': 'application/json'
            })

        return send_request(method, url, headers=headers, data=json.dumps(data), max_retries=6, **kwargs)

    def __http_send_request_post(self, path, data, **kwargs):
        url = self.__get_request_url_by_path(path)

        return self.__http_send_request('post', url, data, **kwargs)

    @staticmethod
    def __post_request_with_retries(method, url, headers, max_retries, retry_methods=RETRY_METHODS):
        factor = 0.8
        methods = frozenset(list(retry_methods) + ['POST'])

        return make_request(method, url, headers, max_retries, backoff_factor=factor, retry_methods=methods)

    def create_status(self, owner, repo, context, sha, state, url, description=''):
        data = {
            'context': context,
            'state': state,
            'description': description,
            'target_url': url
        }

        path = 'repos/{}/{}/statuses/{}'.format(owner, repo, sha)

        res = self.__http_send_request_post(path, data, make_request=self.__post_request_with_retries)

        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}

    def get_latest_merge_commits_tree_hashes(self, owner, repo, num=5):
        """
        Возвращает список tree_hash мерж-коммитов
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param num: необходимое количество мерж-коммитов
        """
        merge_commits = []
        sha = None

        for index in range(0, MAX_GET_COMMITS_REQUESTS):
            commits = self.get_commits(owner, repo, sha)
            # Не включаем стартовый коммит в результаты
            if sha:
                commits = commits[1:]
            # Мерж-коммиты пул-реквестов можно определить по длине массива parents
            merge_commits += filter(lambda c: len(c['parents']) == 2, commits)

            if len(merge_commits) >= num:
                break

            sha = commits[-1]['sha']

        return map(lambda c: c['commit']['tree']['sha'], merge_commits[:num])

    def get_pr_info(self, owner, repo, pr_number):
        """
        Возвращает информацию о пулл-реквесте
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        :param pr_number: номер пулл-реквеста
        :type pr_number: int
        :return: информацию о пулл-реквесте (@see https://developer.github.com/v3/pulls/#get-a-single-pull-request)
        :rtype: dict
        """
        path = 'repos/{owner}/{repo}/pulls/{pr_number}'.format(
            owner=owner,
            repo=repo,
            pr_number=pr_number
        )

        url = self.__get_request_url_by_path(path)

        res = send_request('get', url)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)

    def get_pr_title(self, owner, repo, pr_number):
        pr = self.get_pr_info(owner, repo, pr_number)

        return pr['title']

    def get_commits(self, owner, repo, sha=None):
        """
        Возвращает список коммитов
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param sha: хеш коммита (если передан, то все коммиты до него будут пропущены)
        :return: список коммитов (@see https://developer.github.com/v3/repos/commits/#list-commits-on-a-repository)
        :rtype: list
        """
        path = 'repos/{}/{}/commits'.format(owner, repo)
        if sha is not None:
            path += '?sha=' + sha

        url = self.__get_request_url_by_path(path)

        res = send_request('get', url)

        assert res.status_code == 200, {'status_code': res.status_code, 'text': res.text}

        return json.loads(res.text)

    def get_content(self, owner, repo, path):
        """
        Возвращает содержимое указанного файла в репозитории
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param path: путь до файла без начального слеша
        :return: содержимое файла
        :rtype: str|None
        """
        path = 'repos/{}/{}/contents/{}'.format(owner, repo, path)
        url = self.__get_request_url_by_path(path)
        res = send_request('get', url, headers={'Accept': 'application/vnd.github.v3.raw'})

        assert res.status_code in [200, 404], {'status_code': res.status_code, 'text': res.text}

        return res.text if res.status_code == 200 else None

    def list_pull_request_files(self, owner, repo, pr_number):
        """
        Возвращает список измененных файлов в пулл-реквесте
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param pr_number: номер пулл-реквеста
        :return: список измененных файлов в пулл-реквесте
        :rtype: list
        """
        path = 'repos/{owner}/{repo}/pulls/{pr_number}/files'.format(
            owner=owner,
            repo=repo,
            pr_number=pr_number
        )

        return map(lambda i: i['filename'], self.request(path))

    def get_branch(self, owner, repo, branch):
        """
        Возвращает информацию о ветке
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param branch: название ветки
        :return: информация о ветке
        :rtype: dict
        """
        path = 'repos/{owner}/{repo}/branches/{branch}'.format(owner=owner, repo=repo, branch=branch)

        return self.request(path)

    def list_required_checks(self, owner, repo, branch):
        """
        Возвращает список обязательных проверок для ветки
        :param owner: владелец репозитория
        :param repo: имя репозитория
        :param branch: название ветки
        :return: список обязательных проверок
        :rtype: list
        """
        return self.get_branch(owner, repo, branch)['protection']['required_status_checks']['contexts']

    def create_ref(self, owner, repo, ref, sha):
        """
        Создает сслыку на коммит
        :param owner: владелец репозитория
        :type owner: str
        :param repo: имя репозитория
        :type repo: str
        :param ref: название ссылки
        :type ref: str
        :param sha: хэш коммита
        :type sha: str
        :return: информация о созданной ссылке
        :rtype: dict
        """
        path = 'repos/{}/{}/git/refs'.format(owner, repo)
        data = {
            'ref': ref,
            'sha': sha
        }

        res = self.__http_send_request_post(path, data, make_request=self.__post_request_with_retries)

        assert res.status_code == 201, {'status_code': res.status_code, 'text': res.text}

        return res


class GitHubStatus(utils.Enum):
    SUCCESS = 'success'
    PENDING = 'pending'
    ERROR = 'error'
    FAILURE = 'failure'


def parse_github_url(url):
    """
    @param url:
    @type url: str
    @return: Dictionary with domain, owner and repo
    @rtype: dict
    """
    parsed_info = {}
    patterns = {
        'https': r'https://(?P<domain>.+)/(?P<owner>.+)/(?P<repo>.+).git',
        'ssh': r'git@(?P<domain>.+):(?P<owner>.+)/(?P<repo>.+).git',
        'git': r'git://(?P<domain>.+)/(?P<owner>.+)/(?P<repo>.+).git',
        'git+ssh': r'git\+ssh://(?P<domain>.+)/(?P<owner>.+)/(?P<repo>.+).git'
    }

    for protocol, regex in patterns.iteritems():
        match = re.compile(regex).match(url)

        if not match:
            continue

        matches = match.groupdict()
        parsed_info.update(matches)

        break

    if not parsed_info:
        raise Exception('Url {} is not valid git url'.format(url))

    return parsed_info


def convert_sandbox_status_to_github_state(task_status):
    """
    :param task_status: Sandbox task status
    :type task_status: str
    :return:
    """
    state = GitHubStatus.PENDING
    if task_status in ctt.Status.Group.SUCCEED:
        state = GitHubStatus.SUCCESS
    elif task_status == ctt.Status.FAILURE:
        state = GitHubStatus.FAILURE
    elif task_status in ctt.Status.Group.BREAK:
        state = GitHubStatus.ERROR
    elif task_status in ctt.Status.Group.REALEXECUTE:
        state = GitHubStatus.PENDING

    logging.debug('Task status "{}", GitHub state "{}"'.format(task_status, state))

    return state


def format_pull_request_url(owner, repo, pr_number):
    path = '{owner}/{repo}/pull/{pr_number}'.format(owner=owner, repo=repo, pr_number=pr_number)

    return urlparse.urlunparse((GITHUB_API_PROTOCOL, GITHUB_HOST_NAME, path, None, None, None))
