# coding=utf-8
import re
import six

from sandbox import sdk2
from sandbox.sandboxsdk import process


# Copy-Paste from python 3.6 shlex, adapted for python 2 (regexps being ascii by default)
_find_unsafe = re.compile(r'[^\w@%+=:,./-]').search


# Copy-Paste from python 3.6 shlex
def quote(s):
    """Return a shell-escaped version of the string *s*."""
    if not s:
        return "''"
    if _find_unsafe(s) is None:
        return s
    # use single quotes, and put single quotes into double quotes
    # the string $'b is then quoted as '$'"'"'b'
    return "'" + s.replace("'", "'\"'\"'") + "'"


def run_process(command_args, shell=True, use_equal=False,  **kwargs):
    """
    :param command_args: массив из команды и аргументов
    :type command_args: iterable
    :param use_equal: использовать '=' как delimiter для options, например --file=some если стоит True
    :type use_equal: bool
    :param shell: запускать ли как shell команду
    :type shell: bool
    :return: объект запущенного процесса (subprocess.Popen)
    """
    for key, val in six.iteritems(kwargs):
        if isinstance(val, sdk2.Path):
            kwargs[key] = str(val)

    return process.run_process(
        format_args_for_run_process(command_args, use_equal=use_equal),
        shell=shell,
        **kwargs
    )


def run_process_with_custom_log(command_args, log_path, shell=True, **kwargs):
    """
    :param command_args: массив из команды и аргументов
    :type command_args: iterable
    :type shell: bool
    :type log_path: str
    """
    for key, val in six.iteritems(kwargs):
        if isinstance(val, sdk2.Path):
            kwargs[key] = str(val)

    with open(log_path, 'w') as log:
        process.run_process(
            format_args_for_run_process(command_args),
            stdout=log,
            stderr=log,
            shell=shell,
            **kwargs
        )


def format_args_for_run_process(command_args, use_equal=False):
    """
    :param command_args: массив из команды и аргументов
    :type command_args: iterable
    :param use_equal: использовать '=' как delimiter для options, например --file=some если стоит True
    :type use_equal: bool
    :return: argument for run_process function
    :rtype: list
    """
    command_arguments = []
    for arg in command_args:
        if isinstance(arg, (six.binary_type, six.text_type)):
            command_arguments.append(arg)
        elif isinstance(arg, sdk2.Path):
            command_arguments.append(str(arg))
        elif isinstance(arg, dict):
            command_arguments += format_options(arg, use_equal)
        else:
            raise Exception('{} is not supported type'.format(type(arg)))

    return [arg.encode('utf8') for arg in command_arguments]


def format_option(opt_key, opt_val, use_equal=False):
    """
    :type opt_key: unicode
    :type opt_val: unicode
    :type use_equal: bool
    :param use_equal: использовать '=' как delimiter для options, например --file=some если стоит True
    :rtype: unicode
    """
    val = u'{}'.format(opt_val).replace("'", "'\\''")

    return u'{key}{delimiter}\'{val}\''.format(
        key=opt_key,
        delimiter='=' if use_equal else ' ',
        val=val
    )


def format_options(options, use_equal=False):
    """
    :type options: dict
    :param use_equal: использовать '=' как delimiter для options, например --file=some если стоит True
    :type use_equal: bool
    :return:
    :rtype: list of str
    """
    command_arguments = []
    for key, val in six.iteritems(options):
        if not val:
            continue

        opt_key = u'--{}'.format(key)
        if len(key) == 1:
            opt_key = u'-{}'.format(key)

        if val is True:
            command_arguments.append(opt_key)
        else:
            if isinstance(val, sdk2.Path):
                val = six.text_type(val)

            if isinstance(val, six.binary_type):
                val = val.decode('utf8')

            if isinstance(val, list) or isinstance(val, tuple):
                for opt_val in val:
                    command_arguments.append(format_option(opt_key, opt_val, use_equal))
            else:
                command_arguments.append(format_option(opt_key, val, use_equal))

    return command_arguments
