import logging
import os
import subprocess

import sandbox.common.types.client as ctc
import sandbox.projects.common.environments as env
from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt

logger = logging.getLogger(__name__)


class GeneratedSound(sdk2.Resource):
    """ Audio file with Alice voiced text """
    sync_upload_to_mds = True


class GenerateTtsAlice(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # Require 1 CPU core
        ram = 1024  # Require 1024 Mib RAM
        # Download specific resources to task
        client_tags = ctc.Tag.LINUX_XENIAL
        resources = [
            3342809651,
        ]
        environments = [env.SandboxNodeNpmEnvironment('14.18.2')]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Task to generate alice audio for scooter audio package.'
        max_restarts = 2
        kill_timeout = 5 * 60
        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT,
                ],
                recipients=['vdovkin@yandex-team.ru'],
                transport=ctn.Transport.EMAIL,
            ),
        ]
        owner = 'vdovkin'
        tags = ['scooters']

        text = sdk2.parameters.String('Text for alice voice over', required=True, multiline=True)
        apikey = sdk2.parameters.YavSecret(
            "YAV secret identifier of api key for Yandex SpeechKit https://developer.tech.yandex.ru/services/6",
            default="sec-01g8ekn46agjps7gxjc8cy0e56#scooters_speechkit_key",
            required=True,
        )

        with sdk2.parameters.Output():
            audio_file = sdk2.parameters.Resource(
                'Generated audio file with Alice voiced text',
            )

    @staticmethod
    def get_path(resource):
        return str(sdk2.ResourceData(resource).path)

    def on_execute(self):
        tts_script_data = sdk2.ResourceData(sdk2.Resource[3342809651])
        tts_script_path = str(tts_script_data.path / 'supertost.js')
        output_audio_file = 'output_audio.wav'

        os.environ["NODE_TLS_REJECT_UNAUTHORIZED"] = "0"
        cmd = [
            'node', tts_script_path,
            '--filename', output_audio_file,
            '--text', str(self.Parameters.text),
            '--key', self.Parameters.apikey.data()[self.Parameters.apikey.default_key],
        ]
        logger.info('Running audio generation script')
        with sdk2.helpers.ProcessLog(self, logger="audio-generation-script-log") as pl:
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)

        self.Parameters.audio_file = GeneratedSound(
            self, output_audio_file, output_audio_file)
        sdk2.ResourceData(self.Parameters.audio_file).ready()
