import logging
import os
import subprocess

import sandbox.common.types.client as ctc
import sandbox.projects.common.environments as env
from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.projects.scooters.generate_tts_alice import GeneratedSound
from sandbox.projects.scooters.ninebot_audio_package import NinebotAudioBitrateSelector
from sandbox.projects.scooters.ninebot_audio_package import NinebotAudioPackage
from sandbox.projects.scooters.ninebot_audio_package import NinebotAudioPackageName
from sandbox.projects.scooters.ninebot_audio_package import NinebotAudioPackageVersion


logger = logging.getLogger(__name__)


class GenerateTtsNinebotAudioPackage(sdk2.Task):
    class Context(sdk2.Task.Context):
        package_child_task_id = 0

    class Requirements(sdk2.Requirements):
        cores = 1  # Require 1 CPU core
        ram = 1024  # Require 1024 Mib RAM
        # Download specific resources to task
        client_tags = ctc.Tag.LINUX_XENIAL
        resources = [
            3342809651,
        ]
        environments = [env.SandboxNodeNpmEnvironment('14.18.2')]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Task to generate alice audio and build audio package for Ninebot v2.'
        max_restarts = 2
        kill_timeout = 5 * 60
        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT,
                ],
                recipients=['vdovkin@yandex-team.ru'],
                transport=ctn.Transport.EMAIL,
            ),
        ]
        owner = 'vdovkin'
        tags = ['scooters']

        with sdk2.parameters.Group('SpeechKit') as speechkit:
            apikey = sdk2.parameters.YavSecret(
                "YAV secret identifier of api key for Yandex SpeechKit https://developer.tech.yandex.ru/services/6",
                default="sec-01g8ekn46agjps7gxjc8cy0e56#scooters_speechkit_key",
                required=True,
            )

        with sdk2.parameters.Group('Package info') as package_meta:
            package_version = NinebotAudioPackageVersion()
            package_name = NinebotAudioPackageName()
            bitrate = NinebotAudioBitrateSelector()

        with sdk2.parameters.Group(
            'Text of scooter system sounds (IOT trigger playback, can\'t be used in V0 command)',
        ) as sounds_text:
            SFX0_power_on_text = sdk2.parameters.String('Power on', required=True)
            SFX1_power_off_text = sdk2.parameters.String('Power off', required=True)
            SFX2_unlock_text = sdk2.parameters.String('Unlock', required=True)
            SFX3_lock_text = sdk2.parameters.String('Lock', required=True)
            SFX7_illegal_moving_text = sdk2.parameters.String('Illegal moving', required=True)

        with sdk2.parameters.Group(
            'Text of custom sounds (server triggered playback, index for playing through V0 command)',
        ) as custom_sounds_text:
            SFX5_battery_low_text = sdk2.parameters.String('1: Battery low', required=True)
            SFX8_locating_text = sdk2.parameters.String('2: Locating', required=True)
            SFX9_alarm_text = sdk2.parameters.String(
                '3: Alarm for service finding (leave empty for default)', default=None)
            SFX6_empty_text = sdk2.parameters.String(
                '6: Tips for driving into geofence (Unused) (leave empty for default)', default=None)
            SFX13_slowdown_20_text = sdk2.parameters.String('91: Slowing down to 20', required=True)
            SFX14_slowdown_15_text = sdk2.parameters.String('92: Slowing down to 15', required=True)
            SFX15_slowdown_10_text = sdk2.parameters.String('93: Slowing down to 10', required=True)
            SFX16_slowdown_0_text = sdk2.parameters.String('94: Disable acceleration', required=True)
            SFX17_opened_hood_text = sdk2.parameters.String('95: Battery hood is opened', required=True)
            SFX18_falling_ground_text = sdk2.parameters.String('96: Scooter is fallen', required=True)
            SFX19_lifted_up_text = sdk2.parameters.String('97: Scooter has been lifted', required=True)
            SFX20_broken_text = sdk2.parameters.String('98: Scooter is broken', required=True)
            SFX21_empty_text = sdk2.parameters.String('99: Empty (can be used)', default=None)

        with sdk2.parameters.Group(
            'Text of unused scooter system sounds (cant be used in V0 command, leave empty for default)',
        ) as unused_system_sounds_text:
            SFX4_empty_text = sdk2.parameters.String(
                'Timeout prompt for communication with vehicle (Unused) (leave empty for default)', default=None)
            SFX10_empty_text = sdk2.parameters.String(
                'SIM card loss alarm (Unused) (leave empty for default)', default=None)
            SFX11_empty_text = sdk2.parameters.String(
                'Tips for opening the battery compartment cover (Unused) (leave empty for default)', default=None)
            SFX12_empty_text = sdk2.parameters.String(
                'Vehicle configuration succeeded (Unused) (leave empty for default)', default=None)

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Ready to use Ninebot v2 audio package') as out_group:
                audio_package_bin = sdk2.parameters.Resource(
                    'Ninebot audio package binary',
                )
                audio_package_checksum = sdk2.parameters.Integer(
                    'Ninebot audio package checksum',
                )

            with sdk2.parameters.Group(
                'Scooter system sounds (IOT trigger playback, can\'t be used in V0 command)',
            ) as sounds_audio:
                SFX0_power_on_audio = sdk2.parameters.Resource('Power on')
                SFX1_power_off_audio = sdk2.parameters.Resource('Power off')
                SFX2_unlock_audio = sdk2.parameters.Resource('Unlock')
                SFX3_lock_audio = sdk2.parameters.Resource('Lock')
                SFX7_illegal_moving_audio = sdk2.parameters.Resource('Illegal moving')

            with sdk2.parameters.Group(
                'Custom sounds (server triggered playback, index for playing through V0 command)',
            ) as custom_sounds_audio:
                SFX5_battery_low_audio = sdk2.parameters.Resource('1: Battery low')
                SFX8_locating_audio = sdk2.parameters.Resource('2: Locating')
                SFX9_alarm_audio = sdk2.parameters.Resource('3: Alarm for service finding')
                SFX6_empty_audio = sdk2.parameters.Resource('6: Tips for driving into geofence (Unused, can be used)')
                SFX13_slowdown_20_audio = sdk2.parameters.Resource('91: Slowing down to 20')
                SFX14_slowdown_15_audio = sdk2.parameters.Resource('92: Slowing down to 15')
                SFX15_slowdown_10_audio = sdk2.parameters.Resource('93: Slowing down to 10')
                SFX16_slowdown_0_audio = sdk2.parameters.Resource('94: Disable acceleration')
                SFX17_opened_hood_audio = sdk2.parameters.Resource('95: Battery hood is opened')
                SFX18_falling_ground_audio = sdk2.parameters.Resource('96: Scooter is fallen')
                SFX19_lifted_up_audio = sdk2.parameters.Resource('97: Scooter has been lifted')
                SFX20_broken_audio = sdk2.parameters.Resource('98: Scooter is broken')
                SFX21_empty_audio = sdk2.parameters.Resource('99: Unused custom sound #9 (can be used)')

            with sdk2.parameters.Group(
                'Unused scooter system sounds (can\'t be used in V0 command)',
            ) as unused_system_sounds_audio:
                SFX4_empty_audio = sdk2.parameters.Resource('Timeout prompt for communication with vehicle (Unused)')
                SFX10_empty_audio = sdk2.parameters.Resource('SIM card loss alarm (Unused)')
                SFX11_empty_audio = sdk2.parameters.Resource('Tips for opening the battery compartment cover (Unused)')
                SFX12_empty_audio = sdk2.parameters.Resource('Vehicle configuration succeeded (Unused)')

    @staticmethod
    def get_path(resource):
        return str(sdk2.ResourceData(resource).path)

    def on_execute(self):
        tts_script_data = sdk2.ResourceData(sdk2.Resource[3342809651])
        tts_script_path = str(tts_script_data.path / 'supertost.js')
        os.environ["NODE_TLS_REJECT_UNAUTHORIZED"] = "0"

        text_to_output = [
            (self.Parameters.SFX0_power_on_text, 'SFX0_power_on_audio'),
            (self.Parameters.SFX1_power_off_text, 'SFX1_power_off_audio'),
            (self.Parameters.SFX2_unlock_text, 'SFX2_unlock_audio'),
            (self.Parameters.SFX3_lock_text, 'SFX3_lock_audio'),
            (self.Parameters.SFX4_empty_text, 'SFX4_empty_audio'),
            (self.Parameters.SFX5_battery_low_text, 'SFX5_battery_low_audio'),
            (self.Parameters.SFX6_empty_text, 'SFX6_empty_audio'),
            (self.Parameters.SFX7_illegal_moving_text, 'SFX7_illegal_moving_audio'),
            (self.Parameters.SFX8_locating_text, 'SFX8_locating_audio'),
            (self.Parameters.SFX9_alarm_text, 'SFX9_alarm_audio'),
            (self.Parameters.SFX10_empty_text, 'SFX10_empty_audio'),
            (self.Parameters.SFX11_empty_text, 'SFX11_empty_audio'),
            (self.Parameters.SFX12_empty_text, 'SFX12_empty_audio'),
            (self.Parameters.SFX13_slowdown_20_text, 'SFX13_slowdown_20_audio'),
            (self.Parameters.SFX14_slowdown_15_text, 'SFX14_slowdown_15_audio'),
            (self.Parameters.SFX15_slowdown_10_text, 'SFX15_slowdown_10_audio'),
            (self.Parameters.SFX16_slowdown_0_text, 'SFX16_slowdown_0_audio'),
            (self.Parameters.SFX17_opened_hood_text, 'SFX17_opened_hood_audio'),
            (self.Parameters.SFX18_falling_ground_text, 'SFX18_falling_ground_audio'),
            (self.Parameters.SFX19_lifted_up_text, 'SFX19_lifted_up_audio'),
            (self.Parameters.SFX20_broken_text, 'SFX20_broken_audio'),
            (self.Parameters.SFX21_empty_text, 'SFX21_empty_audio'),
        ]
        default_resources = {
            'SFX4_empty_audio': 3336620363,
            'SFX6_empty_audio': 3336624495,
            'SFX9_alarm_audio': 3336629252,
            'SFX10_empty_audio': 3336630796,
            'SFX11_empty_audio': 3336632198,
            'SFX12_empty_audio': 3336634535,
            'SFX21_empty_audio': 3336634535,
        }

        for text, output_name in text_to_output:
            with self.memoize_stage['tts-step-' + output_name](commit_on_entrance=False):
                if not text:
                    if output_name in default_resources:
                        setattr(self.Parameters, output_name, sdk2.Resource[default_resources[output_name]])
                    continue
                output_audio_file = output_name + '.wav'
                cmd = [
                    'node', tts_script_path,
                    '--filename', output_audio_file,
                    '--text', text,
                    '--key', self.Parameters.apikey.data()[self.Parameters.apikey.default_key],
                ]
                logger.info('Running audio generation script for ' + output_audio_file)
                with sdk2.helpers.ProcessLog(self, logger="audio-generation-script-log") as pl:
                    subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stdout)
                setattr(self.Parameters, output_name, GeneratedSound(self, output_audio_file, output_audio_file))
                sdk2.ResourceData(getattr(self.Parameters, output_name)).ready()

        with self.memoize_stage.generate_ninebot_package(commit_on_entrance=False):
            audio_package_task = NinebotAudioPackage(
                self,
                description="Child task of {}".format(self.id),
                owner=self.owner,
                package_version=self.Parameters.package_version,
                package_name=self.Parameters.package_name,
                bitrate=self.Parameters.bitrate,
                SFX0_power_on=self.Parameters.SFX0_power_on_audio,
                SFX1_power_off=self.Parameters.SFX1_power_off_audio,
                SFX2_unlock=self.Parameters.SFX2_unlock_audio,
                SFX3_lock=self.Parameters.SFX3_lock_audio,
                SFX4_empty=self.Parameters.SFX4_empty_audio,
                SFX5_battery_low=self.Parameters.SFX5_battery_low_audio,
                SFX6_empty=self.Parameters.SFX6_empty_audio,
                SFX7_illegal_moving=self.Parameters.SFX7_illegal_moving_audio,
                SFX8_locating=self.Parameters.SFX8_locating_audio,
                SFX9_alarm=self.Parameters.SFX9_alarm_audio,
                SFX10_empty=self.Parameters.SFX10_empty_audio,
                SFX11_empty=self.Parameters.SFX11_empty_audio,
                SFX12_empty=self.Parameters.SFX12_empty_audio,
                SFX13_slowdown_20=self.Parameters.SFX13_slowdown_20_audio,
                SFX14_slowdown_15=self.Parameters.SFX14_slowdown_15_audio,
                SFX15_slowdown_10=self.Parameters.SFX15_slowdown_10_audio,
                SFX16_slowdown_0=self.Parameters.SFX16_slowdown_0_audio,
                SFX17_opened_hood=self.Parameters.SFX17_opened_hood_audio,
                SFX18_falling_ground=self.Parameters.SFX18_falling_ground_audio,
                SFX19_lifted_up=self.Parameters.SFX19_lifted_up_audio,
                SFX20_broken=self.Parameters.SFX20_broken_audio,
                SFX21_empty=self.Parameters.SFX21_empty_audio,
            )
            audio_package_task.save().enqueue()
            self.Context.package_child_task_id = audio_package_task.id
            raise sdk2.WaitOutput({audio_package_task.id: ('audio_package_bin', 'audio_package_checksum')},
                                  wait_all=True)

        child_id = self.Context.package_child_task_id
        self.Parameters.audio_package_bin = sdk2.Task[child_id].Parameters.audio_package_bin
        self.Parameters.audio_package_checksum = sdk2.Task[child_id].Parameters.audio_package_checksum
