import ctypes
import logging
import subprocess

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task


logger = logging.getLogger(__name__)


class NinebotAudioPackageBinary(sdk2.Resource):
    """ Ninebot audio package binary """


class NinebotAudioBitrateSelector(sdk2.parameters.String):
    name = 'bitrate'
    description = 'Bitrate of audio in package'
    required = True
    default_value = '16kbps'
    choices = [(item, item) for item in (
        '14kbps', '16kbps', '18kbps', '20kbps',
        '24kbps', '28kbps', '32kbps', '34kbps',
    )]
    ui = sdk2.parameters.String.UI("select")


class NinebotAudioPackageVersion(sdk2.parameters.StrictString):
    name = 'package_version'
    description = 'Version'
    required = True
    regexp = r'\d{1}\.\d{1}\.\d{1}'
    default_value = '1.0.0'


class NinebotAudioPackageName(sdk2.parameters.String):
    name = 'package_name'
    description = 'Filename'
    required = True
    default_value = 'audio_package_v100_moscow_1.bin'


class NinebotAudioPackage(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1  # Require 1 CPU core
        ram = 1024  # Require 1024 Mib RAM
        client_tags = ctc.Tag.WINDOWS
        # dns = ctm.DnsType.DNS64  # Enable NAT64 to access to IPv4-only hosts
        # Download specific resources to task
        resources = [
            3333344593, 3336620363, 3336634535,
            3336629252, 3336630796, 3336632198,
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = ('Task to build audio package for Ninebot v2 '
                       'with Omni.exe tool in Windows.')
        max_restarts = 2
        kill_timeout = 30 * 60
        notifications = [
            sdk2.Notification(
                statuses=[
                    ctt.Status.FAILURE,
                    ctt.Status.EXCEPTION,
                    ctt.Status.TIMEOUT,
                ],
                recipients=['vdovkin@yandex-team.ru'],
                transport=ctn.Transport.EMAIL,
            ),
        ]
        owner = 'vdovkin'
        tags = ['scooters', 'windows']

        with sdk2.parameters.Group('Package info') as package_meta:
            package_version = NinebotAudioPackageVersion()
            package_name = NinebotAudioPackageName()
            bitrate = NinebotAudioBitrateSelector()

        with sdk2.parameters.Group(
            'Scooter system sounds (IOT trigger playback, can\'t be used in V0 command)',
        ) as iot_trigger_system_sounds:
            SFX0_power_on = sdk2.parameters.Resource('Power on', required=True)
            SFX1_power_off = sdk2.parameters.Resource('Power off', required=True)
            SFX2_unlock = sdk2.parameters.Resource('Unlock', required=True)
            SFX3_lock = sdk2.parameters.Resource('Lock', required=True)
            SFX7_illegal_moving = sdk2.parameters.Resource('Illegal moving', required=True)

        with sdk2.parameters.Group(
            'Custom sounds (server triggered playback, index for playing through V0 command)',
        ) as custom_sounds:
            SFX5_battery_low = sdk2.parameters.Resource('1: Battery low', required=True)
            SFX8_locating = sdk2.parameters.Resource('2: Locating', required=True)
            SFX9_alarm = sdk2.parameters.Resource('3: Alarm for service finding', default=3336629252, required=True)
            SFX6_empty = sdk2.parameters.Resource('6: Tips for driving into geofence (Unused)', default=3336624495,
                                                  required=True)
            SFX13_slowdown_20 = sdk2.parameters.Resource('91: Slowing down to 20', required=True)
            SFX14_slowdown_15 = sdk2.parameters.Resource('92: Slowing down to 15', required=True)
            SFX15_slowdown_10 = sdk2.parameters.Resource('93: Slowing down to 10', required=True)
            SFX16_slowdown_0 = sdk2.parameters.Resource('94: Disable acceleration', required=True)
            SFX17_opened_hood = sdk2.parameters.Resource('95: Battery hood is opened', required=True)
            SFX18_falling_ground = sdk2.parameters.Resource('96: Scooter is fallen', required=True)
            SFX19_lifted_up = sdk2.parameters.Resource('97: Scooter has been lifted', required=True)
            SFX20_broken = sdk2.parameters.Resource('98: Scooter is broken', required=True)
            SFX21_empty = sdk2.parameters.Resource('99: Empty (can be used)', default=3336634535,
                                                   required=True)

        with sdk2.parameters.Group(
            'Unused scooter system sounds (can\'t be used in V0 command)',
        ) as unused_system_sounds:
            SFX4_empty = sdk2.parameters.Resource('Timeout prompt for communication with vehicle (Unused)',
                                                  default=3336620363, required=True)
            SFX10_empty = sdk2.parameters.Resource('SIM card loss alarm (Unused)',
                                                   default=3336630796, required=True)
            SFX11_empty = sdk2.parameters.Resource('Tips for opening the battery compartment cover (Unused)',
                                                   default=3336632198, required=True)
            SFX12_empty = sdk2.parameters.Resource('Vehicle configuration succeeded (Unused)',
                                                   default=3336634535, required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Ready to use Ninebot v2 audio package') as out_group:
                audio_package_bin = sdk2.parameters.Resource(
                    'Binary',
                )
                audio_package_checksum = sdk2.parameters.Integer(
                    'Checksum',
                )

    @staticmethod
    def get_path(resource):
        return str(sdk2.ResourceData(resource).path)

    @staticmethod
    def get_checksum(path):
        checksum = 0
        with open(path, "rb") as f:
            byte = f.read(1)
            while byte != "":
                checksum += ord(byte)
                byte = f.read(1)
        return int(ctypes.c_uint32(checksum).value)

    def on_execute(self):
        audio_tool_data = sdk2.ResourceData(sdk2.Resource[3333344593])
        audio_tool_path = str(audio_tool_data.path / 'Omni.exe')

        cmd = [
            audio_tool_path,
            '-o', self.Parameters.package_name,
            '-v', self.Parameters.package_version,
            '-f', self.get_path(self.Parameters.SFX0_power_on),
            '-f', self.get_path(self.Parameters.SFX1_power_off),
            '-f', self.get_path(self.Parameters.SFX2_unlock),
            '-f', self.get_path(self.Parameters.SFX3_lock),
            '-f', self.get_path(self.Parameters.SFX4_empty),
            '-f', self.get_path(self.Parameters.SFX5_battery_low),
            '-f', self.get_path(self.Parameters.SFX6_empty),
            '-f', self.get_path(self.Parameters.SFX7_illegal_moving),
            '-f', self.get_path(self.Parameters.SFX8_locating),
            '-f', self.get_path(self.Parameters.SFX9_alarm),
            '-f', self.get_path(self.Parameters.SFX10_empty),
            '-f', self.get_path(self.Parameters.SFX11_empty),
            '-f', self.get_path(self.Parameters.SFX12_empty),
            '-f', self.get_path(self.Parameters.SFX13_slowdown_20),
            '-f', self.get_path(self.Parameters.SFX14_slowdown_15),
            '-f', self.get_path(self.Parameters.SFX15_slowdown_10),
            '-f', self.get_path(self.Parameters.SFX16_slowdown_0),
            '-f', self.get_path(self.Parameters.SFX17_opened_hood),
            '-f', self.get_path(self.Parameters.SFX18_falling_ground),
            '-f', self.get_path(self.Parameters.SFX19_lifted_up),
            '-f', self.get_path(self.Parameters.SFX20_broken),
            '-f', self.get_path(self.Parameters.SFX21_empty),
            '-b', self.Parameters.bitrate,
        ]
        logger.info('Running audio tool cmd:\n%s', '\n '.join(cmd))
        out = subprocess.check_output(cmd, stderr=subprocess.PIPE)
        logger.info(out)
        self.Parameters.description += ('\n' + out)

        self.Parameters.audio_package_bin = NinebotAudioPackageBinary(
            self, self.Parameters.package_name, self.Parameters.package_name)
        sdk2.ResourceData(self.Parameters.audio_package_bin).ready()
        self.Parameters.audio_package_checksum = self.get_checksum(self.Parameters.package_name)
