# -*- coding: utf-8 -*-

import logging
import requests
import os

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.scraper_over_yt.ScraperOverYtTestHelper.telegram_sender import TelegramSender
from sandbox.projects.scraper_over_yt.ScraperOverYtMakeTestBatch.scraper_over_yt_checker import ScraperOverYtChecker
from sandbox.projects.scraper_over_yt.ScraperOverYtMakeTestBatch.scraper_over_yt_downloader import ScraperOverYtDownloadQueries
from sandbox.projects.scraper_over_yt.ScraperOverYtMakeTestBatch.scraper_over_yt_creator import ScraperOverYtCreator
from sandbox.projects.scraper_over_yt.ScraperOverYtMakeTestBatch.pool_set_filter import PoolSetFilter


class ScraperOverYtMakeTestBatch(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex-yt'),
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        processed_path = sdk2.parameters.String("Processed table path", required=True)
        filter_string = sdk2.parameters.String("Filter string", required=True)
        path_to_create = sdk2.parameters.String("Path to create test batch", required=True)
        proxy = sdk2.parameters.String("Proxy", default="hahn", required=True)
        yt_token_owner = sdk2.parameters.String('Yt owner', default_value='')
        yt_token_name = sdk2.parameters.String('Yt token name', default_value='')
        soy_token_name = sdk2.parameters.String('Soy token name', default_value='')
        soy_api = sdk2.parameters.String('SOY API', default_value='')
        telegram_token_owner = sdk2.parameters.String('Telegram token owner', default_value='SEARCH-RELEASERS')
        telegram_token_bot = sdk2.parameters.String('Telegram token bot', default_value='rm_telegram_token')
        chat_id = sdk2.parameters.String('Chat ID', default_value='-347783386')

    def on_execute(self):
        import yt.wrapper as yt

        pool_configs = PoolSetFilter(self.Parameters.filter_string)
        soy_token = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.soy_token_name)

        self.Context.download_id = []
        self.Context.soy_api = self.Parameters.soy_api

        yt.config["write_parallel"]["unordered"] = True
        yt.config["read_parallel"]["enable"] = True
        yt.config["write_parallel"]["enable"] = True
        yt.config["proxy"]["url"] = self.Parameters.proxy
        yt.config["token"] = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)

        test_batch_directory = os.path.join(self.Parameters.path_to_create, 'tests')
        test_processed_path = os.path.join(self.Parameters.path_to_create, 'operations_queue', 'processed')

        telegram_sender = TelegramSender(sdk2.Vault.data(self.Parameters.telegram_token_owner, self.Parameters.telegram_token_bot), self.Parameters.chat_id)

        creator = ScraperOverYtCreator(pool_configs, telegram_sender, self.Parameters.processed_path, test_batch_directory, yt)

        downloader = ScraperOverYtDownloadQueries(self.Parameters.soy_api, telegram_sender, soy_token)

        prepared_to_create_tables = creator.create_test_batch()

        pools = pool_configs.get_pool_list()
        pool_str = ', '.join(pools)

        # Dynamic scheduler testing if exists default pool
        if 'default' in pools:
            selected = 0
            max_selected_tables = pool_configs.get_max_selected_tables('default')
            dyn_pool = 'experiments_rm'  # RMDEV-2617
            pool_str += ', ' + dyn_pool

            tables_from_default = creator.select_tables(['default'])
            prepared_to_create_tables[dyn_pool] = []
            for table in tables_from_default['default']:
                try:
                    yt_table = creator.create_table(table['working_directory'], 'default', table['spec'])
                except:
                    logging.info('Anything went wrong while creating table')
                else:
                    if yt_table.has_value_in_spec('batched_requests') and yt_table.get_value_from_spec('batched_requests'):
                        continue
                    yt_table.change_spec('pool', dyn_pool)
                    prepared_to_create_tables[dyn_pool].append(yt_table)
                    selected += 1
                    if selected == max_selected_tables:
                        break

        telegram_sender.send('Pools: {0} creating batch'.format(pool_str))
        errors_count = 0

        for pool in prepared_to_create_tables:
            telegram_sender.send('Batch for {0} created with {1} tables'.format(pool, len(prepared_to_create_tables[pool])))
            if len(prepared_to_create_tables[pool]) == 0:
                errors_count += 1
            for table in prepared_to_create_tables[pool]:
                table.create_input_table()
                downloader.send_and_add_by_row(table)
                self.Context.download_id.append(table.get_id())

        telegram_sender.send('Pools: {0} is downloading'.format(pool_str))
        downloader.start()
        telegram_sender.send('Pools: {0} downloaded and started check stage'.format(pool_str))

        for pool in prepared_to_create_tables:
            for table in prepared_to_create_tables[pool]:
                status = downloader.get_status(table.get_id())
                logging.info(table.get_id())
                logging.info(status)
                table.set_operation_id(status['operation_id'])
                table.set_working_directory(status['working_directory'])
                checker = ScraperOverYtChecker(table, yt, test_processed_path, status, telegram_sender)
                errors_count += checker.check() is False

        if errors_count > 0:
            telegram_sender.send('Pools: {0} has {1} errors!'.format(pool_str, errors_count))
            return common.errors.TaskFailure("Errors")
        else:
            telegram_sender.send('Pools: {0} checked successful!'.format(pool_str))

    def on_break(self, prev_status, status):
        for i in self.Context.download_id:
            logging.info('Abort id = ' + i)
            requests.get(self.Context.soy_api + '/abort?id=' + i)

    def on_finish(self, prev_status, status):
        for i in self.Context.download_id:
            logging.info('Abort id = ' + i)
            requests.get(self.Context.soy_api + '/abort?id=' + i)
