# -*- coding: utf-8 -*-

import logging
import json

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox import common

from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard

from sandbox.projects.scraper_over_yt.ScraperOverYtMakeTestBatch import ScraperOverYtMakeTestBatch
from sandbox.projects.scraper_over_yt.ScraperOverYtTestHelper.telegram_sender import TelegramSender


class ScraperOverYtTestManager(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        processed_path = sdk2.parameters.String("Processed table path", required=True)
        filter_string = sdk2.parameters.String("Filter string", required=True)
        path_to_create = sdk2.parameters.String("Path to create test batch", required=True)
        proxy = sdk2.parameters.String("Proxy", default="hahn", required=True)
        yt_token_owner = sdk2.parameters.String('Yt owner', default_value='')
        yt_token_name = sdk2.parameters.String('Yt token name', default_value='', required=True)
        soy_token_name = sdk2.parameters.String('Soy token name', default_value='')
        soy_api = sdk2.parameters.String('SOY API', default_value='', required=True)
        thread_count = sdk2.parameters.Integer('Thread count', default_value=5)
        telegram_token_owner = sdk2.parameters.String('Telegram token owner', default_value='SEARCH-RELEASERS')
        telegram_token_bot = sdk2.parameters.String('Telegram token bot', default_value='rm_telegram_token')
        chat_id = sdk2.parameters.String('Chat ID', default_value='-347783386')
        deployment_nanny_dashboard_name = sdk2.parameters.String('NannyDashboardName', default_value='scraper_over_yt')
        deployment_release_status = sdk2.parameters.String('NannyDeploymentReleaseStatus', default_value='testing')
        nanny_vault_name = sdk2.parameters.String('NannyVaultName', default_value='robot_testing_scraper_over_yt_samogon_token')
        nanny_vault_owner = sdk2.parameters.String('NannyVaultOwner', default_value='')
        build_task_id_by_receipt = sdk2.parameters.Dict('build_task_id_by_receipt')

    def parse_filter_string(self):
        json_decoded = json.loads(self.Parameters.filter_string)
        splitted = []
        for pool in json_decoded:
            splitted.append('\"{pool}\": {dump}'.format(pool=pool, dump=json.dumps(json_decoded[pool])))

        l = len(splitted)
        pool_per_thread = l / self.Parameters.thread_count

        splitted_to_subtasks = []

        for i in range(self.Parameters.thread_count):
            filter_string_per_pool = ','.join(splitted[i * pool_per_thread:i * pool_per_thread + pool_per_thread])

            if l % self.Parameters.thread_count > i:
                filter_string_per_pool += splitted[self.Parameters.thread_count * pool_per_thread + i]

            splitted_to_subtasks.append('{' + filter_string_per_pool + '}')

        if l < self.Parameters.thread_count:
            return splitted_to_subtasks[:l]

        return splitted_to_subtasks

    def create_subtask(self, filter_string):
        task = ScraperOverYtMakeTestBatch(
            self, description="Created from SDK2-task",
            processed_path=self.Parameters.processed_path,
            filter_string=filter_string,
            path_to_create=self.Parameters.path_to_create,
            proxy=self.Parameters.proxy,
            yt_token_owner=self.Parameters.yt_token_owner,
            yt_token_name=self.Parameters.yt_token_name,
            soy_token_name=self.Parameters.soy_token_name,
            soy_api=self.Parameters.soy_api
        )
        t = task.enqueue().id
        logging.info(t)
        return t

    def create_child_tasks(self):
        sub_filters = self.parse_filter_string()
        tasks = []

        for i in sub_filters:
            tasks.append(self.create_subtask(i))

        return tasks

    def make_deploy_task(self, build_task_id, receipt):
        deploy_task_params = {
            'deployment_nanny_dashboard_name': self.Parameters.deployment_nanny_dashboard_name,
            'deployment_nanny_dashboard_recipe': receipt,
            'deployment_task_id': build_task_id,
            'deployment_release_status': self.Parameters.deployment_release_status,
            'vault_name': self.Parameters.nanny_vault_name,
            'vault_owner': self.Parameters.nanny_vault_owner,
            'deployment_nanny_bool_wait': True,
            'services_from_recipe': True,
        }
        deploy_task = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type](
            self,
            description='Run testing deployment.',
            **deploy_task_params
        )
        deploy_task.save().enqueue()
        return deploy_task.id

    def on_execute(self):
        telegram_sender = TelegramSender(sdk2.Vault.data(self.Parameters.telegram_token_owner, self.Parameters.telegram_token_bot), self.Parameters.chat_id)

        assert self.Parameters.build_task_id_by_receipt

        with self.memoize_stage.deploy:
            tasks = [self.make_deploy_task(build_task_id, receipt)
                     for receipt, build_task_id in self.Parameters.build_task_id_by_receipt.items()]

            self.Context.deploy_task_ids = tasks
            self.Context.save()
            raise sdk2.WaitTask(
                tasks,
                [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                wait_all=True)

        for task_info in self.find(id=self.Context.deploy_task_ids).limit(len(self.Context.deploy_task_ids)):
            if task_info.status not in ctt.Status.Group.SUCCEED:
                err_msg = "Failed deploy task: {}".format(task_info)
                telegram_sender.send(err_msg)
                raise common.errors.TaskFailure(err_msg)

        with self.memoize_stage.create_children:
            telegram_sender.send('Deploy testing success. Start downloading')
            subtasks = self.create_child_tasks()
            self.Context.subtask_ids = subtasks

            raise sdk2.WaitTask(
                subtasks,
                [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                wait_all=True
            )

        subtasks = list(self.find(id=self.Context.subtask_ids).limit(int(self.Parameters.thread_count)))

        failed = [subtask for subtask in subtasks if subtask.status not in ctt.Status.Group.SUCCEED]
        if failed:
            err_msg = "Failed subtasks: {}".format(failed)
            telegram_sender.send(err_msg)
            raise common.errors.TaskFailure(err_msg)
