import os

from sandbox import common
from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.projects.common.nanny import nanny
from sandbox.projects.sdc.common import constants
from sandbox.projects.sdc.common_tasks.PrepareTask import PrepareTask
from sandbox.projects.sdc.resource_types import SdcBinariesArchive

from sandbox.projects.sdc.SdcBuildNoTests import SdcBuildNoTests
from sandbox.projects.sdc.SdcBuildWithTests import SdcBuildWithTests


class PrepareBinariesResource(nanny.ReleaseToNannyTask2, PrepareTask):
    TEAMCITY_POLL_INTERVAL = 5 * 60
    BINARIES_ARTIFACT_PATH = 'binaries/binaries.tar.gz'

    description = 'Prepare sdc binaries from given branch or commit'

    class Parameters(PrepareTask.Parameters):
        # Task input parameters
        branch_or_commit = sdk2.parameters.String('Branch or commit/revision for sdc binaries', required=True)
        ctest_disabled = sdk2.parameters.Bool('Run without ctests', required=False, default_value=False)
        build_sdc_tc_build_id = sdk2.parameters.Integer('Build SDC teamcity build id to reuse for binaries',
                                                        required=False)
        build_sdc_sandbox_task_id = sdk2.parameters.Integer('Build SDC sandbox build id to reuse for binaries',
                                                            required=False)
        prepare_sdc_binaries_archive = sdk2.parameters.Bool('Prepare SDC binaries archive', default_value=True)
        program = sdk2.parameters.String('Program to release')

        # Task Service parameters
        kill_timeout = 5 * 60  # Set 5 min kill timeout

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC

    class Context(sdk2.Context):
        is_already_on_teamcity = False
        teamcity_build_task_id = None

        is_already_on_sandbox = False
        buildsdc_sandbox_task_id = None

        contains_branch = None
        binaries_commit = None
        binaries_branch = None

    @property
    def is_sandbox_buildsdc_found(self):
        return self.Context.is_already_on_sandbox

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
        self.mark_released_resources(additional_parameters["release_status"], ttl=90)

    def _register_buildsdc(self, buildsdc_task):
        task_id = buildsdc_task.id
        buildsdc_status = buildsdc_task.status

        buildsdc_task_url = 'https://sandbox.yandex-team.ru/task/{}'.format(task_id)

        self.LOGGER_IMPORTANT.info('BuildSdc task: %s', buildsdc_task_url)
        self.LOGGER_IMPORTANT.info('BuildSdc status is: %s', buildsdc_status)

        self.Context.is_already_on_sandbox = True
        self.Context.buildsdc_sandbox_task_id = task_id

        terminate_statuses = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK
        if buildsdc_status not in terminate_statuses:
            self.LOGGER_IMPORTANT.info('Task in progress -> wait until done')
            raise sdk2.WaitTask(buildsdc_task, statuses=terminate_statuses)

        self.LOGGER_IMPORTANT.info('BuildSdc already finished -> continue')

    def _look_for_existing_buildsdc_on_sandbox(self):
        buildsdc_task = None

        sb_task_id = self.Parameters.build_sdc_sandbox_task_id
        if sb_task_id:
            self.LOGGER_IMPORTANT.info('look for existing buildsdc - sandbox task id: %r',
                                       sb_task_id)
            buildsdc_task = sdk2.Task.find(id=sb_task_id).first()
            if buildsdc_task is None:
                raise common.errors.TaskFailure("Can't find sandbox task for given id")
        else:
            task_class = self._get_sandbox_buildsdc_class()
            hints = [self.Context.binaries_commit]
            reuse_statuses = ctt.Status.Group.EXECUTE + ctt.Status.Group.QUEUE + ctt.Status.Group.SUCCEED

            self.LOGGER_IMPORTANT.info('Look for existing sandbox task (class: %s, hints:%s)',
                                       task_class, hints)

            buildsdc_task = task_class.find(status=reuse_statuses,
                                            hints=hints,
                                            all_hints=True,
                                            children=True).first()

        if buildsdc_task:
            self._register_buildsdc(buildsdc_task)

        return buildsdc_task

    def _spawn_new_buildsdc_task(self):
        assert self.Context.binaries_commit

        task_class = self._get_sandbox_buildsdc_class()
        spawn_kwargs = self._get_sandbox_spawn_params(task_class)

        self.LOGGER_IMPORTANT.info('Spawning new buildsdc(type: %s) args: %s',
                                   task_class, spawn_kwargs)

        task = task_class(self, **spawn_kwargs)
        task.enqueue()

        self._register_buildsdc(task)

    def _get_sandbox_buildsdc_class(self):
        if self.Parameters.ctest_disabled:
            return SdcBuildNoTests
        return SdcBuildWithTests

    def _get_sandbox_spawn_params(self, task_class):
        spawn_kwargs = {
            'vcs_type': self.Parameters.vcs_type,
            'branch': self.Context.binaries_branch,
            'commit': self.Context.binaries_commit,
            'upload_porto': True,
        }

        if not self.Context.binaries_branch and self.Parameters.vcs_type == constants.VCS_ARC:
            spawn_kwargs['branch'] = 'trunk'  # set default branch for arc

        if task_class == SdcBuildNoTests:
            # TODO(markperez): Remove after this steps become default for this configuration
            spawn_kwargs['vcs_steps_path'] = 'tc_build/build_sdc_no_tests'

        return spawn_kwargs

    def _ensure_buildsdc_on_sandbox_is_ok(self):
        sb_task_id = self.Context.buildsdc_sandbox_task_id
        buildsdc_task = sdk2.Task.find(id=sb_task_id).first()
        if buildsdc_task.status != ctt.Status.SUCCESS:
            self.LOGGER_IMPORTANT.info('BuildSdc has non success status => exit with failure')
            raise common.errors.TaskFailure('Sanbox task {} has non success state'.format(sb_task_id))
        return buildsdc_task

    def _look_for_existing_buildsdc_on_teamcity(self):
        if not self.Parameters.build_sdc_tc_build_id:
            return

        # Reuse specific teamcity build (only OLD PR case) TODO(markperez): remove later
        teamcity_build_url = 'https://teamcity.yandex-team.ru/viewLog.html?buildId={}'.format(
            self.Parameters.build_sdc_tc_build_id)
        self.LOGGER_IMPORTANT.info('Build %s was reused for binaries', teamcity_build_url)
        self.Context.is_already_on_teamcity = True
        self.Context.teamcity_build_task_id = self.Parameters.build_sdc_tc_build_id

    def _resolve_binaries_mds_url_from_teamcity(self):
        assert self.Context.teamcity_build_task_id
        build_props = self._get_build_resulting_props(self.Context.teamcity_build_task_id)
        props_dict = {}
        for p in build_props:
            name = p.attrib['name']
            value = p.attrib.get('value', '')
            props_dict[name] = value

        return self._resolve_binaries_mds_url_from_dict(props_dict)

    def _resolve_binaries_mds_url_from_dict(self, props_dict):
        binaries_mds_url = props_dict.get('binaries_mds_url') or props_dict.get('archive_url')
        already_uploaded_yt_path = props_dict.get('porto_layer_yt_path')

        return already_uploaded_yt_path, binaries_mds_url

    def _prepare_resource_from_sandbox(self):
        buildsdc_task = self._ensure_buildsdc_on_sandbox_is_ok()
        self.LOGGER_IMPORTANT.info('[prepare resource from sandbox] buildsdc is OK -> cook resource')

        # Extract required params & cook resource
        params = buildsdc_task.Parameters.runtime_parameters
        found_yt_path, binaries_mds_url = self._resolve_binaries_mds_url_from_dict(params)
        self._prepare_resource(found_yt_path, binaries_mds_url)

    def _prepare_resource_from_teamcity(self):
        found_yt_path, binaries_mds_url = self._resolve_binaries_mds_url_from_teamcity()
        self._prepare_resource(found_yt_path, binaries_mds_url)

    def _prepare_resource(self, existed_yt_path, binaries_mds_url):
        self.LOGGER_IMPORTANT.info('[Prepare resource] existed_yt_path: %r binaries_mds_url: %r',
                                   existed_yt_path, binaries_mds_url)

        need_upload = not existed_yt_path or self.Parameters.prepare_sdc_binaries_archive
        if not need_upload:
            # do not create resource if we already created porto layer
            self.LOGGER_IMPORTANT.info('Porto layer {} was already uploaded'.format(existed_yt_path))
            return
        # This code is still needed for isolate cloud deploy (@alkedr):
        # https://tsum.yandex-team.ru/pipe/projects/sdc-web-simulator/release/60e6a5da01aa363dd1d7cdc1
        if not binaries_mds_url:
            raise common.errors.TaskFailure('Binaries mds url is not resolved')

        try:
            import boto3
            import botocore
            from six.moves.urllib.parse import urlparse

            url_parsed = urlparse(binaries_mds_url)
            bucket = url_parsed.netloc.split('.')[0]
            if bucket == 's3-private':
                # This is not correct bucket name. Get it from first path element,
                # https://st.yandex-team.ru/MDSSUPPORT-96#5df36be833c14c0534f7967d
                bucket = url_parsed.path.split('/')[1]
                endpoint_url = '{}://{}'.format(url_parsed.scheme, url_parsed.netloc)
                key = url_parsed.path.lstrip('/{}'.format(bucket))
            else:
                bucket_part = url_parsed.netloc.replace('{}.'.format(bucket), '', 1)
                endpoint_url = '{}://{}'.format(url_parsed.scheme, bucket_part)
                key = url_parsed.path.lstrip('/')

            b3_config = botocore.client.Config(
                s3={'addressing_style': 'virtual'},
                retries={'max_attempts': 5}
            )
            b3_session = boto3.session.Session()

            b3_client = b3_session.client('s3',
                                          endpoint_url=endpoint_url,
                                          config=b3_config,
                                          aws_access_key_id=self.aws_key_id,
                                          aws_secret_access_key=self.aws_secret_key)

            local_binaries_path = os.path.basename(self.BINARIES_ARTIFACT_PATH)
            try:
                b3_client.download_file(bucket, key, local_binaries_path)
            except boto3.exceptions.Boto3Error as e:
                self.LOGGER_IMPORTANT.exception('Some problems, while downloading resource from s3 %s', repr(e))
                raise common.errors.TaskFailure('Cannot download resource')

            resource = SdcBinariesArchive(
                self, 'Sdc binaries archive', os.path.basename(local_binaries_path),
                ttl=1, branch=self.Context.binaries_branch, commit=self.Context.binaries_commit
            )
            if self.Parameters.program:
                resource.program = self.Parameters.program
            sdk2.ResourceData(resource).ready()
        except Exception as e:
            self.LOGGER_IMPORTANT.exception(
                'Cannot prepare resource: %s. MdsURL = %s', repr(e), binaries_mds_url)
            raise common.errors.TaskFailure('Cannot prepare resource')

    def get_nanny_task_component(self, additional_parameters):
        return self.Parameters.program

    def on_execute(self):
        # Parse input parameters
        with self.memoize_stage.resolve_commit_or_branch:
            self.Context.contains_branch = False
            self.Context.binaries_branch = None
            self.Context.binaries_commit = None
            (self.Context.contains_branch, self.Context.binaries_branch,
             self.Context.binaries_commit) = self._contains_branch()

        # Look for sandbox buildsdc task
        with self.memoize_stage.check_is_already_built_on_sandbox:
            # Wait running sandbox task if found
            self._look_for_existing_buildsdc_on_sandbox()

        # Sandbox build is not found
        if not self.is_sandbox_buildsdc_found:

            # Look for existing teamcity build (handle OLD PRs with teamcity_build_id parameter)
            with self.memoize_stage.check_if_already_built_on_teamcity:
                self._look_for_existing_buildsdc_on_teamcity()

            # If teamcity build is not found -> spawn sandbox
            if not self.Context.is_already_on_teamcity:
                with self.memoize_stage.spawn_sandbox_buildsdc:
                    self._spawn_new_buildsdc_task()

        if self.is_sandbox_buildsdc_found:
            # If sandbox found -> task should be already done (we raised Wait on prev iterations)
            self.LOGGER_IMPORTANT.info('Suitable sandbox task is found -> cook resource from it')
            # Cook resoure from sandbox
            self._prepare_resource_from_sandbox()
            # Finish our task -> resource was uploaded using sandbox task
            return
        elif self.Context.is_already_on_teamcity:
            # Wait until teamcity build is done & upload resource from it
            self._check_teamcity_task_status(self.Context.teamcity_build_task_id, self.TEAMCITY_POLL_INTERVAL)
            self._prepare_resource_from_teamcity()
            return

        # Unreachable condition, remove when teamcity will completely removed from this task
        raise common.errors.TaskFailure('BuildSDC is not found (or failed to spawn)')
