import logging

from sandbox import sdk2
from sandbox.projects.sdc.common_tasks.BaseSdcTask import BaseSdcTask
from sandbox.sdk2 import yav

import sandbox.common.types.client as ctc

log = logging.getLogger(__name__)


class SdcAwsCloneLanesMap(BaseSdcTask):
    class Requirements(BaseSdcTask.Requirements):
        ram = 32 * 1024  # 32GB
        disk_space = 30 * 1024  # 30 GB
        client_tags = ctc.Tag.SSD

    class Parameters(BaseSdcTask.Parameters):
        vcs_steps_path = sdk2.parameters.String(
            'Path to 0.sh script to Run',
            default='aws_deploy_copy/lanes_map',
        )
        lanes_map_base_versions = sdk2.parameters.List(
            'Lanes Map Index names without regions',
            default=['ci/maps-release'],
        )
        lanes_map_regions = sdk2.parameters.List(
            'Lanes Map regions to copy.'
            ' Use single "@" to mark full version without region suffix',
            default=[
                'annarbor',
                'columbus',
                'delivery_aa_ml',
                'delivery_ann-arbor',
                'delivery_columbus',
                'delivery_fish_point',
                'delivery_tucson',
                'detroit',
                'lasvegas',
            ],
        )
        foreign_aws_bucket = sdk2.parameters.String(
            'AWS Bucket name',
            default='sdc-deploy',
        )
        foreign_aws_endpoint_url = sdk2.parameters.String(
            'AWS Endpoint URL',
            default='https://s3.us-east-2.amazonaws.com',
        )
        foreign_aws_key_prefix = sdk2.parameters.String(
            'AWS Key Prefix',
            default='sdc-lanes-map/cas',
        )
        foreign_store_debug = sdk2.parameters.Integer(
            'Number of --debug flags to infra.maps.lanes_cas',
            default=0,
        )

    def on_prepare(self):
        super(SdcAwsCloneLanesMap, self).on_prepare()
        secrets = yav.Yav(
            sdg_aws_sdc_deploy_publisher=yav.Secret('sec-01fx317z32txnjjacp2yktpn7n'),
        )
        self.sdg_aws_sdc_deploy_publisher = secrets.sdg_aws_sdc_deploy_publisher

    def secret_list(self):
        r = super(SdcAwsCloneLanesMap, self).secret_list()
        # no need to hide access_key_id
        r.append(self.sdg_aws_sdc_deploy_publisher['secret_access_key'])
        return r

    def get_non_overridable_env_variables(self):
        r = super(SdcAwsCloneLanesMap, self).get_non_overridable_env_variables()
        r.update({
            'T__FOREIGN_S3_ACCESS_KEY_ID': self.sdg_aws_sdc_deploy_publisher['access_key_id'],
            'T__FOREIGN_S3_BUCKET': self.Parameters.foreign_aws_bucket,
            'T__FOREIGN_S3_ENDPOINT_URL': self.Parameters.foreign_aws_endpoint_url,
            'T__FOREIGN_S3_KEY_PREFIX': self.Parameters.foreign_aws_key_prefix,
            'T__FOREIGN_STORE_DEBUG': ' '.join(['--debug'] * self.Parameters.foreign_store_debug),
            'T__LANES_MAP_VERSIONS': ' '.join(self.combine_lanes_map_versions()),
            'T__SECRET__FOREIGN_S3_SECRET_ACCESS_KEY':
                self.sdg_aws_sdc_deploy_publisher['secret_access_key'],
        })
        return r

    def combine_lanes_map_versions(self):
        if not self.Parameters.lanes_map_base_versions:
            raise ValueError('base_versions list are empty')
        for i, base in enumerate(self.Parameters.lanes_map_base_versions):
            if '/' not in base:
                raise ValueError('base_versions[{}] invalid value {!r}'.format(i, base))

        if not self.Parameters.lanes_map_regions:
            raise ValueError('regions list is empty')

        sources = set(self.Parameters.lanes_map_base_versions)
        regions = set(self.Parameters.lanes_map_regions)

        r = []
        if '@' in regions:
            r.extend(sources)
        regions.discard('@')

        r.extend('/'.join([base, region]) for base in sources for region in regions)
        return sorted(r)
